/*
 *  ALTerator - ALT Linux configuration project
 *
 *  Copyright (c) 2004,2005 ALT Linux Ltd.
 *  Copyright (c) 2004,2005 Alexey Voinov
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 *  USA.
 */
#ifndef UTILS_GUILE_HH
#define UTILS_GUILE_HH

#include <libguile.h>
#include <string>

namespace scm
{
	extern const std::string commondir;

	class scm
	{
		SCM value_;

	public:
		scm(SCM value): value_(value) {}
		scm(const std::string& value)
			: value_(scm_mem2string(value.data(), value.size())) {}
		scm(long value): value_(SCM_MAKINUM(value)) {}

		std::string as_str(void) const
		{
			return std::string(SCM_STRING_CHARS(value_),
						SCM_STRING_LENGTH(value_));
		}

		long as_long(void) const
		{
			return SCM_INUM(value_);
		}

		SCM as_scm(void) const { return value_; }
		operator bool (void) const { return SCM_NFALSEP(value_); }
	};

	inline void init(void)
	{
		::scm_init_guile();
	}

	inline scm load(const char *filename)
	{
		return scm(::scm_c_primitive_load(filename));
	}

	inline scm load(const scm& filename)
	{
		return scm(::scm_primitive_load(filename.as_scm()));
	}

	inline scm null_p(const scm& val)
	{
		return scm(::scm_null_p(val.as_scm()));
	}

	inline bool pair_p(const scm& val)
	{
		return SCM_CONSP(val.as_scm());
	}

	inline scm cons(const scm& x, const scm& y)
	{
		return scm(::scm_cons(x.as_scm(), y.as_scm()));
	}

	inline scm car(const scm& val)
	{
		return scm(SCM_CAR(val.as_scm()));
	}

	inline scm cdr(const scm& val)
	{
		return scm(SCM_CDR(val.as_scm()));
	}

	inline scm cadr(const scm& val)
	{
		return scm(SCM_CADR(val.as_scm()));
	}

	inline scm cddr(const scm& val)
	{
		return scm(SCM_CDDR(val.as_scm()));
	}

	inline scm lookup(const scm& name)
	{
		return scm(scm_lookup(name.as_scm()));
	}

	inline scm lookup(const char *name)
	{
		return scm(scm_c_lookup(name));
	}

	inline scm variable_ref(const scm& val)
	{
		return scm(::scm_variable_ref(val.as_scm()));
	}

	inline scm variable_ref(const char *name)
	{
		return variable_ref(lookup(name));
	}

	inline scm call(const scm& proc)
	{
		return scm(::scm_call_0(proc.as_scm()));
	}

	inline scm call(const scm& proc, const scm& par1)
	{
		return scm(::scm_call_1(proc.as_scm(), par1.as_scm()));
	}

	inline scm call(const scm& proc, const scm& par1, const scm& par2)
	{
		return scm(::scm_call_2(proc.as_scm(),
					par1.as_scm(), par2.as_scm()));
	}

	inline scm to_string(const scm& val,
			const scm& prn = SCM_UNDEFINED)
	{
		return scm(::scm_object_to_string(val.as_scm(), prn.as_scm()));
	}

	inline scm symbol(const char *str)
	{
		return scm(::scm_str2symbol(str));
	}

	inline scm eq_p(const scm& x, const scm& y)
	{
		return scm(::scm_eq_p(x.as_scm(), y.as_scm()));
	}
}

#endif // UTILS_GUILE_HH
