#!/usr/bin/guile -s
!#
;;;
;;; ALTerator - ALT Linux configuration project
;;;
;;; Copyright (c) 2005 ALT Linux Ltd.
;;; Copyright (c) 2005 Alexey Voinov
;;;
;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2 of the License, or
;;; (at your option) any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
;;; USA.
;;;
(use-modules (ice-9 syncase))
(load "")

(define (with-socket family type protocol proc)
  (let ((s (socket family type protocol)))
    (proc s)
    (close-port s)))

(define-syntax aif
  (syntax-rules ()
    ((_ var expr then else) (let ((var expr)) (if var then else)))
    ((_ var test expr then else) (let ((var expr)) (if (test var) then else)))))

(define *dl-current* "")
(define *dl-total* 0)
(define *dl-url* 0)
(define *dl-mutex* (make-mutex))
(define *dl-tasks* '())
(define *exit-mutex* (make-mutex))
(define *exit* #f)

(define (wait-for pred . args)
  (if (not (apply pred args))
      (begin (yield)
             (sleep 1)
             (apply wait-for pred args))))

(define (threaded-accept s)
  (wait-for char-ready? s)
  (accept s))

(define (server sock-name)
  (with-socket
   PF_UNIX SOCK_STREAM 0
   (lambda (s)
     (bind s AF_UNIX sock-name)
     (listen s 10)
     (let loop ((c (car (threaded-accept s))))
       (begin-thread (let loop ((command (read c)))
                       (cond ((eof-object? command) #f)
                             
                             ((eq? command '+)
                              (aif task eof-object? #f (dl-append task))
                              (loop (read c)))

                             ((eq? command '?)
                              (display-status c)
                              (loop (read c)))
                             
                             ((eq? command 'x)
                              (close-port c)
                              (exit-server))
                             
                             (else (loop (read c))))))
       (loop (threaded-accept s))))))

(define (exit-server)
  (with-mutex *exit-mutex* (set! *exit* #t)))

(define (display-status s)
  (write (list *dl-url* *dl-current* *dl-total*) s))

; fixme: generate filename from url?
(define (dl-append task)
  (with-mutex *dl-mutex* (set! *dl-tasks* (append *dl-tasks* (list task)))))

(define (get-dl-task)
  (with-mutex *dl-mutex*
              (if (empty? *dl-tasks*) #f
                  (let ((task (car *dl-tasks*)))
                    (set! *dl-tasks* (cdr *dl-tasks*))
                    task))))

(define (main)
  (begin-thread (server "/tmp/sock-tst"))
  (call-with-current-continuation
   (lambda (return)
     (let loop ()
       (with-mutex *exit-mutex* (if *exit* (return 0)))
       (if (not (install-all (download-all "/tmp/packages/")))
           (sleep 3))
       (loop)))))

(define *curl-h* (curl-easy-init))

; untested
; fixme: protect from errors
(define (download-package url filename)
  (with-output-to-file filename
    (lambda ()
      (curl-easy-set-url *curl-h* url)
      (curl-easy-set-progress-function *curl-h*
                                       (lambda (dlt dln ult uln)
                                         (set! *dl-total* dlt)
                                         (set! *dl-current* dln)
                                         (set! *dl-url* url)
                                         (yield)))
      (curl-easy-set-write-function *curl-h*
                                    (lambda (str)
                                      (display str)
                                      (yield)
                                      #t))
      (curl-easy-perform *curl-h*)))
  filename)

; untested
(define (download-all)
  (let loop ((files '()))
    (if (>= (length files) 7) files
        (let ((task (get-dl-task task)))
          (if (not task) files
              (append files
                      (list (download-package (car task) (cdr task)))))))))

; untested
(define (install-all files)
  (if (empty? files) #f
      (let* ((db (rpm-db-open))
             (ts (rpm-create-transaction db)))
        (for-each (lambda (package) (rpm-trans-install ts package)) files)
        (rpm-dep-order ts)
        (rpm-run-transaction ts
                             (lambda (what name amount total)
                               (format #t "~A ~A ~A ~A" what name amount total))))))
  

               
