/*
 *  ALTerator - ALT Linux configuration project
 *
 *  Copyright (c) 2004,2005 ALT Linux Ltd.
 *  Copyright (c) 2004,2005 Alexey Voinov
 *  Copyright (c) 2004,2005 Stanislav Ievlev
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 *  USA.
 */
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <module.hh>
#include <utils/os.hh>

#include <iostream>

namespace
{
	static std::string name_from_command(const std::string& command)
	{

		std::string::size_type pos = command.find_first_of(" \t");
		std::string name;

		if(pos == std::string::npos) name = command;
		else name = command.substr(0, pos);

		pos = name.rfind("/");
		if(pos == std::string::npos) return name;
		else return name.substr(pos + 1);
	}
}

channel::channel(int in, int out,int err):
	is(out),
	os(in),
	es(err)
{}

module::module(const std::string& command,bool need_stderr)
	: name_(name_from_command(command)), command_(command),
	  pid_(0), in_(-1), out_(-1), err_(-1), io_(0),
	  need_stderr_(need_stderr)
{
}

module::module(const std::string& name, const std::string& command,bool need_stderr)
	: name_(name), command_(command),
	  pid_(0), in_(-1), out_(-1), err_(-1), io_(0),
	  need_stderr_(need_stderr)
{
}

module::~module(void)
{
	shutdown_io();
}

void module::startup_io(void)
{
	if(!io_.get())
	{
		//if(!is_running()) return;
		if((in_ < 0 || out_ < 0) ||
		   (need_stderr_ && err_<0)) throw err::io("no active fd");
		io_ = new channel(in_, out_, err_);
	}
}

void module::shutdown_io(void)
{
	io_ = 0;
	if(in_ >= 0) { ::close(in_); in_ = -1; }
	if(out_ >= 0) { ::close(out_); out_ = -1; }
	if(err_ >= 0) { ::close(err_); err_ = -1; }
}

bool module::is_running(void) const
{
	if(pid_)
		try
		{
			return alt::wait(pid_, alt::nohang).is_running();
		} catch(...) {}
	return false;
}

void module::start(const std::string& args)
{
	if(is_running()) return;
	shutdown_io();

	int stdin_in = -1, stdout_out = -1, stderr_out = -1;
	try
	{
		alt::pipe(stdin_in, in_);
		alt::pipe(out_, stdout_out);
		if (need_stderr_)
			alt::pipe(err_, stderr_out);
		else
		{
		  if ((stderr_out = open("/dev/null",O_WRONLY)) < 0)
		  	throw err::io("unable to open /dev/null"); 
		}
		if(!(pid_ = alt::fork()))
		{
			try
			{
				alt::setpgid();//note: this is a protection from SIGTERM received by parent.

				if (in_ >= 0) ::close(in_);
				if (out_ >=0) ::close(out_);
				if (err_ >=0) ::close(err_);

				::close(STDIN_FILENO); alt::dup2(stdin_in, STDIN_FILENO);
				::close(STDOUT_FILENO); alt::dup2(stdout_out, STDOUT_FILENO);
				::close(STDERR_FILENO); alt::dup2(stderr_out, STDERR_FILENO);

				if(args.empty()) alt::exec(command_);
				else alt::exec(command_ + " " + args);
			}
			catch(...)
			{
			}
			exit(EXIT_FAILURE);
		}
		::close(stdin_in); ::close(stdout_out); ::close(stderr_out);
		startup_io();
	}
	catch(...)
	{
		if(stdin_in >= 0) ::close(stdin_in);
		if(stdout_out >= 0) ::close(stdout_out);
		if(stderr_out >= 0) ::close(stderr_out);
		if(in_ >= 0) ::close(in_);
		if(out_ >= 0) ::close(out_);
		if(err_ >= 0) ::close(err_);
		throw;
	}
}

void module::stop(void)
{
	if (pid_)
	{
		shutdown_io();
		alt::kill(pid_, SIGINT);
		alt::wait(pid_);
		pid_ = 0;
	}
}

std::istream& module::is(void)
{
	if(io_.get()) return io_->is;
	throw err::io("io subsys is not active");
}

std::ostream& module::os(void)
{
	if(io_.get()) return io_->os;
	throw err::io("io subsys is not active");
}

std::istream& module::es(void)
{
	if(need_stderr_ && io_.get()) return io_->es;
	throw err::io("io subsys is not active");
}

void module::close_os(void)
{
	::close(in_); in_ = -1;
}
