/*
 *  ALTerator - ALT Linux configuration project
 *
 *  Copyright (c) 2005 ALT Linux Ltd.
 *  Copyright (c) 2005 Alexey Voinov
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 *  USA.
 */
#include <stddef.h>
#include <curl/curl.h>
#include <libguile.h>

#include <stdio.h>

static scm_t_bits handler_tag;

static void scm_wrap_curl_error(const char *subr, CURLcode code)
{
	if(code == CURLE_OK) return;
	scm_error(scm_misc_error_key, subr, curl_easy_strerror(code), SCM_BOOL_F, SCM_BOOL_F);
}

static size_t scm_curl_easy_std_write_f(void *ptr, size_t size, size_t nmemb, void *userp)
{
	SCM func = userp;
	SCM data = scm_mem2string(ptr, size * nmemb);
	SCM result = scm_call_1(func, data);
	return (SCM_NFALSEP(result)) ? size * nmemb: 0;
}

SCM scm_curl_easy_set_write_function(SCM h, SCM func)
{
	CURL *handle;

	SCM_ASSERT(SCM_SMOB_PREDICATE(handler_tag, h), h, SCM_ARG1, "curl-easy-set-write-function");

	handle = (CURL *)SCM_SMOB_DATA(h);
	scm_wrap_curl_error("curl-easy-set-write-function",
			    curl_easy_setopt(handle, CURLOPT_WRITEFUNCTION, scm_curl_easy_std_write_f));
	scm_wrap_curl_error("curl-easy-set-write-function",
			    curl_easy_setopt(handle, CURLOPT_WRITEDATA, func));

	return SCM_UNSPECIFIED;
}

static int scm_curl_easy_std_progress_f(void *userp, double dltotal, double dlnow, double ultotal, double ulnow)
{
	SCM func = userp;
	SCM result = scm_call_4(func,
				scm_make_real(dltotal), scm_make_real(dlnow),
				scm_make_real(ultotal), scm_make_real(ulnow));
	return 0;
}

SCM scm_curl_easy_set_progress_function(SCM h, SCM func)
{
	CURL *handle;

	SCM_ASSERT(SCM_SMOB_PREDICATE(handler_tag, h), h, SCM_ARG1, "curl-easy-set-progress-function");

	handle = (CURL *)SCM_SMOB_DATA(h);
	scm_wrap_curl_error("curl-easy-set-progress-function",
			    curl_easy_setopt(handle, CURLOPT_PROGRESSFUNCTION, scm_curl_easy_std_progress_f));
	scm_wrap_curl_error("curl-easy-set-progress-function",
			    curl_easy_setopt(handle, CURLOPT_PROGRESSDATA, func));
	scm_wrap_curl_error("curl-easy-set-progress-function",
			    curl_easy_setopt(handle, CURLOPT_NOPROGRESS, 0));

	return SCM_UNSPECIFIED;
}

SCM scm_curl_easy_set_url(SCM h, SCM url)
{
	CURL *handle;

	SCM_ASSERT(SCM_SMOB_PREDICATE(handler_tag, h), h, SCM_ARG1, "curl-easy-set-url");
	SCM_ASSERT(SCM_STRINGP(url), url, SCM_ARG2, "curl-easy-set-url");
	handle = (CURL *)SCM_SMOB_DATA(h);
	scm_wrap_curl_error("curl-easy-set-url",
			    curl_easy_setopt(handle, CURLOPT_URL, SCM_STRING_CHARS(url)));

	 return SCM_UNSPECIFIED;
}

SCM scm_curl_easy_perform(SCM h)
{
	CURL *handle;

	SCM_ASSERT(SCM_SMOB_PREDICATE(handler_tag, h), h, SCM_ARG1, "curl-easy-perform");

	handle = (CURL *)SCM_SMOB_DATA(h);
	scm_wrap_curl_error("curl-easy-perform", curl_easy_perform(handle));

	return SCM_UNSPECIFIED;
}

SCM scm_curl_easy_init(void)
{
	CURL *handle = curl_easy_init();
	if(!handle) scm_error(scm_misc_error_key, "curl-easy-init", "cannot initialize libcurl", SCM_BOOL_F, SCM_BOOL_F);
	SCM_RETURN_NEWSMOB(handler_tag, handle);
}

void scm_init_curl(void)
{
	handler_tag = scm_make_smob_type("easy-curl-handler", 0);

	scm_c_define_gsubr("curl-easy-set-write-function", 2, 0, 0,
			   (SCM (*)())scm_curl_easy_set_write_function);
	scm_c_define_gsubr("curl-easy-set-progress-function", 2, 0, 0,
			   (SCM (*)())scm_curl_easy_set_progress_function);
	scm_c_define_gsubr("curl-easy-set-url", 2, 0, 0,
			   (SCM (*)())scm_curl_easy_set_url);
	scm_c_define_gsubr("curl-easy-perform", 1, 0, 0,
			   (SCM (*)())scm_curl_easy_perform);
	scm_c_define_gsubr("curl-easy-init", 0, 0, 0,
			   (SCM (*)())scm_curl_easy_init);
}
