/*
 *  ALTerator - ALT Linux configuration project
 *
 *  Copyright (c) 2004,2005 ALT Linux Ltd.
 *  Copyright (c) 2004,2005 Alexey Voinov
 *  Copyright (c) 2004,2005 Stanislav Ievlev
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 *  USA.
 */
#include <qapplication.h>
#include <qvbox.h>
#include <dialog-qt.hh>
#include <utils/strnum.hh>

#include <qmainwindow.h>

#include <iostream>

namespace
{
	long percent_to_width(long width)
	{
		return ( QApplication::desktop()->width() * width ) / 100;
	}
	
	long percent_to_height(long height)
	{
		return ( QApplication::desktop()->height() * height ) / 100;
	}

	long width_to_percent(long width)
	{
		return ( QApplication::desktop()->width() * 100 ) / width;
	}
	
	long height_to_percent(long height)
	{
		return ( QApplication::desktop()->height() * 100 ) / height;
	}
}

struct attribute_text
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		widget->setText(QString::fromUtf8(value.c_str()));
	}

	template <typename W>
	std::string get(W *widget)
	{
		return widget->text().utf8().data();
	}
};

struct attribute_mask
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		widget->setInputMask(value);
	}

	template <typename W>
	std::string get(W *widget)
	{
		return widget->inputMask();
	}
};

struct attribute_margin
{
	template <typename L>
	void set(L *layout, const std::string& value)
	{
		QBoxLayout *l = dynamic_cast<QBoxLayout *>(layout);
		if(l) l->setMargin(percent_to_height(str2num(value)));
	}

	template <typename L>
	std::string get(L *layout)
	{
		QBoxLayout *l = dynamic_cast<QBoxLayout *>(layout);
		if(l) return num2str(height_to_percent(l->margin()));
		return "";
	}
};

struct attribute_spacing
{
	template <typename L>
	void set(L *layout, const std::string& value)
	{
		QBoxLayout *l = dynamic_cast<QBoxLayout *>(layout);
		if(l) l->setSpacing(percent_to_height(str2num(value)));
	}

	template <typename L>
	std::string get(L *layout)
	{
		QBoxLayout *l = dynamic_cast<QBoxLayout *>(layout);
		if(l) return num2str(height_to_percent(l->spacing()));
		return "";
	}
};

struct attribute_caption
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		widget->setCaption(value);
	}

	template <typename W>
	std::string get(W *widget)
	{
		return widget->caption();
	}
};

struct attribute_align
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		if(value == "left") _set(widget, Qt::AlignLeft);
		else if(value == "right") _set(widget, Qt::AlignRight);
		else if(value == "center") _set(widget, Qt::AlignHCenter);
		else if(value == "justify") _set(widget, Qt::AlignJustify);
		else if(value == "auto") _set(widget, Qt::AlignAuto);
	}

	template <typename W>
	std::string get(W *widget)
	{
		int f = widget->alignment();
		if(f & Qt::AlignLeft) return "left";
		else if(f & Qt::AlignRight) return "right";
		else if(f & Qt::AlignHCenter) return "center";
		else if(f & Qt::AlignJustify) return "justify";
		else return "auto";
	}

private:
	template <typename W>
	void _set(W *widget, Qt::AlignmentFlags f)
	{
		widget->setAlignment((widget->alignment()
				     & (~Qt::AlignHorizontal_Mask)) | f);
	}
};

struct attribute_width
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		widget->setFixedWidth(percent_to_width(str2num(value)));
	}

	template <typename W>
	std::string get(W *widget)
	{
		return num2str(width_to_percent(widget->width()));
	}
};

struct attribute_height
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		widget->setFixedHeight(percent_to_height(str2num(value)));
	}

	template <typename W>
	std::string get(W *widget)
	{
		return num2str(height_to_percent(widget->height()));
	}
};

struct attribute_font
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		QFont font;
		font.setRawName(value);
		widget->setFont(font);
	}

	template <typename W>
	std::string get(W *widget)
	{
		QFont font = widget->font();
		return font.rawName();
	}
};

struct attribute_default
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		if(value == "yes") widget->setDefault(true);
		else if(value == "no") widget->setDefault(false);
	}

	template <typename W>
	std::string get(W *widget)
	{
		return widget->isDefault() ? "yes" : "no";
	}
};

struct attribute_readonly
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		if(value == "yes") widget->setReadOnly(true);
		else if(value == "no") widget->setReadOnly(false);
	}

	template <typename W>
	std::string get(W *widget)
	{
		return widget->isReadOnly() ? "yes" : "no";
	}
};

struct attribute_editable
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		if(value == "yes") widget->setEditable(true);
		else if(value == "no") widget->setEditable(false);
	}

	template <typename W>
	std::string get(W *widget)
	{
		return widget->editable() ? "yes" : "no";
	}
};

struct attribute_enabled
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		if(value == "yes") widget->setEnabled(true);
		else if(value == "no") widget->setEnabled(false);
	}

	template <typename W>
	std::string get(W *widget)
	{
		return widget->isEnabled() ? "yes" : "no";
	}
};

struct attribute_hidden
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		if(value == "yes") widget->hide();
		else if(value == "no") widget->show();
	}

	template <typename W>
	std::string get(W *widget)
	{
		return widget->isHidden() ? "yes" : "no";
	}
};


struct attribute_flat
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		if(value == "yes") widget->setFlat(true);
		else if(value == "no") widget->setFlat(false);
	}

	template <typename W>
	std::string get(W *widget)
	{
		return widget->isFlat() ? "yes" : "no";
	}
};

struct attribute_echo
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		if(value == "yes") widget->setEchoMode(QLineEdit::Normal);
		else if(value == "no") widget->setEchoMode(QLineEdit::NoEcho);
		else if(value == "stars") widget->setEchoMode(QLineEdit::Password);
	}

	template <typename W>
	std::string get(W *widget)
	{
		int f = widget->echoMode();
		if(f == QLineEdit::Normal) return "yes";
		else if(f == QLineEdit::NoEcho) return "no";
		else if(f == QLineEdit::Password) return "stars";
		return "";
	}
};


struct attribute_sizepolicy
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		QSizePolicy::SizeType type = QSizePolicy::Preferred;

		if (value == "fixed") type = QSizePolicy::Fixed; 
		else if (value == "expanding") type = QSizePolicy::Expanding;

		widget->setSizePolicy(type,type);
	}

	template <typename W>
	std::string get(W *widget)
	{
		return "";
	}
};


struct attribute_pixmap
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		widget->setPixmap(QPixmap(value));
	}

	template <typename W>
	std::string get(W *widget)
	{
		const QPixmap *pixmap = widget->pixmap();
		return pixmap ? "yes" : "no";
	}
};

struct attribute_backroundpixmap
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		widget->setPaletteBackgroundPixmap(QPixmap(value));
	}

	template <typename W>
	std::string get(W *widget)
	{
		const QPixmap *pixmap = widget->paletteBackgroundPixmap();
		return pixmap ? "yes" : "no";
	}
};

struct attribute_count
{
	template <typename W>
	std::string get(W *widget)
	{
		return num2str(widget->count());
	}
};

struct attribute_curtext
{
	template <typename W>
	std::string get(W *widget)
	{
		return widget->currentText().utf8().data();
	}
};

struct attribute_current
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		widget->setCurrentItem(str2num(value));
	}

	template <typename W>
	std::string get(W *widget)
	{
		return num2str(widget->currentItem());
	}
};

struct attribute_checked
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		if(value == "yes") widget->setChecked(true);
		else if(value == "no") widget->setChecked(false);
	}

	template <typename W>
	std::string get(W *widget)
	{
		return widget->isChecked() ? "yes" : "no";
	}
};

//TODO: QVBox is a buggy, we must replace it with QWidget + QVBoxLayout
qt_dialog::qt_dialog(QMainWindow *wnd)
	: wnd_(wnd),
	  top_(new QVBox(wnd))
{
	wnd->layout()->add(top_);
	top_->layout()->setAutoAdd(false);
	wnd->setCentralWidget(top_);
}

struct destroy
{
	void operator()(widget *ptr) const { delete ptr; }
};

qt_dialog::~qt_dialog(void)
{
	std::for_each(ptrs_.begin(), ptrs_.end(), destroy());
	delete top_;
}

widget *qt_dialog::remember_pointer(widget *ptr)
{
	ptrs_.push_back(ptr);
	return ptr;
}

widget *qt_dialog::make_button(void)
{
	return remember_pointer(new qt_button(top_));
}

widget *qt_dialog::make_label(void)
{
	return remember_pointer(new qt_label(top_));
}

widget *qt_dialog::make_lineedit(void)
{
	return remember_pointer(new qt_lineedit(top_));
}

widget *qt_dialog::make_textbox(void)
{
	return remember_pointer(new qt_textbox(top_));
}

widget *qt_dialog::make_checkbox(void)
{
	return remember_pointer(new qt_checkbox(top_));
}

widget *qt_dialog::make_progressbar(void)
{
	return remember_pointer(new qt_progressbar(top_));
}

widget *qt_dialog::make_listbox(void)
{
	return remember_pointer(new qt_listbox(top_));
}

widget *qt_dialog::make_combobox(void)
{
	return remember_pointer(new qt_combobox(top_));
}

widget *qt_dialog::make_vbox(void)
{
	return remember_pointer(new qt_vbox(top_));
}

widget *qt_dialog::make_hbox(void)
{
	return remember_pointer(new qt_hbox(top_));
}

void qt_dialog::name_it(const std::string& name, widget* w)
{
	names_[name] = w;
}

widget *qt_dialog::who_is_it(const std::string& name) const
{
	std::map<std::string, widget*>::const_iterator i = names_.find(name);
	if(i == names_.end()) return 0;
	return i->second;
}

long qt_dialog::run(void)
{
	top_->show();
	long i = qApp->exec();
	return i;
}

widget& qt_dialog::set_attr(const std::string& name, const std::string& value, const std::string&)
{
	if(name == "margin") attribute_margin().set(top_->layout(), value);
	else if(name == "spacing") attribute_spacing().set(top_->layout(), value);
	else if(name == "width") attribute_width().set(wnd_, value);
	else if(name == "height") attribute_height().set(wnd_, value);
	else if(name == "caption") attribute_caption().set(wnd_, value);
	else if(name == "backgroundpixmap") attribute_backroundpixmap().set(wnd_, value);
	return *this;
}

std::string qt_dialog::get_attr(const std::string& name)
{
	if(name == "margin") return attribute_margin().get(top_->layout());
	else if(name == "spacing") return attribute_spacing().get(top_->layout());
	else if(name == "width") return attribute_width().get(wnd_);
	else if(name == "height") return attribute_height().get(wnd_);
	else if(name == "caption") return attribute_caption().get(wnd_);
	else if(name == "backgroundpixmap") return attribute_backroundpixmap().get(wnd_);
	return "";
}

widget& qt_dialog::insert(widget *w)
{
	qt_widget *q = dynamic_cast<qt_widget *>(w);
	if(q) top_->layout()->add(q->get_widget());
	return *this;
}

//
// ------------
//

widget& qt_widget::insert(widget *)
{
	return *this;
}

void qt_widget::on_click(void)
{
	qApp->exit(get_event("on-click"));
}

void qt_widget::on_change(const QString&)
{
	qApp->exit(get_event("on-change"));
}

void qt_widget::on_change()
{
	qApp->exit(get_event("on-change"));
}

void qt_widget::on_return(void)
{
	qApp->exit(get_event("on-return"));
}

void qt_widget::on_select(void)
{
	qApp->exit(get_event("on-select"));
}

widget& qt_widget::set_attr(const std::string& name, const std::string& value, const std::string&)
{
	if(name == "width") attribute_width().set(get_widget(), value);
	else if(name == "hidden") attribute_hidden().set(get_widget(), value);
	else if(name == "height") attribute_height().set(get_widget(), value);
	else if(name == "enabled") attribute_enabled().set(get_widget(), value);
	return *this;
}

std::string qt_widget::get_attr(const std::string& name)
{
	if(name == "width") return attribute_width().get(get_widget());
	else if(name == "hidden") return attribute_hidden().get(get_widget());
	else if(name == "height") return attribute_height().get(get_widget());
	else if(name == "enabled") return attribute_enabled().get(get_widget());
	return "";
}

//
// ------------
//

qt_button::qt_button(QWidget *parent): wnd(new QPushButton(parent))
{
	wnd->setBackgroundOrigin(QWidget::WindowOrigin);
	connect(wnd, SIGNAL(clicked()), SLOT(on_click()));
}

widget& qt_button::set_attr(const std::string& name, const std::string& value, const std::string&)
{
	if(name == "text") attribute_text().set(wnd, value);
	else if(name == "default") attribute_default().set(wnd, value);
	else if(name == "flat") attribute_default().set(wnd, value);
	else if(name == "font") attribute_font().set(wnd, value);
	else if(name == "pixmap") attribute_pixmap().set(wnd, value);
	else qt_widget::set_attr(name,value);
	return *this;
}

std::string qt_button::get_attr(const std::string& name)
{
	if(name == "text") return attribute_text().get(wnd);
	else if(name == "default") return attribute_default().get(wnd);
	else if(name == "flat") return attribute_flat().get(wnd);
	else if(name == "font") return attribute_font().get(wnd);
	else if(name == "pixmap") return attribute_pixmap().get(wnd);
	else return qt_widget::get_attr(name);
}

QWidget *qt_button::get_widget(void) const
{
	return wnd;
}

//
// ------------
//

qt_label::qt_label(QWidget *parent): wnd(new QLabel(parent))
{
	wnd->setBackgroundOrigin(QWidget::WindowOrigin);
}

widget& qt_label::set_attr(const std::string& name, const std::string& value, const std::string&)
{
	if(name == "text") { attribute_text().set(wnd, value); text_ = value; }
	else if(name == "font") attribute_font().set(wnd, value);
	else if(name == "pixmap") attribute_pixmap().set(wnd, value);
	else if(name == "align") attribute_align().set(wnd, value);
	else qt_widget::set_attr(name,value);
	return *this;
}

std::string qt_label::get_attr(const std::string& name)
{
	if(name == "text") return text_;
	if(name == "font") return attribute_font().get(wnd);
	else if(name == "align") return attribute_align().get(wnd);
	else if(name == "pixmap") return attribute_pixmap().get(wnd);
	else return qt_widget::get_attr(name);
}

QWidget *qt_label::get_widget(void) const
{
	return wnd;
}

//
// ------------
//

qt_checkbox::qt_checkbox(QWidget *parent): wnd(new QCheckBox(parent))
{
	connect(wnd, SIGNAL(clicked()), SLOT(on_click()));
	wnd->setBackgroundOrigin(QWidget::WindowOrigin);
}

widget& qt_checkbox::set_attr(const std::string& name, const std::string& value, const std::string&)
{
	if(name == "text") { attribute_text().set(wnd, value); text_ = value; }
	else if(name == "font") attribute_font().set(wnd, value);
	else if(name == "pixmap") attribute_pixmap().set(wnd, value);
	else if(name == "checked") attribute_checked().set(wnd, value);
	else qt_widget::set_attr(name,value);
	return *this;
}

std::string qt_checkbox::get_attr(const std::string& name)
{
	if(name == "text") return text_;
	if(name == "font") return attribute_font().get(wnd);
	else if(name == "pixmap") return attribute_pixmap().get(wnd);
	else if(name == "checked") return attribute_checked().get(wnd); 
	else return qt_widget::get_attr(name);
}

QWidget *qt_checkbox::get_widget(void) const
{
	return wnd;
}

//
// ------------
//

qt_progressbar::qt_progressbar(QWidget *parent): wnd(new QProgressBar(parent))
{
	wnd->setBackgroundOrigin(QWidget::WindowOrigin);
}

widget& qt_progressbar::set_attr(const std::string& name, const std::string& value, const std::string&)
{
	if(name == "total") wnd->setTotalSteps(str2num(value));
	else if(name == "value") wnd->setProgress(str2num(value));
	else if(name == "font") attribute_font().set(wnd, value);
	else qt_widget::set_attr(name,value);
	return *this;
}

std::string qt_progressbar::get_attr(const std::string& name)
{
	if(name == "total") return num2str(wnd->totalSteps());
	if(name == "value") return num2str(wnd->progress());
	if(name == "font") return attribute_font().get(wnd);
	else return qt_widget::get_attr(name);
}

QWidget *qt_progressbar::get_widget(void) const
{
	return wnd;
}


//
// ------------
//

qt_lineedit::qt_lineedit(QWidget *parent): wnd(new QLineEdit(parent))
{
	wnd->setBackgroundOrigin(QWidget::WindowOrigin);
	connect(wnd, SIGNAL(textChanged(const QString&)),
			SLOT(on_change(const QString&)));
	connect(wnd, SIGNAL(returnPressed()), SLOT(on_return()));
}

widget& qt_lineedit::set_attr(const std::string& name, const std::string& value, const std::string&)
{
	if(name == "text") attribute_text().set(wnd, value);
	else if(name == "align") attribute_align().set(wnd, value);
	else if(name == "echo") attribute_echo().set(wnd, value);
	else if(name == "readonly") attribute_readonly().set(wnd, value);
	else if(name == "mask") attribute_mask().set(wnd, value);
	else qt_widget::set_attr(name,value);
	return *this;
}

std::string qt_lineedit::get_attr(const std::string& name)
{
	if(name == "text") return attribute_text().get(wnd);
	else if(name == "align") return attribute_align().get(wnd);
	else if(name == "echo") return attribute_echo().get(wnd);
	else if(name == "readonly") return attribute_readonly().get(wnd);
	else if(name == "mask") return attribute_mask().get(wnd);
	else return qt_widget::get_attr(name);
}

QWidget *qt_lineedit::get_widget(void) const
{
	return wnd;
}

//
// ------------
//

qt_textbox::qt_textbox(QWidget *parent): wnd(new QTextEdit(parent))
{
	wnd->setBackgroundOrigin(QWidget::WindowOrigin);
	connect(wnd, SIGNAL(textChanged()),SLOT(on_change()));
	connect(wnd, SIGNAL(returnPressed()), SLOT(on_return()));
}

widget& qt_textbox::set_attr(const std::string& name, const std::string& value, const std::string&)
{
	if(name == "text") attribute_text().set(wnd, value);
	else if(name == "text-append") wnd->append(value);
	else if(name == "align") attribute_align().set(wnd, value);
	else if(name == "readonly") attribute_readonly().set(wnd, value);
	else qt_widget::set_attr(name,value);
	return *this;
}

std::string qt_textbox::get_attr(const std::string& name)
{
	if(name == "text") return attribute_text().get(wnd);
	else if(name == "align") return attribute_align().get(wnd);
	else if(name == "readonly") return attribute_readonly().get(wnd);
	else return qt_widget::get_attr(name);
}

QWidget *qt_textbox::get_widget(void) const
{
	return wnd;
}


//
// ------------
//

qt_listbox::qt_listbox(QWidget *parent): wnd(new QListBox(parent))
{
	wnd->setBackgroundOrigin(QWidget::WindowOrigin);
	connect(wnd, SIGNAL(selectionChanged()), SLOT(on_select()));
	connect(wnd,
	        SIGNAL(doubleClicked(QListBoxItem*)),
		SLOT(on_dbl_click(QListBoxItem*)));
}

void qt_listbox::on_dbl_click(QListBoxItem*)
{
	qApp->exit(get_event("on-double-click"));
}

widget& qt_listbox::set_attr(const std::string& name, const std::string& value, const std::string& subctrlid)
{
	if (subctrlid.empty())
	{//general settings
		if(name == "font") attribute_font().set(wnd, value);
		else if(name == "current") attribute_current().set(wnd, value);
		else if(name == "remove") {
			if (value == "all") wnd->clear();
		}
		else qt_widget::set_attr(name,value);
	}
	else
	{//listbox item settings
		if (name == "text")
			wnd->changeItem(*wnd->item(str2num(subctrlid))->pixmap(),
					value,
					str2num(subctrlid));
		else if (name == "pixmap")
			wnd->changeItem(QPixmap(value),
					wnd->item(str2num(subctrlid))->text(),
					str2num(subctrlid));
	}
	return *this;
}

std::string qt_listbox::get_attr(const std::string& name)
{
	if(name == "text") return attribute_curtext().get(wnd);
	else if(name == "count") attribute_count().get(wnd);
	else if(name == "font") return attribute_font().get(wnd);
	else if(name == "current") return attribute_current().get(wnd);
	return qt_widget::get_attr(name);
}

QWidget *qt_listbox::get_widget(void) const
{
	return wnd;
}

widget& qt_listbox::insert(widget *w)
{
	qt_label *q = dynamic_cast<qt_label *>(w);
	if(q)
	{
		QLabel *l = dynamic_cast<QLabel *>(q->get_widget());
		l->hide();
		if(l->pixmap()) wnd->insertItem(*l->pixmap(),
		                                QString::fromUtf8(q->get_attr("text").c_str()));
		else wnd->insertItem(l->text());
	}
	return *this;
}

//
// ------------
//

qt_combobox::qt_combobox(QWidget *parent): wnd(new QComboBox(parent))
{
	wnd->setBackgroundOrigin(QWidget::WindowOrigin);
	connect(wnd, SIGNAL(activated(const QString&)), SLOT(on_change(const QString&)));
}

widget& qt_combobox::set_attr(const std::string& name, const std::string& value, const std::string&)
{
	if(name == "font") attribute_font().set(wnd, value);
	else if(name == "current") attribute_current().set(wnd, value);
	else if(name == "editable") attribute_editable().set(wnd, value);
	else qt_widget::set_attr(name,value);
	return *this;
}

std::string qt_combobox::get_attr(const std::string& name)
{
	if(name == "text") return attribute_curtext().get(wnd);
	else if(name == "count") attribute_count().get(wnd);
	else if(name == "font") return attribute_font().get(wnd);
	else if(name == "current") return attribute_current().get(wnd);
	else if(name == "editable") return attribute_editable().get(wnd);
	return qt_widget::get_attr(name);
}

QWidget *qt_combobox::get_widget(void) const
{
	return wnd;
}

widget& qt_combobox::insert(widget *w)
{
	qt_label *q = dynamic_cast<qt_label *>(w);
	if(q)
	{
		QLabel *l = dynamic_cast<QLabel *>(q->get_widget());
		l->hide();
		if(l->pixmap()) wnd->insertItem(*l->pixmap(),
		                                QString::fromUtf8(q->get_attr("text").c_str()));
		else wnd->insertItem(l->text());
	}
	return *this;
}


//
// ------------
//

qt_vbox::qt_vbox(QWidget *parent): wnd(new QFrame(parent))
{
	wnd->setBackgroundOrigin(QWidget::WindowOrigin);
	new QVBoxLayout(wnd);
	wnd->layout()->setAlignment(Qt::AlignTop);
}

widget& qt_vbox::set_attr(const std::string& name, const std::string& value, const std::string&)
{
	if (name == "margin") attribute_margin().set(wnd->layout(), value);
	else if (name == "spacing") attribute_spacing().set(wnd->layout(), value);
	else if (name == "sizepolicy") attribute_sizepolicy().set(wnd, value);
	else qt_widget::set_attr(name,value);
	return *this;
}

std::string qt_vbox::get_attr(const std::string& name)
{
	if(name == "margin") return attribute_margin().get(wnd->layout());
	else if(name == "spacing") return attribute_spacing().get(wnd->layout());
	
	return qt_widget::get_attr(name);
}

QWidget *qt_vbox::get_widget(void) const
{
	return wnd;
}

widget& qt_vbox::insert(widget *w)
{
	qt_widget *q = dynamic_cast<qt_widget *>(w);
	QWidget *wid = 0;
	if(q && (wid=q->get_widget()))
	{
		bool hidden = wid->isHidden();
		wid->reparent(wnd, QPoint());
		if (hidden) wid->hide(); //bug in qt, hide hidden objects after reparent
		((QBoxLayout*)wnd->layout())->addWidget(wid);
	}
	return *this;
}

//
// ------------
//

qt_hbox::qt_hbox(QWidget *parent): wnd(new QFrame(parent))
{
	wnd->setBackgroundOrigin(QWidget::WindowOrigin);
	new QHBoxLayout(wnd);
	wnd->layout()->setAlignment(Qt::AlignTop);
}

widget& qt_hbox::set_attr(const std::string& name, const std::string& value, const std::string&)
{
	if(name == "margin") attribute_margin().set(wnd->layout(), value);
	else if(name == "spacing") attribute_spacing().set(wnd->layout(), value);
	else if (name == "sizepolicy") attribute_sizepolicy().set(wnd, value);
	else qt_widget::set_attr(name,value);
	return *this;
}

std::string qt_hbox::get_attr(const std::string& name)
{
	if(name == "margin") return attribute_margin().get(wnd->layout());
	else if(name == "spacing") return attribute_spacing().get(wnd->layout());
	return qt_widget::get_attr(name);
}

QWidget *qt_hbox::get_widget(void) const
{
	return wnd;
}

widget& qt_hbox::insert(widget *w)
{
	qt_widget *q = dynamic_cast<qt_widget *>(w);
	QWidget *wid = 0;
	if(q && (wid=q->get_widget()))
	{
		bool hidden = wid->isHidden();
		wid->reparent(wnd, QPoint());
		if (hidden) wid->hide(); //bug in qt, hide hidden objects after reparent
		((QBoxLayout*)wnd->layout())->addWidget(wid);
	}
	return *this;
}
