/*
 *  ALTerator - ALT Linux configuration project
 *
 *  Copyright (c) 2004,2005 ALT Linux Ltd.
 *  Copyright (c) 2004,2005 Alexey Voinov
 *  Copyright (c) 2004,2005 Stanislav Ievlev
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 *  USA.
 */
#include <locale.h>

#include <dialog-scm.hh>
#include <dialog-qt.hh>
#include <mapper.hh>
#include <command_io.hh>
#include <bus_io.hh>
#include <utils/str.hh>

#include <iostream>
#include <sstream>


#include <qapplication.h>

#define DBUS_API_SUBJECT_TO_CHANGE
#include <dbus/dbus.h>
#include <dbus/connection.h> 

template <typename T>
void set_global(const std::string& name,T value)
{
	call(scm::variable_ref("set-global!"),
		scm::scm(name),
		scm::scm(value));
}


extern scm::scm cmd_to_list(const command&);
dialog *load_dialog(QMainWindow *wnd, const std::string& filename)
{
	qt_dialog *d = new qt_dialog(wnd);
	call(scm::variable_ref("load-dialog"),
	     wrap_scm_dialog(d), scm::scm(filename));
	//re-center window
	wnd->move((QApplication::desktop()->width() - wnd->width())/2,
		  (QApplication::desktop()->height() - wnd->height())/2);
	return d;
}

class look
{
	mapper *m_;
	QMainWindow *wnd_;
	dialog *dlg_;

	enum cmd_result
	{
		stop, /**< stop work */
		go, /**< continue work */
		error /**< process errors */
	};
public:
	look(const std::string& filename)
		: m_(new scm_mapper(filename)),
		  wnd_(new QMainWindow()),
		  dlg_(0)
	{
		wnd_->show();
	}

	~look(void)
	{
		delete m_;
		delete dlg_;
		delete wnd_;
	}

	void process_bus_message(DBusMessage* message);
	command view(const command& cmd);
	void file(const command& cmd);
	void attr(const command& cmd);
	cmd_result send_event(const command& cmd);
	cmd_result perform(const command& cmd);
	void run(const command& cmd);

private:
	cmd_result perform_woo(const command& cmd);
	cmd_result perform_cmds(scm::scm cmds);
};

void look::file(const command& cmd)
{
	scm_c_define("*globals*", cmd_to_list(cmd).as_scm());
	dialog *newdlg = load_dialog(wnd_, cmd("name"));
	delete dlg_;
	dlg_ = newdlg;
}

command look::view(const command& cmd)
{
	try
	{
		command newcmd;
		m_->map(cmd("id"), newcmd);
		newcmd.add(cmd, context::keepold);
		return newcmd;
	}
	catch(const mapper::missing&)
	{
		std::cerr << "no such id: " << cmd("id") << std::endl;
	}
	return command("/ctrl/look")("action", "quit");
}

void look::attr(const command& cmd)
{
	try
	{
		//analyze ctrlid,search for sub-ctrlid
		const std::string ctrlid = cmd("ctrlid");
		std::string::size_type pos = ctrlid.find(":");
		const std::string real_ctrlid = (pos == std::string::npos)?ctrlid:ctrlid.substr(0,pos);
		const std::string subctrlid = (pos == std::string::npos)?"":ctrlid.substr(pos+1,std::string::npos);
		widget *w = dlg_->who_is_it(real_ctrlid);
		if(w) w->set_attr(cmd("field"), cmd("value"),subctrlid);
	}
	catch(...) //ignore command if some keys are missing
	{
	}
}

look::cmd_result look::send_event(const command &cmd)
{
	try
	{
		widget *w = dlg_->who_is_it(cmd("ctrlid"));
		if(w) return perform_cmds(call(scm::variable_ref("current-command-set-get"),
					       scm::scm(w->get_event(cmd("event"))),
					       wrap_scm_dialog(dlg_)));
	}
	catch(...) //ignore command if some keys are missing
	{
	}
	return go;
}

look::cmd_result look::perform(const command& cmd)
{
	if(cmd.name() == "/ctrl/look")
	{
		if(cmd("action") == "file") file(cmd);
		else if(cmd("action") == "view") return perform(view(cmd));
		else if(cmd("action") == "attr") attr(cmd);
		else if(cmd("action") == "send") return send_event(cmd);
		else if(cmd("action") == "quit") return stop;
		return go;
	}
	return perform_woo(cmd);
}

look::cmd_result look::perform_woo(const command& cmd)
{
	// FIXME: hacks-hacks-hacks...
	// I need to rewrite bus_io.
	std::ostringstream devnull;
	std::ostringstream message;
	message << cmd;
	send(">>", message.str(), std::cin, std::cout, devnull);

	while(std::cin)
	{
		std::string addr;
		getline(std::cin, addr);
		if(addr == "<<")
		{
			std::istringstream is(receive(std::cin, std::cout,
								devnull));
			command cmd;
			while(is >> cmd)
			{
				if(alt::starts_with(cmd.name(), "/error"))
				{
					if (cmd.exists("reason"))
						set_global("reason",cmd("reason"));
					return error;
				}
			}
			break;
		}
		else std::cout << "<>" << std::endl;
	}
	return go;
}

void look::run(const command& start)
{
	if(!perform(start)) return;
	if(!dlg_)
	{
		std::cerr << "no dialog to work with" << std::endl;
		return;
	}

	long result;
	while((result = dlg_->run()) != 0)
	{
		cmd_result res = perform_cmds(call(scm::variable_ref(
					     "current-command-set-get"),
				     scm::scm(result),
				     wrap_scm_dialog(dlg_)));
		if (res == stop)
			break;
		else if (res == error)
		{
			if(perform_cmds(call(scm::variable_ref("current-command-set-get"),
					 scm::scm(dlg_->get_event("on-error")),
					 wrap_scm_dialog(dlg_))) != go)
			break;
		}
	}
}

command& list_to_cmd(scm::scm, command&);

look::cmd_result look::perform_cmds(scm::scm cmds)
{
	while(pair_p(cmds))
	{
		command cmd;
		cmd_result res = perform(list_to_cmd(car(cmds), cmd));
		if (res != go) return res;
		cmds = cdr(cmds);
	}
	return go;
}

template <typename T>
void set_dbus_arg(int num,T val)
{
	std::ostringstream str;
	str<<"dbus-arg"<<num;
	set_global(str.str(),val);
}

void look::process_bus_message(DBusMessage* message)
{
	const char *sender = dbus_message_get_sender(message);
	const char *destination = dbus_message_get_destination(message);
	const char *interface = dbus_message_get_interface(message);
	const char *member = dbus_message_get_member(message);
	
	set_global("dbus-sender",(sender?sender:""));
	set_global("dbus-destination",(destination?destination:""));
	set_global("dbus-interface",(interface?interface:""));
	set_global("dbus-member",(member?member:""));

	
	DBusMessageIter iter;
	dbus_message_iter_init (message, &iter);
	int count=0;
	do
	{
		switch (dbus_message_iter_get_arg_type (&iter))
		{
			case DBUS_TYPE_STRING:
				{
					const char *str;
          				//dbus_message_iter_get_basic (&iter, &str); -- new interface
					str=dbus_message_iter_get_string(&iter); // -- old interface
					set_dbus_arg(count,str);
				}
				break;
			case DBUS_TYPE_INT32:
				{
					dbus_int32_t int32;
					//dbus_message_iter_get_basic (&iter, &int32); -- new interface
	  				int32 = dbus_message_iter_get_int32 (&iter);// -- old inteface
					set_dbus_arg(count,int32);
				}
				break;
			case DBUS_TYPE_BOOLEAN:
				{
					dbus_bool_t b;
					//dbus_message_iter_get_basic (&iter, &b); -- new interface
	  				b = dbus_message_iter_get_boolean (&iter);// -- old interface
					set_dbus_arg(count,b);
				}
				break;
			default:
				set_dbus_arg(count,"#unsupported#");
				break;
		}
		++count;
	}while (dbus_message_iter_next (&iter));
	set_global("dbus-args-count",count);
	perform_cmds(call(scm::variable_ref("current-command-set-get"),
	   		  scm::scm(dlg_->get_event("on-dbus")),
	   		  wrap_scm_dialog(dlg_)));
}



static
DBusHandlerResult dbus_filter_func (DBusConnection *connection,DBusMessage *message,void *user_data)
{
	look* l = (look*)(user_data);
	std::cerr<<"dbus_filter_func\n";
	l->process_bus_message(message);
	//#define DBUS_INTERFACE_LOCAL "org.freedesktop.DBus.Local"
	//  if (dbus_message_is_signal (message,DBUS_INTERFACE_LOCAL,"Disconnected")) exit (0);
	return DBUS_HANDLER_RESULT_HANDLED;
}

//FIXME: currently we will work only with system bus
void init_dbus(look *l,DBusBusType type)
{
	DBusConnection *connection;
	DBusError error;

	dbus_error_init (&error);
	connection = dbus_bus_get (type, &error);
	if (connection == NULL)
	{
		std::cerr<<"Failed to open connection to message bus:"<<
		((type == DBUS_BUS_SYSTEM) ? "system" : "session")
		<<"message:"<<error.message<<std::endl;
		dbus_error_free (&error);
		return; //ignore
	}
	dbus_bus_add_match(connection,"type='signal'",&error);   
	if (dbus_error_is_set (&error))
  	{
		std::cerr<<"Failed to add dbus match:"<<error.message<<std::endl;
		dbus_error_free (&error);
    		exit(1);
  	}

  	if (!dbus_connection_add_filter(connection, dbus_filter_func,l, NULL))
	{
		std::cerr<<"Failed to add dbus filter:"<<std::endl;
          	exit(1);
	}

	DBusQt::Connection   *conn = new DBusQt::Connection();//Note: we don't need to destruct this object
	conn->dbus_connection_setup_with_qt_main(connection);
}

extern void init_scm_woo(void);
extern void init_scm_gettext(void);
extern void init_scm_glob(void);


int main(int argc, char *argv[])
{
	setlocale(LC_ALL,"");
	QApplication a(argc, argv);
	scm::init();

	init_scm_dialog();
	init_scm_woo();
	init_scm_gettext();
	init_scm_glob();
	
	scm::load(scm::commondir+"dialog.scm");
	scm::load(scm::commondir+"mapper.scm");

	a.connect(&a, SIGNAL(lastWindowClosed()), &a, SLOT(quit()));
	look l((argc>=2)?argv[1]:"testmap.scm");
	init_dbus(&l,DBUS_BUS_SYSTEM);
	l.run(command("/ctrl/look")("action", "view")("id", "/"));

	std::cout << "xx" << std::endl;
	return 0;
}
