/*
 *  ALTerator - ALT Linux configuration project
 *
 *  Copyright (c) 2004,2005 ALT Linux Ltd.
 *  Copyright (c) 2004,2005 Alexey Voinov
 *  Copyright (c) 2004,2005 Stanislav Ievlev
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 *  USA.
 */

struct attribute_text
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		widget->setText(QString::fromUtf8(value.c_str()));
	}

	template <typename W>
	std::string get(W *widget)
	{
		return widget->text().utf8().data();
	}
};

struct attribute_mask
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		widget->setInputMask(value);
	}

	template <typename W>
	std::string get(W *widget)
	{
		return widget->inputMask();
	}
};

struct attribute_margin
{
	template <typename L>
	void set(L *layout, const std::string& value)
	{
		QBoxLayout *l = dynamic_cast<QBoxLayout *>(layout);
		if(l) l->setMargin(percent_to_height(str2num(value)));
	}

	template <typename L>
	std::string get(L *layout)
	{
		QBoxLayout *l = dynamic_cast<QBoxLayout *>(layout);
		if(l) return num2str(height_to_percent(l->margin()));
		return "";
	}
};

struct attribute_spacing
{
	template <typename L>
	void set(L *layout, const std::string& value)
	{
		QBoxLayout *l = dynamic_cast<QBoxLayout *>(layout);
		if(l) l->setSpacing(percent_to_height(str2num(value)));
	}

	template <typename L>
	std::string get(L *layout)
	{
		QBoxLayout *l = dynamic_cast<QBoxLayout *>(layout);
		if(l) return num2str(height_to_percent(l->spacing()));
		return "";
	}
};

struct attribute_caption
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		widget->setCaption(value);
	}

	template <typename W>
	std::string get(W *widget)
	{
		return widget->caption();
	}
};

struct attribute_align
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		if(value == "left") _set(widget, Qt::AlignLeft);
		else if(value == "right") _set(widget, Qt::AlignRight);
		else if(value == "center") _set(widget, Qt::AlignHCenter);
		else if(value == "justify") _set(widget, Qt::AlignJustify);
		else if(value == "auto") _set(widget, Qt::AlignAuto);
	}

	template <typename W>
	std::string get(W *widget)
	{
		int f = widget->alignment();
		if(f & Qt::AlignLeft) return "left";
		else if(f & Qt::AlignRight) return "right";
		else if(f & Qt::AlignHCenter) return "center";
		else if(f & Qt::AlignJustify) return "justify";
		else return "auto";
	}

private:
	template <typename W>
	void _set(W *widget, Qt::AlignmentFlags f)
	{
		widget->setAlignment((widget->alignment()
				     & (~Qt::AlignHorizontal_Mask)) | f);
	}
};

struct attribute_width
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		widget->setFixedWidth(percent_to_width(str2num(value)));
	}

	template <typename W>
	std::string get(W *widget)
	{
		return num2str(width_to_percent(widget->width()));
	}
};

struct attribute_height
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		widget->setFixedHeight(percent_to_height(str2num(value)));
	}

	template <typename W>
	std::string get(W *widget)
	{
		return num2str(height_to_percent(widget->height()));
	}
};

struct attribute_font
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		QFont font;
		font.setRawName(value);
		widget->setFont(font);
	}

	template <typename W>
	std::string get(W *widget)
	{
		QFont font = widget->font();
		return font.rawName();
	}
};

struct attribute_default
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		if(value == "yes") widget->setDefault(true);
		else if(value == "no") widget->setDefault(false);
	}

	template <typename W>
	std::string get(W *widget)
	{
		return widget->isDefault() ? "yes" : "no";
	}
};

struct attribute_readonly
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		if(value == "yes") widget->setReadOnly(true);
		else if(value == "no") widget->setReadOnly(false);
	}

	template <typename W>
	std::string get(W *widget)
	{
		return widget->isReadOnly() ? "yes" : "no";
	}
};

struct attribute_editable
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		if(value == "yes") widget->setEditable(true);
		else if(value == "no") widget->setEditable(false);
	}

	template <typename W>
	std::string get(W *widget)
	{
		return widget->editable() ? "yes" : "no";
	}
};

struct attribute_enabled
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		if(value == "yes") widget->setEnabled(true);
		else if(value == "no") widget->setEnabled(false);
	}

	template <typename W>
	std::string get(W *widget)
	{
		return widget->isEnabled() ? "yes" : "no";
	}
};

struct attribute_hidden
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		if(value == "yes") widget->hide();
		else if(value == "no") widget->show();
	}

	template <typename W>
	std::string get(W *widget)
	{
		return widget->isHidden() ? "yes" : "no";
	}
};


struct attribute_flat
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		if(value == "yes") widget->setFlat(true);
		else if(value == "no") widget->setFlat(false);
	}

	template <typename W>
	std::string get(W *widget)
	{
		return widget->isFlat() ? "yes" : "no";
	}
};

struct attribute_echo
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		if(value == "yes") widget->setEchoMode(QLineEdit::Normal);
		else if(value == "no") widget->setEchoMode(QLineEdit::NoEcho);
		else if(value == "stars") widget->setEchoMode(QLineEdit::Password);
	}

	template <typename W>
	std::string get(W *widget)
	{
		int f = widget->echoMode();
		if(f == QLineEdit::Normal) return "yes";
		else if(f == QLineEdit::NoEcho) return "no";
		else if(f == QLineEdit::Password) return "stars";
		return "";
	}
};


struct attribute_sizepolicy
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		QSizePolicy::SizeType type = QSizePolicy::Preferred;

		if (value == "fixed") type = QSizePolicy::Fixed; 
		else if (value == "expanding") type = QSizePolicy::Expanding;

		widget->setSizePolicy(type,type);
	}

	template <typename W>
	std::string get(W *widget)
	{
		return "";
	}
};


struct attribute_pixmap
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		widget->setPixmap(QPixmap(value));
	}

	template <typename W>
	std::string get(W *widget)
	{
		const QPixmap *pixmap = widget->pixmap();
		return pixmap ? "yes" : "no";
	}
};

struct attribute_backroundpixmap
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		widget->setPaletteBackgroundPixmap(QPixmap(value));
	}

	template <typename W>
	std::string get(W *widget)
	{
		const QPixmap *pixmap = widget->paletteBackgroundPixmap();
		return pixmap ? "yes" : "no";
	}
};

struct attribute_count
{
	template <typename W>
	std::string get(W *widget)
	{
		return num2str(widget->count());
	}
};

struct attribute_curtext
{
	template <typename W>
	std::string get(W *widget)
	{
		return widget->currentText().utf8().data();
	}
};

struct attribute_current
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		widget->setCurrentItem(str2num(value));
	}

	template <typename W>
	std::string get(W *widget)
	{
		return num2str(widget->currentItem());
	}
};

struct attribute_checked
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		if(value == "yes") widget->setChecked(true);
		else if(value == "no") widget->setChecked(false);
	}

	template <typename W>
	std::string get(W *widget)
	{
		return widget->isChecked() ? "yes" : "no";
	}
};
