/*
 *  ALTerator - ALT Linux configuration project
 *
 *  Copyright (c) 2004,2005 ALT Linux Ltd.
 *  Copyright (c) 2004,2005 Alexey Voinov
 *  Copyright (c) 2004,2005 Stanislav Ievlev
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 *  USA.
 */
#include <iostream>
#include <sstream>
#include <qapplication.h>

#include <qtlook.hh>

namespace
{
	long percent_to_width(long width)
	{
		return ( QApplication::desktop()->width() * width ) / 100;
	}
	
	long percent_to_height(long height)
	{
		return ( QApplication::desktop()->height() * height ) / 100;
	}

	long width_to_percent(long width)
	{
		return ( QApplication::desktop()->width() * 100 ) / width;
	}
	
	long height_to_percent(long height)
	{
		return ( QApplication::desktop()->height() * 100 ) / height;
	}
	
	long str2num(const std::string& str)
	{
		long result;
		std::istringstream is(str);
		is >> result;
		return result;
	}

	std::string num2str(long num)
	{
		std::ostringstream os;
		os << num;
		return os.str();
	}
}

#include <qtattrs.hh>


#define CONSTRUCTOR(NAME) \
	widget *make_##NAME(void) \
	{ \
		return new qt_##NAME(); \
	}

CONSTRUCTOR(button)
CONSTRUCTOR(label)
CONSTRUCTOR(lineedit)
CONSTRUCTOR(textbox)
CONSTRUCTOR(checkbox)
CONSTRUCTOR(progressbar)
CONSTRUCTOR(listbox)
CONSTRUCTOR(combobox)
CONSTRUCTOR(vbox)
CONSTRUCTOR(hbox)

//
// ------------
//

void qt_widget::insert(widget *)
{
	std::cerr<<"warning: attempt to insert to non-containter widget"<<std::endl;
}

#define EVENT_CALLBACK(NAME,ARGS,EVENT) \
	void qt_widget::on_##NAME(ARGS) \
	{ \
		call_event_holder(#EVENT); \
	}

EVENT_CALLBACK(click,void,on-click)
EVENT_CALLBACK(change,const QString&,on-change)
EVENT_CALLBACK(change,void,on-change)
EVENT_CALLBACK(return,void,on-return)
EVENT_CALLBACK(select,void,on-select)
EVENT_CALLBACK(double_click,QListBoxItem*,on-double-click)

void qt_widget::set_attr(const std::string& name, const std::string& value)
{
	if(name == "width") attribute_width().set(get_widget(), value);
	else if(name == "hidden") attribute_hidden().set(get_widget(), value);
	else if(name == "height") attribute_height().set(get_widget(), value);
	else if(name == "enabled") attribute_enabled().set(get_widget(), value);
}

std::string qt_widget::get_attr(const std::string& name)
{
	if(name == "width") return attribute_width().get(get_widget());
	else if(name == "hidden") return attribute_hidden().get(get_widget());
	else if(name == "height") return attribute_height().get(get_widget());
	else if(name == "enabled") return attribute_enabled().get(get_widget());
	return "";
}

//
// ------------
//

qt_button::qt_button(QWidget *parent): wnd(new QPushButton(parent))
{
	wnd->setBackgroundOrigin(QWidget::WindowOrigin);
	connect(wnd, SIGNAL(clicked()), SLOT(on_click()));
}

void qt_button::set_attr(const std::string& name, const std::string& value)
{
	if(name == "text") attribute_text().set(wnd, value);
	else if(name == "default") attribute_default().set(wnd, value);
	else if(name == "flat") attribute_default().set(wnd, value);
	else if(name == "font") attribute_font().set(wnd, value);
	else if(name == "pixmap") attribute_pixmap().set(wnd, value);
	else qt_widget::set_attr(name,value);
}

std::string qt_button::get_attr(const std::string& name)
{
	if(name == "text") return attribute_text().get(wnd);
	else if(name == "default") return attribute_default().get(wnd);
	else if(name == "flat") return attribute_flat().get(wnd);
	else if(name == "font") return attribute_font().get(wnd);
	else if(name == "pixmap") return attribute_pixmap().get(wnd);
	else return qt_widget::get_attr(name);
}

//
// ------------
//

qt_label::qt_label(QWidget *parent): wnd(new QLabel(parent))
{
	wnd->setBackgroundOrigin(QWidget::WindowOrigin);
}

void qt_label::set_attr(const std::string& name, const std::string& value)
{
	if(name == "text") { attribute_text().set(wnd, value); text_ = value; }
	else if(name == "font") attribute_font().set(wnd, value);
	else if(name == "pixmap") attribute_pixmap().set(wnd, value);
	else if(name == "align") attribute_align().set(wnd, value);
	else qt_widget::set_attr(name,value);
}

std::string qt_label::get_attr(const std::string& name)
{
	if(name == "text") return text_;
	if(name == "font") return attribute_font().get(wnd);
	else if(name == "align") return attribute_align().get(wnd);
	else if(name == "pixmap") return attribute_pixmap().get(wnd);
	else return qt_widget::get_attr(name);
}

//
// ------------
//

qt_checkbox::qt_checkbox(QWidget *parent): wnd(new QCheckBox(parent))
{
	connect(wnd, SIGNAL(clicked()), SLOT(on_click()));
	wnd->setBackgroundOrigin(QWidget::WindowOrigin);
}

void qt_checkbox::set_attr(const std::string& name, const std::string& value)
{
	if(name == "text") { attribute_text().set(wnd, value); text_ = value; }
	else if(name == "font") attribute_font().set(wnd, value);
	else if(name == "pixmap") attribute_pixmap().set(wnd, value);
	else if(name == "checked") attribute_checked().set(wnd, value);
	else qt_widget::set_attr(name,value);
}

std::string qt_checkbox::get_attr(const std::string& name)
{
	if(name == "text") return text_;
	if(name == "font") return attribute_font().get(wnd);
	else if(name == "pixmap") return attribute_pixmap().get(wnd);
	else if(name == "checked") return attribute_checked().get(wnd); 
	else return qt_widget::get_attr(name);
}

//
// ------------
//

qt_progressbar::qt_progressbar(QWidget *parent): wnd(new QProgressBar(parent))
{
	wnd->setBackgroundOrigin(QWidget::WindowOrigin);
}

void qt_progressbar::set_attr(const std::string& name, const std::string& value)
{
	if(name == "total") wnd->setTotalSteps(str2num(value));
	else if(name == "value") wnd->setProgress(str2num(value));
	else if(name == "font") attribute_font().set(wnd, value);
	else qt_widget::set_attr(name,value);
}

std::string qt_progressbar::get_attr(const std::string& name)
{
	if(name == "total") return num2str(wnd->totalSteps());
	if(name == "value") return num2str(wnd->progress());
	if(name == "font") return attribute_font().get(wnd);
	else return qt_widget::get_attr(name);
}

//
// ------------
//

qt_lineedit::qt_lineedit(QWidget *parent): wnd(new QLineEdit(parent))
{
	wnd->setBackgroundOrigin(QWidget::WindowOrigin);
	connect(wnd, SIGNAL(textChanged(const QString&)),
			SLOT(on_change(const QString&)));
	connect(wnd, SIGNAL(returnPressed()), SLOT(on_return()));
}

void qt_lineedit::set_attr(const std::string& name, const std::string& value)
{
	if(name == "text") attribute_text().set(wnd, value);
	else if(name == "align") attribute_align().set(wnd, value);
	else if(name == "echo") attribute_echo().set(wnd, value);
	else if(name == "readonly") attribute_readonly().set(wnd, value);
	else if(name == "mask") attribute_mask().set(wnd, value);
	else qt_widget::set_attr(name,value);
}

std::string qt_lineedit::get_attr(const std::string& name)
{
	if(name == "text") return attribute_text().get(wnd);
	else if(name == "align") return attribute_align().get(wnd);
	else if(name == "echo") return attribute_echo().get(wnd);
	else if(name == "readonly") return attribute_readonly().get(wnd);
	else if(name == "mask") return attribute_mask().get(wnd);
	else return qt_widget::get_attr(name);
}

//
// ------------
//

qt_textbox::qt_textbox(QWidget *parent): wnd(new QTextEdit(parent))
{
	wnd->setBackgroundOrigin(QWidget::WindowOrigin);
	connect(wnd, SIGNAL(textChanged()),SLOT(on_change()));
	connect(wnd, SIGNAL(returnPressed()), SLOT(on_return()));
}

void qt_textbox::set_attr(const std::string& name, const std::string& value)
{
	if(name == "text") attribute_text().set(wnd, value);
	else if(name == "text-append") wnd->append(value);
	else if(name == "align") attribute_align().set(wnd, value);
	else if(name == "readonly") attribute_readonly().set(wnd, value);
	else qt_widget::set_attr(name,value);
}

std::string qt_textbox::get_attr(const std::string& name)
{
	if(name == "text") return attribute_text().get(wnd);
	else if(name == "align") return attribute_align().get(wnd);
	else if(name == "readonly") return attribute_readonly().get(wnd);
	else return qt_widget::get_attr(name);
}

//
// ------------
//

qt_listbox::qt_listbox(QWidget *parent): wnd(new QListBox(parent))
{
	wnd->setBackgroundOrigin(QWidget::WindowOrigin);
	connect(wnd, SIGNAL(selectionChanged()), SLOT(on_select()));
	connect(wnd,
	        SIGNAL(doubleClicked(QListBoxItem*)),
		SLOT(on_dbl_click(QListBoxItem*)));
}

void qt_listbox::set_attr(const std::string& name, const std::string& value)
{
	if(name == "font") attribute_font().set(wnd, value);
	else if(name == "current") attribute_current().set(wnd, value);
	else if(name == "remove") { if (value == "all") wnd->clear(); }
	else qt_widget::set_attr(name,value);
}

void qt_listbox::set_attr_sub(const std::string& name, const std::string& value,int num)
{
	if (name == "text")
		wnd->changeItem(*wnd->item(num)->pixmap(),
				value,
				num);
	else if (name == "pixmap")
			wnd->changeItem(QPixmap(value),
					wnd->item(num)->text(),
					num);
}

std::string qt_listbox::get_attr(const std::string& name)
{
	if(name == "text") return attribute_curtext().get(wnd);
	else if(name == "count") attribute_count().get(wnd);
	else if(name == "font") return attribute_font().get(wnd);
	else if(name == "current") return attribute_current().get(wnd);
	return qt_widget::get_attr(name);
}

void qt_listbox::insert(widget *w)
{
	qt_label *q = dynamic_cast<qt_label *>(w);
	if (!q) return;

	QLabel *l = dynamic_cast<QLabel *>(q->get_widget());
	if (!l) return;

	l->hide();
	if(l->pixmap()) wnd->insertItem(*l->pixmap(),
		                        QString::fromUtf8(q->get_attr("text").c_str()));
	else wnd->insertItem(l->text());
}

//
// ------------
//

qt_combobox::qt_combobox(QWidget *parent): wnd(new QComboBox(parent))
{
	wnd->setBackgroundOrigin(QWidget::WindowOrigin);
	connect(wnd, SIGNAL(activated(const QString&)), SLOT(on_change(const QString&)));
}

void qt_combobox::set_attr(const std::string& name, const std::string& value)
{
	if(name == "font") attribute_font().set(wnd, value);
	else if(name == "current") attribute_current().set(wnd, value);
	else if(name == "editable") attribute_editable().set(wnd, value);
	else qt_widget::set_attr(name,value);
}

std::string qt_combobox::get_attr(const std::string& name)
{
	if(name == "text") return attribute_curtext().get(wnd);
	else if(name == "count") attribute_count().get(wnd);
	else if(name == "font") return attribute_font().get(wnd);
	else if(name == "current") return attribute_current().get(wnd);
	else if(name == "editable") return attribute_editable().get(wnd);
	return qt_widget::get_attr(name);
}

void qt_combobox::insert(widget *w)
{
	qt_label *q = dynamic_cast<qt_label *>(w);
	if (!q) return;

	QLabel *l = dynamic_cast<QLabel *>(q->get_widget());
	if (!l) return;
	
	l->hide();
	if(l->pixmap()) wnd->insertItem(*l->pixmap(),
	                                QString::fromUtf8(q->get_attr("text").c_str()));
	else wnd->insertItem(l->text());
}


//
// ------------
//

qt_vbox::qt_vbox(QWidget *parent): wnd(new QFrame(parent))
{
	wnd->setBackgroundOrigin(QWidget::WindowOrigin);
	new QVBoxLayout(wnd);
	wnd->layout()->setAlignment(Qt::AlignTop);
}

void qt_vbox::set_attr(const std::string& name, const std::string& value)
{
	if (name == "margin") attribute_margin().set(wnd->layout(), value);
	else if (name == "spacing") attribute_spacing().set(wnd->layout(), value);
	else if (name == "sizepolicy") attribute_sizepolicy().set(wnd, value);
	else qt_widget::set_attr(name,value);
}

std::string qt_vbox::get_attr(const std::string& name)
{
	if(name == "margin") return attribute_margin().get(wnd->layout());
	else if(name == "spacing") return attribute_spacing().get(wnd->layout());
	return qt_widget::get_attr(name);
}

void qt_vbox::insert(widget *w)
{
	qt_widget *q = dynamic_cast<qt_widget *>(w);
	if (!q) return;

	QWidget *wid=q->get_widget();
	if (!wid) return;

	bool hidden = wid->isHidden();
	wid->reparent(wnd, QPoint());
	if (hidden) wid->hide(); //bug in qt, hide hidden objects after reparent
	((QBoxLayout*)wnd->layout())->addWidget(wid);
}

//
// ------------
//

qt_hbox::qt_hbox(QWidget *parent): wnd(new QFrame(parent))
{
	wnd->setBackgroundOrigin(QWidget::WindowOrigin);
	new QHBoxLayout(wnd);
	wnd->layout()->setAlignment(Qt::AlignTop);
}

void qt_hbox::set_attr(const std::string& name, const std::string& value)
{
	if(name == "margin") attribute_margin().set(wnd->layout(), value);
	else if(name == "spacing") attribute_spacing().set(wnd->layout(), value);
	else if (name == "sizepolicy") attribute_sizepolicy().set(wnd, value);
	else qt_widget::set_attr(name,value);
}

std::string qt_hbox::get_attr(const std::string& name)
{
	if(name == "margin") return attribute_margin().get(wnd->layout());
	else if(name == "spacing") return attribute_spacing().get(wnd->layout());
	return qt_widget::get_attr(name);
}

void qt_hbox::insert(widget *w)
{
	qt_widget *q = dynamic_cast<qt_widget *>(w);
	if (!q) return;

	QWidget *wid = q->get_widget();
	if (!wid) return;

	bool hidden = wid->isHidden();
	wid->reparent(wnd, QPoint());
	if (hidden) wid->hide(); //bug in qt, hide hidden objects after reparent
	((QBoxLayout*)wnd->layout())->addWidget(wid);
}
