/*
 *  ALTerator - ALT Linux configuration project
 *
 *  Copyright (c) 2004,2005 ALT Linux Ltd.
 *  Copyright (c) 2004,2005 Alexey Voinov
 *  Copyright (c) 2004,2005 Stanislav Ievlev
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 *  USA.
 */
#ifndef EVAL_HH
#define EVAL_HH

#include <string>
#include <list>
#include <map>
#include <stdexcept>
#include <iosfwd>
#include <context.hh>

#include <utils/functor.hh>

struct result
{
	enum {eof, string, list} t_;
	std::string str_;
	std::list<result> lst_;

	result(void): t_(eof) {}
	result(const std::string& str): t_(string), str_(str) {}
	result(const std::list<result>& lst): t_(list), lst_(lst) {}

	bool is_eof(void) const { return t_ == eof; }
	bool is_string(void) const { return t_ == string; }
	bool is_list(void) const { return t_ == list; }
};

struct parse_error: public std::runtime_error
{
	parse_error(const std::string& str): std::runtime_error(str) {}
};

struct eval_error: public std::runtime_error
{
	eval_error(const std::string& str): std::runtime_error(str) {}
};

struct evaluator;

typedef result (*func)(const std::list<result>&, const evaluator&);
//typedef alt::functor<result,const std::list<result>&,const evaluator&> func;

struct evaluator
{
	std::map<std::string, func> macrotab_;
	std::map<std::string, func> functab_;
	std::map<std::string, context> ctxttab_;

	std::string
	operator()(const std::string&) const;

	result
	operator()(const result&) const;

	bool is_special(const std::string& name) const
	{
		return macrotab_.find(name) != macrotab_.end();
	}

	bool is_function(const std::string& name) const
	{
		return macrotab_.find(name) != macrotab_.end();
	}

	func special(const std::string&) const;
	func function(const std::string&) const;
};


result
func_lookup(const std::list<result>&, const evaluator&);

void
print(std::ostream& os, const result& r);

result
read(std::istream& is);

#endif // EVAL_HH
