;;;
;;; ALTerator - ALT Linux configuration project
;;;
;;; Copyright (c) 2004,2005 ALT Linux Ltd.
;;; Copyright (c) 2004,2005 Alexey Voinov
;;; Copyright (c) 2004,2005 Stanislav Ievlev
;;;
;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2 of the License, or
;;; (at your option) any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
;;; USA.
;;;
(use-modules (ice-9 debug))
;(define *sandbox* #f)

;(defmacro later x `(lambda () ,@x))

;(define (sandbox)
;  (if *sandbox* *sandbox*
;      (begin (set! *sandbox* (make-module))
;	     (module-define! *sandbox* 'button button)
;	     (module-define! *sandbox* 'later later)
;	     *sandbox*)))

(dynamic-call "scm_init_glob"
              (dynamic-link "libguile-glob.so"))


(define *globals* '())

(define (current-dialog) #f)

(define (set-current-dialog! dlg)
  (set! current-dialog (lambda () dlg)))

(define (current-command-set) #f)

(define (set-current-command-set! cs)
  (set! current-command-set (lambda () cs)))

(define (with-dialog dlg thunk)
  (let ((saved-dlg #f))
    (dynamic-wind
      (lambda ()
	(set! saved-dlg (current-dialog))
	(set-current-dialog! dlg))
      thunk
      (lambda ()
	(set-current-dialog! saved-dlg)))))

(define (allowed-type? x)
  (or (widget? x)
      (symbol? x)
      (string? x)
      (number? x)
      (procedure? x)
      (pair? x)))

(define (include filename)
  (with-input-from-file filename
    (lambda ()
      (let loop ((e (read)))
	(if (eof-object? e) '()
	  (let ((x (eval e (current-module))))
	    (if (allowed-type? x)
	      (cons x (loop (read)))
	      (loop (read)))))))))


(define (load-dialog dlg filename)
  (set-current-command-set! (make-command-set))
  (with-dialog dlg
    (lambda ()
      (parse-args (current-dialog)
		  (include filename)))))

(define-macro (declare-attribute x)
  (let ((name (gensym)))
    `(define (,x . ,name)
       (if (null? ,name) ',x
	 (list 'attr ',x (car ,name))))))

(define-macro (declare-keyword x)
  `(define ,x ',x))

(define-macro (declare-widget name create . attrs)
  (let ((names (map (lambda (x) (gensym)) attrs))
	(args (gensym)))
    `(define (,name ,@names . ,args)
       (parse-args
	 ,(let loop ((attrs attrs) (names names))
	    (if (null? attrs) `(,create (current-dialog))
		`(widget-set-attr ,(loop (cdr attrs) (cdr names))
				  (convert-name ,(symbol->string (car attrs)))
                                  (convert-attr-value ,(car names)))))
	 ,args))))

(define-macro (declare-event name)
  (let ((cmds (gensym)))
    `(define (,name . ,cmds)
       (list 'event
	     ,(symbol->string name)
	     ,(let ((cs (gensym))
		    (n (gensym))
		    (loop (gensym)))
		`(let* ((,cs (current-command-set))
			(,n (,cs 'new)))
		   (let ,loop ((,cmds ,cmds))
		     (if (null? ,cmds) ,n
		       (begin
			 (cond
			   ((procedure? (car ,cmds))  (,cs 'add ,n (car ,cmds)))
			   ((pair? (caar ,cmds)) (,loop (car ,cmds)))
			   (else (,cs 'add ,n (car ,cmds))))
			 (,loop (cdr ,cmds)))))))))))


(declare-attribute text)
(declare-attribute text-append)
(declare-attribute editable)
(declare-attribute mask)
(declare-attribute width)
(declare-attribute height)
(declare-attribute default)
(declare-attribute flat)
(declare-attribute font)
(declare-attribute align)
(declare-attribute margin)
(declare-attribute spacing)
(declare-attribute caption)
(declare-attribute echo)
(declare-attribute pixmap)
(declare-attribute backgroundpixmap)
(declare-attribute sizepolicy)
(declare-attribute current)
(declare-attribute readonly)
(declare-attribute enabled)
(declare-attribute checked)
(declare-attribute total)
(declare-attribute value)
(declare-attribute hidden)

(declare-keyword yes)
(declare-keyword no)
(declare-keyword center)
(declare-keyword left)
(declare-keyword right)
(declare-keyword stars)

(declare-widget button dlg-make-button text)
(declare-widget label dlg-make-label text)
(declare-widget edit dlg-make-lineedit text)
(declare-widget vbox dlg-make-vbox)
(declare-widget hbox dlg-make-hbox)
(declare-widget listbox dlg-make-listbox)
(declare-widget combobox dlg-make-combobox)
(declare-widget checkbox dlg-make-checkbox text)
(declare-widget progressbar dlg-make-progressbar)
(declare-widget textbox dlg-make-textbox text)

(declare-event on-click)
(declare-event on-double-click)
(declare-event on-change)
(declare-event on-return)
(declare-event on-select)
(declare-event on-error)
(declare-event on-dbus)


(define (convert-name name)
  (cond ((string? name) name)
	((symbol? name) (symbol->string name))
	((number? name) (number->string name))
	(else (error "invalid name for attribute:" name))))


(define (convert-attr-value value)
  (format #f "~A" value))


(define (convert-event-value value)
  (if (string? value)
      (string->number value)
      value))


(define (parse-pair-arg widget arg)
  (case (car arg)
    ((attr)
     (widget-set-attr widget
		      (convert-name (cadr arg))
		      (convert-attr-value (caddr arg))))
    ((event)
     (widget-set-event widget
		       (convert-name (cadr arg))
		       (convert-event-value (caddr arg))))
    (else
      (parse-args widget arg))))


(define (parse-args widget args)
  (let loop ((args args))
    (if (null? args) widget
	(let ((arg (car args)))
	  (loop (cond ((null? arg)
		       (cdr args))
		      ((widget? arg)
		       (widget-insert widget arg)
		       (cdr args))

		      ((pair? arg)
		       (parse-pair-arg widget arg)
		       (cdr args))

		      ((and (or (symbol? arg) (string? arg))
			    (pair? (cdr args)))
		       (widget-set-attr widget
					(convert-name arg)
					(convert-attr-value (cadr args)))
		       (cddr args))

		      ((and (procedure? arg) (pair? (cdr args)))
		       (parse-pair-arg widget (arg (cadr args)))
		       (cddr args))

		      (else
			(error "invalid attribute specification:" args))))))))


(define (id name widget)
  (dlg-name-it (current-dialog) (convert-name name) widget))


(define (vector-append v1 v2)
  (apply vector (append (vector->list v1) (vector->list v2))))


(define (make-command-set)
  (let ((n 1)
	(cmds (make-vector 10 '())))
    (lambda (cmd . args)
      (case cmd
	((new) (let ((r n))
		 (set! n (+ 1 n))
		 (if (<= (vector-length cmds) n)
		     (set! cmds (vector-append cmds (make-vector 10 '()))))
		 r))
	((add) (let ((num (car args))
		     (cmd (cdr args)))
		 (if (>= num n) (error "invalid command index" n)
		     (if (not (null? cmd))
			 (vector-set! cmds num (append (vector-ref cmds num)
						       cmd))))
		 num))
	((get) (let ((num (car args)))
		 (vector-ref cmds num)))
	(else (error "invalid command"))))))

(define (new-set cs) (cs 'new))
(define (add-to-set! cs num cmd) (cs 'add num cmd))
(define (get-from-set cs num) (cs 'get num))

(define (current-command-set-get num dlg)
  (with-dialog dlg
    (lambda ()
      (process-cmd-set
	   (if (negative? num) '()
	       (get-from-set (current-command-set) num))
	   '()))))

(define (always-list cmd)
  (if (and (pair? cmd) (pair? (car cmd)))
    cmd
    (list cmd)))

(define (process-cmd-set cmds result)
  (if (null? cmds)
    result
      (process-cmd-set (cdr cmds)
		       (append result
				 (if (procedure? (car cmds))
				   (process-cmd-set (always-list ((car cmds))) '())
				   (list (eval-command (car cmds))))))))

(define (eval-command cmd)
  (if (null? cmd) '()
      (let ((first (car cmd))
            (rest (cdr cmd)))
        (if (procedure? first) (set! first (first)))
        (if (pair? first)
            (append (eval-command first) (eval-command rest))
            (cons first (eval-command rest))))))

(define look-quit '(/ctrl/look action quit))
(define (look-attr ctrlid field value)
  `(/ctrl/look action attr
	       ctrlid ,ctrlid
	       field ,(if (procedure? field) (field) field)
	       value ,value))

(define (look-view viewid . args)
  `(/ctrl/look action view
	       id ,viewid
	       ,@args))

(define (look-attr-get ctrlid field)
  (widget-get-attr
    (dlg-who-is-it (current-dialog) (convert-name ctrlid))
    (convert-name (if (procedure? field) (field) field))))

(define (look-send ctrlid event)
  `(/ctrl/look action send
	       ctrlid ,ctrlid
	       event ,event))

(define (woo-write name . args) `(,name action write ,@args))
(define (woo-new name . args) `(,name action new ,@args))
(define (woo-delete name . args) `(,name action delete ,@args))

(define (woo-cache-common type possible . args)
  (if (and (not (null? args))
	   (= (length args) 2)
	   (member (convert-name (car args)) possible))
    `(/ctrl/cache action ,type 
		  ,(convert-name (car args)) 
		  ,(convert-name (cadr args)))
    `(/ctrl/cache action ,type)))

(define (woo-cache-reset . args)
  (apply woo-cache-common "reset" '("id") args))

(define (woo-cache-unignore . args)
  (apply woo-cache-common "unignore" '("id") args))

(define (woo-cache-ignore . args)
  (apply woo-cache-common "ignore" '("id" "count") args))

(define (global name)
  (cond-cdr  (assoc (convert-name name) *globals*)))

(define (set-global! name value)
  (let ((res (assoc (convert-name name) *globals*)))
    (if res (set-cdr! res (convert-name value))
      (set! *globals*
	(append *globals* (list (cons (convert-name name)
				      (convert-name value))))))))
