/*
 *  ALTerator - ALT Linux configuration project
 *
 *  Copyright (c) 2003-2005 ALT Linux Ltd.
 *  Copyright (c) 2003-2005 Alexey Voinov
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 *  USA.
 */
#include <utils/os.hh>
#include <grp.h>
#include <pwd.h>

pid_t
alt::fork(void)
{
	pid_t child = ::fork();
	if(child == -1) throw err::sys();
	return child;
}

void
alt::setuid(uid_t uid)
{
	if(::setuid(uid) < 0) throw err::sys();
}

void
alt::setgid(gid_t gid)
{
	if(::setgroups(0, NULL) < 0) throw err::sys();
	if(::setgid(gid) < 0) throw err::sys();
}

uid_t
alt::uid_by_name(const std::string& name)
{
	struct passwd *pw = ::getpwnam(name.c_str());
	if(!pw) throw err::sys();
	return pw->pw_uid;
}

gid_t
alt::gid_by_name(const std::string& name)
{
	struct group *gr = ::getgrnam(name.c_str());
	if(!gr) throw err::sys();
	return gr->gr_gid;
}

void
alt::pipe(int& in, int& out)
{
	int fds[2];
	if(::pipe(fds) == -1) throw err::sys();
	in=fds[0], out=fds[1];
}

void
alt::dup2(int old_fd, int new_fd)
{
	if(::dup2(old_fd, new_fd) == -1) err::sys();
}

void
alt::kill(pid_t pid, int sig)
{
	if(::kill(pid, sig) == -1) throw err::sys();
}

void
alt::setpgid(pid_t pid, pid_t pgid)
{
	if(::setpgid(pid, pgid) == -1) throw err::sys();
}

pid_t
alt::getpgid(pid_t pid)
{
	pid_t pgid = ::getpgid(pid);
	if(pgid == -1) throw err::sys();
	return pgid;
}
