/*
 *  ALTerator - ALT Linux configuration project
 *
 *  Copyright (c) 2004,2005 ALT Linux Ltd.
 *  Copyright (c) 2004,2005 Alexey Voinov
 *  Copyright (c) 2004,2005 Stanislav Ievlev
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 *  USA.
 */
#include <bus/bus.hh>
#include <bus/proto.hh>
#include <bus/console.hh>

namespace
{
	struct proto
	{
		typedef void (proto::*state_t)(void);
		state_t state;
		bool dirdown;
		woobus::module *current, *neigbour;
		std::string msgbuffer;

		proto(void): state(&proto::wait_msg), current(0), neigbour(0) {}

		void wait_msg(void);
		void offer_msg(void);
		void wait_accept(void);
		void send_accept(void);
		void read_msg(void);
		void send_msg(void);
		void send_eom(void);

		void find_neighbour(woobus::module *m);
		void next(state_t);
	};
	static proto sm;

	void proto::wait_msg(void)
	{
		if(modules_begin() == modules_end())
			throw err::proto("no modules found");
		if(!current) current = &(*modules_begin());

		std::string cmd = current->read_line();
		if(cmd == "<<") dirdown = false;
		else if(cmd == ">>") dirdown = true;
		else if(cmd == "xx") throw err::finish();
		else throw err::proto("invalid address");
		find_neighbour(current);
	}

	void proto::offer_msg(void)
	{
		if(!neigbour)
			throw err::proto("neigbour lost");
		neigbour->send_line(dirdown ? ">>" : "<<");
		next(&proto::wait_accept);
	}

	void proto::wait_accept(void)
	{
		if(!neigbour)
			throw err::proto("neigbour lost");
		std::string cmd = neigbour->read_line();
		if(cmd == "<>") find_neighbour(neigbour);
		else if(cmd == "vv") next(&proto::send_accept);
		else throw err::proto("invalid accept");
	}

	void proto::send_accept(void)
	{
		if(!current)
			throw err::proto("current lost");
		current->send_line("vv");
		msgbuffer.clear();
		next(&proto::read_msg);
	}

	void proto::read_msg(void)
	{
		if(!current)
			throw err::proto("current lost");
		msgbuffer = current->read_line();
		if(msgbuffer == "^^") next(&proto::send_eom);
		else next(&proto::send_msg);
	}

	void proto::send_msg(void)
	{
		if(!neigbour)
			throw err::proto("neigbour lost");
		neigbour->send_line(msgbuffer);
		msgbuffer.clear();
		next(&proto::read_msg);
	}

	void proto::send_eom(void)
	{
		if(!neigbour)
			throw err::proto("neigbour lost");
		neigbour->send_line("^^");
		current = neigbour;
		neigbour = 0;
		next(&proto::wait_msg);
	}

	void proto::find_neighbour(woobus::module *m)
	{
		bus_iterator i(m);
		if(dirdown) neigbour = &(*(++i));
		else if(i != modules_begin()) neigbour = &(*(--i));
		else neigbour = 0;

		if(!neigbour) throw err::proto("noone receiving");

		next(&proto::offer_msg);
	}

	void proto::next(proto::state_t s)
	{
		state = s;
	}
}

void step(void)
{
	(sm.*(sm.state))();
}

void run(void)
{
	while(true) step();
}

// FIXME: those functions can be implemented using some kind of map.
// Now information is triplicated, which is bad.
std::string get_state(void)
{
	if(sm.state == &proto::wait_msg) return "wait_msg";
	else if(sm.state == &proto::offer_msg) return "offer_msg";
	else if(sm.state == &proto::wait_accept) return "wait_accept";
	else if(sm.state == &proto::send_accept) return "send_accept";
	else if(sm.state == &proto::read_msg) return "read_msg";
	else if(sm.state == &proto::send_msg) return "send_msg";
	else if(sm.state == &proto::send_eom) return "send_eom";
	else throw err::proto("bug in code");
}

void set_state(const std::string& nstate)
{
	if(nstate == "wait_msg" || nstate == "0")
		sm.next(&proto::wait_msg);
	else if(nstate == "offer_msg" || nstate == "1")
		sm.next(&proto::offer_msg);
	else if(nstate == "wait_accept" || nstate == "2")
		sm.next(&proto::wait_accept);
	else if(nstate == "send_accept" || nstate == "3")
		sm.next(&proto::send_accept);
	else if(nstate == "read_msg" || nstate == "4")
		sm.next(&proto::read_msg);
	else if(nstate == "send_msg" || nstate == "5")
		sm.next(&proto::send_msg);
	else if(nstate == "send_eom" || nstate == "6")
		sm.next(&proto::send_eom);
	else throw err::proto("invalid state");
}

woobus::module *current_module(void)
{
	return sm.current;
}

void set_current_module(woobus::module *m)
{
	sm.current = m;
}

woobus::module *neigbour_module(void)
{
	return sm.neigbour;
}

void set_neighbour_module(woobus::module *m)
{
	sm.neigbour = m;
}
