/*
 *  ALTerator - ALT Linux configuration project
 *
 *  Copyright (c) 2004,2005 ALT Linux Ltd.
 *  Copyright (c) 2004,2005 Alexey Voinov
 *  Copyright (c) 2004,2005 Stanislav Ievlev
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 *  USA.
 */
#include <utils/err.hh>

#include <iostream>
#include <sstream>
#include <list>

#include <utils/str.hh>
#include <command_io.hh>
#include <bus_io.hh>

#include <chooser.hh>

struct same_module
{
	same_module(const module& m): m_(m) {}
	bool operator ()(const module& m) const
	{
		return m_.command() == m.command();
	}
private:
	const module& m_;
};

struct select_cmd
{
	select_cmd(const command& cmd, std::list<module>& handlers)
		: cmd_(cmd), handlers_(handlers) {}

	void operator()(const commander_t& cmd)
	{
		context dummy;
		if(cmd.first.matches(cmd_, dummy)
			&& std::find_if(handlers_.begin(), handlers_.end(),
				same_module(cmd.second)) == handlers_.end())
		{
			handlers_.push_back(cmd.second);
		}
	}

private:
	const command& cmd_;
	std::list<module>& handlers_;
};

struct cmd_convert
{
	cmd_convert(const command& cmd, std::ostream& os): cmd_(cmd), os_(os) {}
	void operator()(module& m)
	{
		m.start("convert");
		m.os() << cmd_;

		command cmd;
		while(m.is() >> cmd) os_ << cmd;
		os_ << command("");
		m.stop();
	}
private:
	const command& cmd_;
	std::ostream& os_;
};


struct cmd_unconvert
{
	cmd_unconvert(const command& first_cmd,const std::string& input,std::ostream& os):
		first_cmd_(first_cmd),
		input_(input),
		os_(os)
	{}
	
	void operator()(module& m)
	{
		m.start("unconvert");
		m.os()<<first_cmd_;

		command input_cmd;
		std::istringstream is(input_);
		while (is>>input_cmd)
		{
			m.os()<<input_cmd;

			command cmd;
			while (m.is() >> cmd)
			{
				if (cmd.name() == "/rcommander/stop") break;
				os_ << cmd;
			}
		}
		m.close_os();
		m.stop();
	}
	
private:
	const command&		first_cmd_;
	std::string		input_;
	std::ostream&		os_;
};

void convert(commanders_db& cmds, const command& cmd,std::ostream& os)
{
	std::list<module> handlers;
	std::for_each(cmds.begin(), cmds.end(), select_cmd(cmd, handlers));
	if(handlers.empty())
	{
		os << cmd; //trivial by default
		os << command("");//are we really need it?
//		os << command("/error/convert")("reason", "no handlers");
	}
	else
		std::for_each(handlers.begin(), handlers.end(),
				cmd_convert(cmd, os));
}

void split_errors(std::istream& is,std::ostream& os,std::ostream& err)
{
	command cmd;
	while (is>>cmd)
		((alt::starts_with(cmd.name(),"/error/"))?err:os)<<cmd;
}


struct try_join
{
	try_join(command& cmd): cmd_(cmd),res_(true) {}

	void operator()(const std::string& name,const std::string& value)
	{
		if (!res_) return;
		try
		{
			if (cmd_(name) != value)
			{//we collect reason fields for "errors" and reject join on other commands
				if (alt::starts_with(cmd_.name(),"/error"))
					cmd_(name,cmd_(name)+", "+value);
				else
					res_ = false;
			}
		}
		catch(std::runtime_error&)
		{
			cmd_(name,value);
		}
	}
	
	operator bool() const { return res_; }
	
	command& cmd_;
	bool res_;
};


bool join_fields(command& old_cmd,const command& new_cmd)
{
	return new_cmd.for_each(try_join(old_cmd));
}

typedef std::map<std::string,command> cmd_map;

struct as_prefix
{
	as_prefix(const std::string& prefix): prefix_(prefix) {}
	bool operator()(const cmd_map::value_type& i)
	{
		return alt::starts_with(i.first,prefix_);
	}
	const std::string& prefix_;
};


bool really_error(const command& cmd,const cmd_map& cmds)
{
	return std::find_if(cmds.begin(),
			    cmds.end(),
			    as_prefix(cmd.name().substr(strlen("/error?"),std::string::npos))
			    ) == cmds.end();
}


struct print_cmd
{
	print_cmd(std::ostream& os,const cmd_map& cmds):
		os_(os),
		cmds_(cmds)
	{}
	void operator()(const command& cmd)
	{
		if (alt::starts_with(cmd.name(),"/error?"))
		{//drop conditional errors or convert it to normal error
			if (really_error(cmd,cmds_))
			{
				command new_cmd = cmd;
				new_cmd.name("/error"+
				     	     cmd.name().substr(strlen("/error?"),std::string::npos));
				os_<<new_cmd;
			}
		}
		else
			os_<<cmd;
	}
	std::ostream& os_;
	const cmd_map& cmds_;
};

void join_cmds(std::istream& is,std::ostream& os)
{
	//first step: simple mapping
	cmd_map cmds;

	command cmd;
	while(is>>cmd)
	{
		if (cmds.find(cmd.name()) == cmds.end())
			cmds[cmd.name()]=cmd;//first filling of the map
		else
		{//try to join
			if (!join_fields(cmds[cmd.name()],cmd))
			{
				os<<command("/error/unconvert")("action","join")("reason","field conflict");
				return;
			}
		}
	}

	//print commands and process "/error?" messages
	std::for_each(cmds.begin(),cmds.end(),second(print_cmd(os,cmds)));
}

void convert_for_awhile(commanders_db& cmds,
                        commanders_db& rcmds,
			std::istream& is, std::ostream& os)
{
	command cmd;
	while(is)
	{		
		std::string direction;
		getline(is, direction);
		if(">>" == direction)
		{
			if (!cmd.name().empty())
				os<<"<>"<<std::endl;
			else
			{//convert
				//read all available commands and convert it then
				std::istringstream ist(receive(is,os,std::cerr));
			
				ist >> cmd;
				if (alt::starts_with(cmd.name(),"/ctrl/"))
				{//ignore control commands
					send("<<",std::string(),is,os,std::cerr);
					cmd = command();
					continue;
				}
				
				
				std::ostringstream cvt_ost;
				convert(cmds, cmd, cvt_ost);

				//if we find errors - send up, otherwise - down
				std::istringstream cvt_ist(cvt_ost.str());
				std::ostringstream good_ost,err_ost;
				split_errors(cvt_ist,good_ost,err_ost);
				if (!err_ost.str().empty())
				{
					send("<<",err_ost.str(),is,os,std::cerr);
					cmd = command();
				}
				else
					send(">>",good_ost.str(),is,os,std::cerr);
			}
		}
		else if("<<" == direction)
		{
			if (cmd.name().empty())
				os << "<>" << std::endl;
			else
			{//unconvert
				//first of all try to find any handler
				std::list<module> handlers;
				std::for_each(rcmds.begin(), rcmds.end(), select_cmd(cmd, handlers));
				if(handlers.empty())
					os << "<>" << std::endl; //skip input if we have no hanlers
				else
				{
					std::string	input(receive(is,os,std::cerr));
					std::ostringstream cvt_ost;
				
					std::for_each(handlers.begin(), handlers.end(),
					cmd_unconvert(cmd,input,cvt_ost));
				
					std::istringstream cvt2_ist(cvt_ost.str());
					std::ostringstream cvt2_ost;
					join_cmds(cvt2_ist,cvt2_ost);

					std::istringstream cvt_ist(cvt2_ost.str());
					std::ostringstream good_ost,err_ost;
					split_errors(cvt_ist,good_ost,err_ost);

					//minimum one command to out
					//if (good_ost.str().empty()) good_ost<<command();

					send("<<",
				     	(err_ost.str().empty())?good_ost.str():err_ost.str(),
				     	is,os,std::cerr);
				}
				cmd = command();
			}
		}
		else
			std::cerr << "invalid address line" << std::endl;
	}
}
