/*
 *  ALTerator - ALT Linux configuration project
 *
 *  Copyright (c) 2004,2005 ALT Linux Ltd.
 *  Copyright (c) 2004,2005 Alexey Voinov
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 *  USA.
 */
#ifndef CONTEXT_HH
#define CONTEXT_HH

#include <string>
#include <map>

#include <pair_ops.hh>

class context
{
public:
	typedef std::map<std::string, std::string> params_t;
	typedef std::string (*policy_t)(const std::string&, const std::string&);

	struct missing {};

private:
	params_t params_;
	policy_t default_;

public:
	explicit context(policy_t policy = context::replace)
		: default_(policy) {}

	context(const context& ctx, policy_t policy = context::replace)
		: default_(policy) { add(ctx); }

	inline context&
	operator()(const std::string&, const std::string&);

	inline const std::string&
	operator()(const std::string&) const;

	inline context&
	add(const std::string&, const std::string&);

	context&
	add(const std::string&, const std::string&, policy_t);

	inline context&
	add(const context&);

	context&
	add(const context&, policy_t);

	const std::string&
	get(const std::string&) const;

	bool
	exists(const std::string&) const;

	context&
	remove(const std::string&);

	context&
	remove(const context&);

	inline context&
	set(policy_t);

	inline size_t
	size(void) const;


	inline static std::string
	replace(const std::string&, const std::string&);

	inline static std::string
	keepold(const std::string&, const std::string&);

	inline static std::string
	merge(const std::string&, const std::string&);

	template <typename A> inline A for_each(A a);
	template <typename A> inline A for_each_pair(A a);
	template <typename A> inline A for_each_key(A a);
	template <typename A> inline A for_each_value(A a);
	template <typename A> inline A for_each(A a) const;
	template <typename A> inline A for_each_pair(A a) const;
	template <typename A> inline A for_each_key(A a) const;
	template <typename A> inline A for_each_value(A a) const;
};

context&
context::operator()(const std::string& key, const std::string& value)
{
	return add(key, value);
}

const std::string&
context::operator()(const std::string& key) const
{
	return get(key);
}

inline context&
operator+=(context& ctx, const context& newctx)
{
	return ctx.add(newctx);
}

inline context&
operator-=(context& ctx, const std::string& key)
{
	return ctx.remove(key);
}

inline context&
operator-=(context& ctx, const context& otherctx)
{
	return ctx.remove(otherctx);
}

inline context&
operator|=(context& ctx, context::policy_t policy)
{
	return ctx.set(policy);
}

context&
context::add(const std::string& key, const std::string& value)
{
	return add(key, value, default_);
}

context&
context::add(const context& ctx)
{
	return add(ctx, default_);
}


context&
context::set(policy_t policy)
{
	return (default_ = policy), *this;
}

size_t
context::size(void) const
{
	return params_.size();
}

std::string
context::replace(const std::string&, const std::string& newval)
{
	return newval;
}

std::string
context::keepold(const std::string& oldval, const std::string&)
{
	return oldval;
}

std::string
context::merge(const std::string& oldval, const std::string& newval)
{
	return oldval + newval;
}

template <typename A> A context::for_each_pair(A a)
{
	return std::for_each(params_.begin(), params_.end(), a);
}

template <typename A> A context::for_each(A a)
{
	return for_each_pair(unpair<A>(a));
}

template <typename A> A context::for_each_key(A a)
{
	return for_each_pair(first<A>(a));
}

template <typename A> A context::for_each_value(A a)
{
	return for_each_pair(second<A>(a));
}

template <typename A> A context::for_each_pair(A a) const
{
	return std::for_each(params_.begin(), params_.end(), a);
}

template <typename A> A context::for_each(A a) const
{
	return for_each_pair(unpair<A>(a));
}

template <typename A> A context::for_each_key(A a) const
{
	return for_each_pair(first<A>(a));
}

template <typename A> A context::for_each_value(A a) const
{
	return for_each_pair(second<A>(a));
}

#endif //CONTEXT_HH
