;description: functions to work with internal alterator commands

;each command is a list: (command-name arg1 value1 arg2 value2)
;command name is a symbol, each argument name is symbol, each value is string
;commands with /error/ prefixes are error messages, and may contain single reason field
;with textual description of problem

(define-module (alterator command)
	       :use-module (alterator str)
	       :use-module (alterator algo)
	       :export (command?
			 error-command?
			 command-names=?
			 command-name
			 command-for-each
			 command-find-if
			 command-fold
			 command-arg-ref
			 command-arg-add
			 join-error-commands
			 join-commands))

;simple check that object is valid command: this is a list and number of elements is odd
(define (command? obj)
  (and (list? obj) (odd? (length obj))))

;name of the command
(define (command-name obj) (sure-symbol (car obj)))

;function for processing all command's options
;calls procedure with to args option's name and option's value
(define (command-for-each proc obj)
  (and (pair? obj)
       (let loop ((options (cdr obj)))
	 (if (not (null? options))
	   (begin (proc (car options) (cadr options))
		  (loop (cddr options)))))))

;find_if in command args
(define (command-find-if pred cmd)
  (call-with-current-continuation
    (lambda (exit)
      (command-for-each
	(lambda (name value)
	  (define result (pred name value))
	  (and result (exit result)))
	cmd)
      #f)))

;fold command args
(define (command-fold proc initial cmd)
  (let loop ((current (cdr cmd))
	     (initial initial))
    (if (null? current)
    	initial
	(loop (cddr current) (proc (car current) (cadr current) initial)))))

(define (error-command? cmd)
  (and (command? cmd)
       (or (string-starts-with? (sure-string (command-name cmd)) "/error/")
	   (string-starts-with? (sure-string (command-name cmd)) "/error?/"))))

;query for some option
(define (command-arg-ref cmd argname)
  (and (command? cmd)
       (command-find-if (lambda (name value)
			  (and (eq? name argname)
			       (cons name value)))
			cmd)))

(define (command-arg-add cmd name value)
  (append cmd (list name value)))

(define (command-names=? cmd1 cmd2)
  (eq? (command-name cmd1)
       (command-name cmd2)))

;simple joining of reason fields
(define (join-error-commands cmd1 cmd2)
  (define (join-reasons txt1 txt2)
    (if (or (empty-string? txt1)
	    (empty-string? txt2))
      (string-append txt1 txt2)
      (string-append txt1 "," txt2)))
  (define (get-reason cmd)
    (or (cond-cdr (command-arg-ref cmd 'reason)) ""))
  (or (and (error-command? cmd1)
	   (error-command? cmd2)
	   (command-names=? cmd1 cmd2)
	   `(,(command-name cmd1)
	      reason ,(join-reasons (get-reason cmd1) (get-reason cmd2))))
      (error "invalid commands for join" cmd1 cmd2)))

;join two commands into one
(define (join-commands cmd1 cmd2)
  (define (join-fields name value initial)
    (let ((exist-value (cond-cdr (command-arg-ref cmd1 name))))
      (if exist-value
	(if (string=? exist-value value)
	  initial
	  (error "join field conflict" cmd1 name))
	(command-arg-add initial name value))))
  (or (and (command? cmd1)
	   (command? cmd2)
	   (command-names=? cmd1 cmd2)
	   (command-fold join-fields
			 cmd1
			 cmd2))
      (error "invalid commands for join" cmd1 cmd2)))

