;description: woobus module to operate with backends

(define-module (alterator ensign)
	       :use-module (srfi srfi-13)
	       :use-module (srfi srfi-1)
	       :use-module (alterator str)
	       :use-module (alterator pipe)
	       :use-module (alterator algo)
	       :use-module (alterator command)
	       :use-module (alterator sandbox)
	       :use-module (alterator ensign barnacle)

	       :export (ensign
	       		
			 error-answer?
	       		 prepare-cmds
			 call-mariner))

(define *mariners* '()) ;associative list of loaded backends

;filter: split command name, convert args to associative list
(define (prepare-cmds cmds)
  (let loop ((result '())
	     (cmds cmds))
    (cond
      ((null? cmds) result)
      ((null? (car cmds)) (loop result (cdr cmds))) ;ignore null commands
      ((not (pair? (car cmds))) (error "invalid command in list"))
      ((eq? (caar cmds) '#{}#) (loop result (cdr cmds))); also ignore empty commands
      (else
	(let ((current (car cmds)))
	  (loop (append result 
			(list (cons (string-splitting (sure-string (car current)) #\/)
				    (command-fold acons '() current))))
		(cdr cmds)))))))

;default backend for all commands: always answers - "backend not found"
(define (ghost-mariner object action)
  (lambda args  '(error "backend not found")))

(define (null-mariner object action)
  (lambda args #f))

(use-modules (ice-9 pretty-print))

(define *ensign-known-macros* '(dynamic-call use-modules))
(define (load-midshipman filename)
  (define (known-macros? item)
    (and (pair? item)
        (member (car item) *ensign-known-macros*))) 
  (let* ((macros (list-extract (read-file filename) known-macros?))
        (defs (sandbox-extract-definitions (cdr macros))))
    (eval
      `(begin
        ,@(car macros)
        ,(sandbox-transform-definitions (car defs)
                                        (cdr defs)))
      (current-module))))


;try to load an appropriate backend
;(define (load-mariner sym continue)
(define (load-mariner sym backenddir)
  (let ((old-backend (string-append backenddir "/backend/" (sure-string sym)))
	(new-backend (string-append backenddir "/backend2/" (sure-string sym))))
    (cond
      ((eq? sym 'ctrl) null-mariner)
      ((access? new-backend R_OK) (or (load-midshipman new-backend)
				      ghost-mariner))
      ((access? old-backend R_OK) (make-barnacle old-backend))
      (else ghost-mariner))))

(define (remember-mariner sym backend)
  (and backend
       (set! *mariners* (acons sym backend *mariners*)))
  backend)

(define (resolv-name sym backenddir)
  (or (cond-cdr (assq sym *mariners*))
      (remember-mariner sym (load-mariner sym backenddir))))

;load low-level backend
(define (call-mariner cmd backenddir)
    (let ((name (car cmd))
	(args (cdr cmd)))
    (if (not (null? name))
      (let ((mariner (resolv-name (sure-symbol (car name)) backenddir))
	    (action  (assq 'action args)))
	(if action
	  ((mariner (cdr name) (sure-symbol (cdr action))) args)
	  '(error "action field not found" cmd))))))

(define (error-answer? cmd) (and (pair? cmd) (eq? (car cmd) 'error)))

; new-version: process commands from list and autoconvert results
(define (process-commandlist cmd prev backenddir)
  (define (out-error name reason)
    `(,(string-append "/error" name) reason ,reason))
  (define (out-command name options)
    `(,name ,@options))
  (let ((name (string-append "/" (string-join (car cmd) "/")))
        (answer (call-mariner cmd backenddir)))
    (if answer
      (append prev
              (cond
                ((null? answer)
                 '()); empty answer
                ((error-answer? answer)
                 (list (out-error name (cadr answer))))
                ((and (pair? answer) (pair? (car answer)))
                 (map (lambda (x)
                        (out-command (string-append name "/" (car x)) (cdr x)))
                      answer))
                ((pair? answer)
                 (list (out-command name answer)))
                (else
                  (list (out-error name "unsupported answer type")))))
      prev)))

;main entry point
(define (ensign-main cmd-pair next backenddir)
  (next
    (cons (car cmd-pair)
	  (fold
	    (lambda (x y) (process-commandlist x y backenddir))
	    '()
	    (prepare-cmds (cdr cmd-pair))))))

(define (ensign . args)
  (let ((backenddir (if (null? args) "/usr/lib/alterator/" (car args))))
    (lambda (cmd-pair next)
      (ensign-main cmd-pair next backenddir))))

