;description: working with pipes and subprocesses

(define-module (alterator pipe)
	       :export (create-process
			 stop-process
			 with-input-from-pipe
			 with-ignored-sigpipe))

;extended version of ice-9 popen
(define (read-only? x) (eq? x 'read-only))
(define (write-only? x) (eq? x 'write-only))
(define (read-write? x) (eq? x 'read-write))

;create pipe and made nonbuffered write side
(define (create-pipe)
  (let ((p (pipe)))
    (setvbuf (cdr p) _IONBF)
    p))

;return (pid,read-port,write-port)
;TODO: port other code from ice-9 popen
; e.g. guardian's, hooks, etc.
(define (create-process mode prog . args)
  (let ((read-pipe (and (or (read-only? mode)
			    (read-write? mode))
			(create-pipe)))
	(write-pipe (and (or (write-only? mode)
			     (read-write? mode))
			 (create-pipe))))
    (let ((pid (primitive-fork)))
      (cond 
	((zero? pid) ;child work
	 (if (or (read-only? mode)
		 (read-write? mode))
	   (begin (close-port (car read-pipe))
		  (dup2 (fileno (cdr read-pipe)) 1)))
	 (if (or (write-only? mode)
		 (read-write? mode))
	   (begin (close-port (cdr write-pipe))
		  (dup2 (fileno (car write-pipe)) 0)))
	 (apply execlp prog prog args))
	(else ;parent work
	  (list pid
		(if (or (read-only? mode)
			(read-write? mode))
		  (begin (close-port (cdr read-pipe))
			 (car read-pipe))
		  #f)
		(if (or (write-only? mode)
			(read-write? mode))
		  (begin (close-port (car write-pipe))
			 (cdr write-pipe))
		  #f)))))))

(define (stop-process action p)
  (and (cadr p) (not (port-closed? (cadr p))) (close-port (cadr p)))
  (and (caddr p) (not (port-closed? (caddr p))) (close-port (caddr p)))
  (catch #t
	 (lambda ()
	   (if (eq? action 'terminate) (kill (car p) SIGTERM))
	   (status:exit-val (cdr (waitpid (car p)))))
	 (lambda (key . args) 0)))

(define (with-input-from-pipe cmdline thunk)
  (let ((saved-iport #f)
	(process #f))
    (dynamic-wind
      (lambda ()
	(set! saved-iport (current-input-port))
	(set! process (apply create-process 'read-only (car cmdline) (cdr cmdline)))
	(set-current-input-port (cadr process)))
      thunk
      (lambda()
	(stop-process 'wait process)
	(set-current-input-port saved-iport)))))

(define (with-ignored-sigpipe thunk)
  (let ((saved-action #f))
    (dynamic-wind
      (lambda () (set! saved-action (sigaction SIGPIPE SIG_IGN)))
      thunk
      (lambda () (sigaction SIGPIPE (car saved-action))))))

