/*
 *  ALTerator - ALT Linux configuration project
 *
 *  Copyright (c) 2004,2005 ALT Linux Ltd.
 *  Copyright (c) 2004,2005 Alexey Voinov
 *  Copyright (c) 2004,2005 Stanislav Ievlev
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 *  USA.
 */
#include <iostream>
#include <sstream>
#include <qapplication.h>

#include <qtlook.hh>

#include <qpainter.h>

namespace
{
	long percent_to_width(long width)
	{
		return ( QApplication::desktop()->width() * width ) / 100;
	}
	
	long percent_to_height(long height)
	{
		return ( QApplication::desktop()->height() * height ) / 100;
	}

	long width_to_percent(long width)
	{
		return ( width  * 100 ) / QApplication::desktop()->width();
	}
	
	long height_to_percent(long height)
	{
		return ( height * 100 ) / QApplication::desktop()->height();
	}
	
	#define str2qstr(STR) QString::fromUtf8(STR)

	#define qstr2str(STR) STR.utf8().data()
	
	QWidget *p_widget(qt_widget *parent)
	{
		return parent?static_cast<qt_widget*>(parent)->get_widget():0;
	}

	void qwidget_init(QWidget *wnd,QWidget *parent)
	{
		wnd->setBackgroundOrigin(QWidget::WindowOrigin);
	}
}

void widget_corners_round(QWidget *widget)
{
	QPainter painter(widget);
	
	painter.drawLine(1,1,1,widget->height()-1);
	painter.drawLine(1,1,widget->width()-1,1);
	painter.drawLine(widget->width()-1,1,widget->width()-1,widget->height()-1);
	painter.drawLine(1,widget->height()-1,widget->width()-1,widget->height()-1);
}



#include <qtattrs.hh>

//
// ------------
//

qt_widget::qt_widget(SCM data): data_(data)
{}

qt_widget::~qt_widget()
{}

void qt_widget::call_event_holder(const std::string& name)
{
	scm_call_2(data_,symbol2scm("call"),symbol2scm(name.c_str()));
}

void qt_widget::set_attr(const std::string& name, SCM value)
{
	if(name == "width") attribute_width::set(get_widget(), value);
	else if(name == "hidden") attribute_hidden::set(get_widget(), value);
	else if(name == "height") attribute_height::set(get_widget(), value);
	else if(name == "enabled") attribute_enabled::set(get_widget(), value);
	else if(name == "focus") attribute_focus::set(get_widget(), value);
	else if (name == "sizepolicy") attribute_sizepolicy::set(get_widget(), value);
	else if(name == "widget-id") attribute_widget_id::set(get_widget(),value);
}

SCM qt_widget::get_attr(const std::string& name)
{
	if(name == "width") return attribute_width::get(get_widget());
	else if(name == "hidden") return attribute_hidden::get(get_widget());
	else if(name == "height") return attribute_height::get(get_widget());
	else if(name == "enabled") return attribute_enabled::get(get_widget());
	else if(name == "focus") return attribute_focus::get(get_widget());
	else if(name == "widget-id") return attribute_widget_id::get(get_widget());
	return SCM_UNSPECIFIED;
}

//
// ------------
//


qt_button::qt_button(SCM data,qt_widget *parent):
	qt_widget(data),
	wnd(new QPushButton(p_widget(parent)))
{
	qwidget_init(wnd,p_widget(parent));
	wnd->setSizePolicy(QSizePolicy::Fixed,QSizePolicy::Fixed);
	
	connect(wnd, SIGNAL(clicked()), SLOT(on_click()));
}

void qt_button::set_attr(const std::string& name, SCM value)
{
	if(name == "text") attribute_text::set(wnd.get(), value);
	else if(name == "default") attribute_default::set(wnd.get(), value);
	else if(name == "flat") attribute_flat::set(wnd.get(), value);
	else if(name == "font") attribute_font::set(wnd.get(), value);
	else if(name == "pixmap") attribute_pixmap::set(wnd.get(), value);
	else qt_widget::set_attr(name,value);
}

SCM qt_button::get_attr(const std::string& name)
{
	if(name == "text") return attribute_text::get(wnd.get());
	else if(name == "default") return attribute_default::get(wnd.get());
	else if(name == "flat") return attribute_flat::get(wnd.get());
	else if(name == "font") return attribute_font::get(wnd.get());
	else if(name == "pixmap") return attribute_pixmap::get(wnd.get());
	else return qt_widget::get_attr(name);
}

//
// ------------
//

qt_label::qt_label(SCM data,qt_widget *parent):
	qt_widget(data),
	wnd(new QLabel(p_widget(parent)))
{
	qwidget_init(wnd,p_widget(parent));
}

void qt_label::set_attr(const std::string& name, SCM value)
{
	if(name == "text") attribute_text::set(wnd.get(), value);
	else if(name == "font") attribute_font::set(wnd.get(), value);
	else if(name == "pixmap") attribute_pixmap::set(wnd.get(), value);
	else if(name == "align") attribute_align::set(wnd.get(), value);
	else qt_widget::set_attr(name,value);
}

SCM qt_label::get_attr(const std::string& name)
{
	if(name == "text") return attribute_text::get(wnd.get());
	if(name == "font") return attribute_font::get(wnd.get());
	else if(name == "align") return attribute_align::get(wnd.get());
	else if(name == "pixmap") return attribute_pixmap::get(wnd.get());
	else return qt_widget::get_attr(name);
}

//
// ------------
//

qt_checkbox::qt_checkbox(SCM data,qt_widget *parent):
	qt_widget(data),
	wnd(new QCheckBox(p_widget(parent)))
{
	qwidget_init(wnd,p_widget(parent));
	connect(wnd, SIGNAL(clicked()), SLOT(on_click()));
}

void qt_checkbox::set_attr(const std::string& name, SCM value)
{
	if(name == "text") attribute_text::set(wnd.get(), value);
	else if(name == "font") attribute_font::set(wnd.get(), value);
	else if(name == "pixmap") attribute_pixmap::set(wnd.get(), value);
	else if(name == "checked") attribute_checked::set(wnd.get(), value);
	else qt_widget::set_attr(name,value);
}

SCM qt_checkbox::get_attr(const std::string& name)
{
	if(name == "text") return attribute_text::get(wnd.get());
	if(name == "font") return attribute_font::get(wnd.get());
	else if(name == "pixmap") return attribute_pixmap::get(wnd.get());
	else if(name == "checked") return attribute_checked::get(wnd.get()); 
	else return qt_widget::get_attr(name);
}

//
// ------------
//

qt_groupbox::qt_groupbox(SCM data,qt_widget *parent):
	qt_widget(data),
	wnd(new QGroupBox(p_widget(parent)))
{
	qwidget_init(wnd,p_widget(parent));
	wnd->setColumnLayout(1,Qt::Vertical);
	connect(wnd, SIGNAL(toggled(bool)), SLOT(on_toggle(bool)));
}

void qt_groupbox::set_attr(const std::string& name, SCM value)
{
	if(name == "title") attribute_title::set(wnd.get(), value);
	else if(name == "flat") attribute_flat::set(wnd.get(), value);
	else if(name == "checkable") attribute_checkable::set(wnd.get(), value);
	else if(name == "checked") attribute_checked::set(wnd.get(), value);
	else qt_widget::set_attr(name,value);
}

SCM qt_groupbox::get_attr(const std::string& name)
{
	if(name == "title") return attribute_title::get(wnd.get());
	if(name == "flat") return attribute_flat::get(wnd.get());
	else if(name == "checkable") return attribute_checkable::get(wnd.get());
	else if(name == "checked") return attribute_checked::get(wnd.get()); 
	else return qt_widget::get_attr(name);
}

//
// ------------
//

qt_progressbar::qt_progressbar(SCM data,qt_widget *parent):
	qt_widget(data),
	wnd(new QProgressBar(p_widget(parent)))
{
	qwidget_init(wnd,p_widget(parent));
}

void qt_progressbar::set_attr(const std::string& name, SCM value)
{
	if(name == "total") {
		SCM_ASSERT(SCM_NUMBERP(value),value,SCM_ARG2,"qt_progressbar::set_attr");
		wnd->setTotalSteps(scm2num(value));
	}
	else if(name == "value") {
		SCM_ASSERT(SCM_NUMBERP(value),value,SCM_ARG2,"qt_progressbar::set_attr");
		wnd->setProgress(scm2num(value));
	}
	else if(name == "font") attribute_font::set(wnd.get(), value);
	else qt_widget::set_attr(name,value);
}

SCM qt_progressbar::get_attr(const std::string& name)
{
	if(name == "total") return num2scm(wnd->totalSteps());
	if(name == "value") return num2scm(wnd->progress());
	if(name == "font") return attribute_font::get(wnd.get());
	else return qt_widget::get_attr(name);
}

//
// ------------
//

qt_lineedit::qt_lineedit(SCM data,qt_widget *parent):
	qt_widget(data),
	wnd(new QLineEdit(p_widget(parent)))
{
	qwidget_init(wnd,p_widget(parent));
	connect(wnd, SIGNAL(textChanged(const QString&)),
			SLOT(on_change(const QString&)));
	connect(wnd, SIGNAL(returnPressed()), SLOT(on_return()));
}

void qt_lineedit::set_attr(const std::string& name, SCM value)
{
	if(name == "text") attribute_text::set(wnd.get(), value);
	else if(name == "align") attribute_align::set(wnd.get(), value);
	else if(name == "echo") attribute_echo::set(wnd.get(), value);
	else if(name == "readonly") attribute_readonly::set(wnd.get(), value);
	else if(name == "mask") attribute_mask::set(wnd.get(), value);
	else qt_widget::set_attr(name,value);
}

SCM qt_lineedit::get_attr(const std::string& name)
{
	if(name == "text") return attribute_text::get(wnd.get());
	else if(name == "align") return attribute_align::get(wnd.get());
	else if(name == "echo") return attribute_echo::get(wnd.get());
	else if(name == "readonly") return attribute_readonly::get(wnd.get());
	else if(name == "mask") return attribute_mask::get(wnd.get());
	else return qt_widget::get_attr(name);
}

//
// ------------
//

qt_textbox::qt_textbox(SCM data,qt_widget *parent):
	qt_widget(data),
	wnd(new QTextEdit(p_widget(parent)))
{
	qwidget_init(wnd,p_widget(parent));
	connect(wnd, SIGNAL(textChanged()),SLOT(on_change()));
	connect(wnd, SIGNAL(returnPressed()), SLOT(on_return()));
}

void qt_textbox::set_attr(const std::string& name,SCM  value)
{
	if(name == "text") attribute_text::set(wnd.get(), value);
	else if(name == "text-append") {
		SCM_ASSERT(SCM_STRINGP(value),value,SCM_ARG2,"qt_textbox::set_attr");
		wnd->append(str2qstr(scm2str(value)));
	}
	else if(name == "align") attribute_align::set(wnd.get(), value);
	else if(name == "readonly") attribute_readonly::set(wnd.get(), value);
	else qt_widget::set_attr(name,value);
}

SCM qt_textbox::get_attr(const std::string& name)
{
	if(name == "text") return attribute_text::get(wnd.get());
	else if(name == "align") return attribute_align::get(wnd.get());
	else if(name == "readonly") return attribute_readonly::get(wnd.get());
	else return qt_widget::get_attr(name);
}

//
// ------------
//

qt_listbox::qt_listbox(SCM data,qt_widget *parent):
	qt_widget(data),
	wnd(new QListBox(p_widget(parent)))
{
	qwidget_init(wnd,p_widget(parent));
	connect(wnd, SIGNAL(selectionChanged()), SLOT(delayed_on_select()));
	connect(wnd,
	        SIGNAL(doubleClicked(QListBoxItem*)),
		SLOT(delayed_on_double_click(QListBoxItem*)));
}

void qt_listbox::set_attr(const std::string& name,SCM value)
{
	if(name == "font") attribute_font::set(wnd.get(), value);
	else if(name == "current") attribute_current::set(wnd.get(), value);
	else if(name == "remove") {
				    SCM_ASSERT(SCM_SYMBOLP(value) || SCM_NUMBERP(value),
				    	       value,SCM_ARG2,"qt_listbox::set_attr"); 
				    if (SCM_SYMBOLP(value) && !strcmp(scm2symbol(value),"all"))
				    	wnd->clear();
				    else
				   	wnd->removeItem(scm2num(value));
				  }
	else qt_widget::set_attr(name,value);
}

void qt_listbox::set_extra(const std::string& cmd, const std::string& arg1,
						   const std::string& arg2,
						   SCM arg3)
{
	if (cmd == "append")
	{
		const std::string &text = arg1;
		const std::string &pixmap = arg2;
		if(pixmap.empty())
			wnd->insertItem(str2qstr(text.c_str()));
		else
			wnd->insertItem(QPixmap(pixmap),str2qstr(text.c_str()));

	}
	else if (cmd == "edit")
	{
		SCM_ASSERT(SCM_NUMBERP(arg3), arg3,SCM_ARG3,"qt_listbox::set_extra"); 

		const std::string& attr = arg1;
		const std::string& value = arg2;
		int num = scm2num(arg3);
		
		if (attr == "text")
		{
			if (wnd->pixmap(num))
				wnd->changeItem(*(wnd->pixmap(num)),str2qstr(value.c_str()),num);
			else
				wnd->changeItem(str2qstr(value.c_str()),num);
		}
		else if (attr == "pixmap")
			wnd->changeItem(QPixmap(value),wnd->text(num),num);
	}
}

SCM qt_listbox::get_attr(const std::string& name)
{
	if(name == "text") return attribute_curtext::get(wnd.get());
	else if(name == "count") return attribute_count::get(wnd.get());
	else if(name == "font") return attribute_font::get(wnd.get());
	else if(name == "current") return attribute_current::get(wnd.get());
	return qt_widget::get_attr(name);
}

SCM qt_listbox::get_extra(const std::string& name,SCM arg)
{
	if (name == "text") {
		SCM_ASSERT(SCM_NUMBERP(arg), arg,SCM_ARG2,"qt_listbox::get_extra"); 
		return str2scm(qstr2str(wnd->text(scm2num(arg))));
	}
	return SCM_UNSPECIFIED;
}

//
// ------------
//

qt_combobox::qt_combobox(SCM data,qt_widget *parent):
	qt_widget(data),
	wnd(new QComboBox(p_widget(parent)))
{
	qwidget_init(wnd,p_widget(parent));
	connect(wnd, SIGNAL(activated(const QString&)),
		     SLOT(delayed_on_change(const QString&)));
}

void qt_combobox::set_attr(const std::string& name,SCM value)
{
	if(name == "font") attribute_font::set(wnd.get(), value);
	else if(name == "current") attribute_current::set(wnd.get(), value);
	else if(name == "editable") attribute_editable::set(wnd.get(), value);
	else if(name == "remove") { 
				    SCM_ASSERT(SCM_SYMBOLP(value) || SCM_NUMBERP(value),
				    	       value,SCM_ARG2,"qt_combobox::set_attr"); 
				    if (SCM_SYMBOLP(value) && !strcmp(scm2symbol(value),"all"))
				    	wnd->clear();
				    else
				   	wnd->removeItem(scm2num(value));
				  }
	else qt_widget::set_attr(name,value);
}

void qt_combobox::set_extra(const std::string& cmd, const std::string& arg1,
						   const std::string& arg2,
						   SCM arg3)
{
	if (cmd == "append")
	{
		const std::string &text = arg1;
		const std::string &pixmap = arg2;
		if(pixmap.empty())
			wnd->insertItem(str2qstr(text.c_str()));
		else
			wnd->insertItem(QPixmap(pixmap),str2qstr(text.c_str()));

	}
	else if (cmd == "edit")
	{
		SCM_ASSERT(SCM_NUMBERP(arg3), arg3,SCM_ARG3,"qt_combobox::set_extra"); 

		const std::string& attr = arg1;
		const std::string& value = arg2;
		int num = scm2num(arg3);

		if (attr == "text")
		{
			if (wnd->pixmap(num))
				wnd->changeItem(*(wnd->pixmap(num)),str2qstr(value.c_str()),num);
			else
				wnd->changeItem(str2qstr(value.c_str()),num);
		}
		else if (attr == "pixmap")
			wnd->changeItem(QPixmap(value),wnd->text(num),num);
	}
}

SCM qt_combobox::get_attr(const std::string& name)
{
	if(name == "text") return attribute_curtext::get(wnd.get());
	else if(name == "count") attribute_count::get(wnd.get());
	else if(name == "font") return attribute_font::get(wnd.get());
	else if(name == "current") return attribute_current::get(wnd.get());
	else if(name == "editable") return attribute_editable::get(wnd.get());
	return qt_widget::get_attr(name);
}

SCM qt_combobox::get_extra(const std::string& name,SCM arg)
{
	if (name == "text") {
		SCM_ASSERT(SCM_NUMBERP(arg), arg,SCM_ARG2,"qt_combobox::get_extra");
		return str2scm(qstr2str(wnd->text(scm2num(arg))));
	}
	return SCM_UNSPECIFIED;
}

//
// ------------
//

qt_tabbox::qt_tabbox(SCM data,qt_widget *parent):
	qt_widget(data),
	wnd(new QTabWidget(p_widget(parent)))
{
	qwidget_init(wnd,p_widget(parent));
	connect(wnd, SIGNAL(currentChanged(QWidget *)), SLOT(on_change(QWidget*)));
}

void qt_tabbox::set_attr(const std::string& name,SCM value)
{
	if(name == "current"){
				SCM_ASSERT(SCM_SMOB_PREDICATE(widget_tag, value),
		           		   value, SCM_ARG2, "qt_tabbox::set_attr");
				QWidget* wid = ((qt_widget*)(SCM_SMOB_DATA(value)))->get_widget();
				wnd->showPage(wid);
			     }
	else
	qt_widget::set_attr(name,value);
}

void qt_tabbox::set_extra(const std::string& cmd, const std::string& arg1,
						   const std::string& arg2,
						   SCM arg3)
{
	if (cmd == "add-tab")
	{
		SCM_ASSERT(SCM_SMOB_PREDICATE(widget_tag, arg3),
		           arg3, SCM_ARG3, "qt_tabbox::set_extra");
		QWidget* wid = ((qt_widget*)(SCM_SMOB_DATA(arg3)))->get_widget();
		QString label = str2qstr(arg1.c_str());
		wnd->addTab(wid,label);
	}
}

SCM qt_tabbox::get_attr(const std::string& name)
{
	if (name == "current")
		return attribute_widget_id::get(wnd->currentPage());
	else
		return qt_widget::get_attr(name);
}

//
// ------------
//

qt_vbox::qt_vbox(SCM data,qt_widget *parent):
	qt_widget(data),
	wnd(new QLabel2(p_widget(parent)))
{
	qwidget_init(wnd,p_widget(parent));
	QBoxLayout *l = new QVBoxLayout2(wnd);
	l->setAutoAdd(TRUE);
}

void qt_vbox::set_attr(const std::string& name,SCM value)
{
	if (name == "margin") attribute_margin::set(wnd->layout(), value);
	else if(name == "clear-layout") ((QVBoxLayout2*)(wnd->layout()))->deleteAllItems();
	else if (name == "spacing") attribute_spacing::set(wnd->layout(), value);
	else if (name == "pixmap") attribute_pixmap::set(wnd.get(), value);
	else qt_widget::set_attr(name,value);
}

SCM qt_vbox::get_attr(const std::string& name)
{
	if(name == "margin") return attribute_margin::get(wnd->layout());
	else if(name == "spacing") return attribute_spacing::get(wnd->layout());
	return qt_widget::get_attr(name);
}

//
// ------------
//

qt_hbox::qt_hbox(SCM data,qt_widget *parent):
	qt_widget(data),
	wnd(new QLabel2(p_widget(parent)))
{
	qwidget_init(wnd,p_widget(parent));
	QBoxLayout *l = new QHBoxLayout2(wnd);
	l->setAutoAdd(TRUE);
}

void qt_hbox::set_attr(const std::string& name,SCM value)
{
	if(name == "margin") attribute_margin::set(wnd->layout(), value);
	else if(name == "clear-layout") ((QHBoxLayout2*)(wnd->layout()))->deleteAllItems();
	else if(name == "spacing") attribute_spacing::set(wnd->layout(), value);
	else if (name == "pixmap") attribute_pixmap::set(wnd.get(), value);
	else qt_widget::set_attr(name,value);
}

SCM qt_hbox::get_attr(const std::string& name)
{
	if(name == "margin") return attribute_margin::get(wnd->layout());
	else if(name == "spacing") return attribute_spacing::get(wnd->layout());
	return qt_widget::get_attr(name);
}

//
// ------------
//

qt_splashscreen::qt_splashscreen(SCM data,qt_widget *parent):
	qt_widget(data)
//	wnd(new QSplashScreen())
{
	QPixmap px(300,50);
	px.fill(Qt::lightGray);
	wnd = new QSplashScreen(px);
	wnd->show();
	qwidget_init(wnd,p_widget(parent));
}

void qt_splashscreen::set_attr(const std::string& name,SCM value)
{
	if (name == "pixmap") attribute_pixmap::set(wnd.get(), value);
	if (name == "message") wnd->message(str2qstr(scm2str(value)));
	
	else qt_widget::set_attr(name,value);
}

SCM qt_splashscreen::get_attr(const std::string& name)
{
	return qt_widget::get_attr(name);
}

//
// ------------
//

qt_dialog::qt_dialog(SCM data,qt_widget *parent):
	qt_widget(data),
	wnd(new QDialog2(p_widget(parent)))
{
	qwidget_init(wnd,p_widget(parent));
	QBoxLayout *l = new QVBoxLayout2(wnd);
	l->setAutoAdd(TRUE);
//	wnd->layout()->setAlignment(Qt::AlignTop);
}

void qt_dialog::set_attr(const std::string& name,SCM value)
{
	if(name == "margin") attribute_margin::set(wnd->layout(), value);
	else if(name == "clear-layout") ((QVBoxLayout2*)(wnd->layout()))->deleteAllItems();
	else if(name == "spacing") attribute_spacing::set(wnd->layout(), value);
	else if(name == "caption") attribute_caption::set(wnd.get(), value);
	else if(name == "backgroundpixmap") attribute_backroundpixmap::set(wnd.get(), value);
	else if(name == "full_screen") wnd->setGeometry(QApplication::desktop()->geometry());
	else qt_widget::set_attr(name,value);
}

SCM qt_dialog::get_attr(const std::string& name)
{
	if(name == "margin") return attribute_margin::get(wnd->layout());
	else if(name == "spacing") return attribute_spacing::get(wnd->layout());
	else if(name == "caption") return attribute_caption::get(wnd.get());
	else if(name == "backgroundpixmap") return attribute_backroundpixmap::get(wnd.get());
	return qt_widget::get_attr(name);
}

void qt_dialog::start()
{
	wnd->exec();
}

void qt_dialog::stop()
{
	wnd->done();
}
