#include <qapplication.h>
#include <qvbox.h>
#include <dialog-qt.hh>
#include <utils/strnum.hh>

struct attribute_text
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		widget->setText(value);
	}

	template <typename W>
	std::string get(W *widget)
	{
		return widget->text();
	}
};

struct attribute_margin
{
	template <typename L>
	void set(L *layout, const std::string& value)
	{
		QBoxLayout *l = dynamic_cast<QBoxLayout *>(layout);
		if(l) l->setMargin(str2num(value));
	}

	template <typename L>
	std::string get(L *layout)
	{
		QBoxLayout *l = dynamic_cast<QBoxLayout *>(layout);
		if(l) return num2str(l->margin());
		return "";
	}
};

struct attribute_spacing
{
	template <typename L>
	void set(L *layout, const std::string& value)
	{
		QBoxLayout *l = dynamic_cast<QBoxLayout *>(layout);
		if(l) l->setSpacing(str2num(value));
	}

	template <typename L>
	std::string get(L *layout)
	{
		QBoxLayout *l = dynamic_cast<QBoxLayout *>(layout);
		if(l) return num2str(l->spacing());
		return "";
	}
};

struct attribute_caption
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		widget->setCaption(value);
	}

	template <typename W>
	std::string get(W *widget)
	{
		return widget->caption();
	}
};

struct attribute_align
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		if(value == "left") _set(widget, Qt::AlignLeft);
		else if(value == "right") _set(widget, Qt::AlignRight);
		else if(value == "center") _set(widget, Qt::AlignHCenter);
		else if(value == "justify") _set(widget, Qt::AlignJustify);
		else if(value == "auto") _set(widget, Qt::AlignAuto);
	}

	template <typename W>
	std::string get(W *widget)
	{
		int f = widget->alignment();
		if(f & Qt::AlignLeft) return "left";
		else if(f & Qt::AlignRight) return "right";
		else if(f & Qt::AlignHCenter) return "center";
		else if(f & Qt::AlignJustify) return "justify";
		else return "auto";
	}

private:
	template <typename W>
	void _set(W *widget, Qt::AlignmentFlags f)
	{
		widget->setAlignment((widget->alignment()
				     & (~Qt::AlignHorizontal_Mask)) | f);
	}
};

struct attribute_width
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		widget->resize(str2num(value), widget->height());
	}

	template <typename W>
	std::string get(W *widget)
	{
		return num2str(widget->width());
	}
};

struct attribute_height
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		widget->resize(widget->width(), str2num(value));
	}

	template <typename W>
	std::string get(W *widget)
	{
		return num2str(widget->height());
	}
};

struct attribute_font
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		QFont font;
		font.setRawName(value);
		widget->setFont(font);
	}

	template <typename W>
	std::string get(W *widget)
	{
		QFont font = widget->font();
		return font.rawName();
	}
};

struct attribute_default
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		if(value == "yes") widget->setDefault(true);
		else if(value == "no") widget->setDefault(false);
	}

	template <typename W>
	std::string get(W *widget)
	{
		return widget->isDefault() ? "yes" : "no";
	}
};

struct attribute_flat
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		if(value == "yes") widget->setFlat(true);
		else if(value == "no") widget->setFlat(false);
	}

	template <typename W>
	std::string get(W *widget)
	{
		return widget->isFlat() ? "yes" : "no";
	}
};

struct attribute_echo
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		if(value == "yes") widget->setEchoMode(QLineEdit::Normal);
		else if(value == "no") widget->setEchoMode(QLineEdit::NoEcho);
		else if(value == "stars") widget->setEchoMode(QLineEdit::Password);
	}

	template <typename W>
	std::string get(W *widget)
	{
		int f = widget->echoMode();
		if(f == QLineEdit::Normal) return "yes";
		else if(f == QLineEdit::NoEcho) return "no";
		else if(f == QLineEdit::Password) return "stars";
		return "";
	}
};

struct attribute_pixmap
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		QPixmap pixmap;
		pixmap.load(value);
		widget->setPixmap(pixmap);
	}

	template <typename W>
	std::string get(W *widget)
	{
		const QPixmap *pixmap = widget->pixmap();
		return pixmap ? "yes" : "no";
	}
};

struct attribute_count
{
	template <typename W>
	std::string get(W *widget)
	{
		return num2str(widget->count());
	}
};

struct attribute_curtext
{
	template <typename W>
	std::string get(W *widget)
	{
		return widget->currentText();
	}
};

struct attribute_current
{
	template <typename W>
	void set(W *widget, const std::string& value)
	{
		widget->setCurrentItem(str2num(value));
	}

	template <typename W>
	std::string get(W *widget)
	{
		return num2str(widget->currentItem());
	}
};


qt_dialog::qt_dialog(QWidget *wnd)
	: wnd_(wnd),
	  top_(new QVBox(wnd))
{
	wnd->layout()->add(top_);
	top_->layout()->setAutoAdd(false);
}

struct destroy
{
	void operator()(widget *ptr) const { delete ptr; }
};

qt_dialog::~qt_dialog(void)
{
	std::for_each(ptrs_.begin(), ptrs_.end(), destroy());
	delete top_;
}

widget *qt_dialog::remember_pointer(widget *ptr)
{
	ptrs_.push_back(ptr);
	return ptr;
}

widget *qt_dialog::make_button(void)
{
	return remember_pointer(new qt_button(top_));
}

widget *qt_dialog::make_label(void)
{
	return remember_pointer(new qt_label(top_));
}

widget *qt_dialog::make_textbox(void)
{
	return remember_pointer(new qt_textbox(top_));
}

widget *qt_dialog::make_listbox(void)
{
	return remember_pointer(new qt_listbox(top_));
}

widget *qt_dialog::make_vbox(void)
{
	return remember_pointer(new qt_vbox(top_));
}

widget *qt_dialog::make_hbox(void)
{
	return remember_pointer(new qt_hbox(top_));
}

void qt_dialog::name_it(const std::string& name, widget* w)
{
	names_[name] = w;
}

widget *qt_dialog::who_is_it(const std::string& name) const
{
	std::map<std::string, widget*>::const_iterator i = names_.find(name);
	if(i == names_.end()) return 0;
	return i->second;
}

long qt_dialog::run(void)
{
	top_->show();
	long i = qApp->exec();
	return i;
}

widget& qt_dialog::set_attr(const std::string& name, const std::string& value)
{
	if(name == "margin") attribute_margin().set(top_->layout(), value);
	else if(name == "spacing") attribute_spacing().set(top_->layout(), value);
	else if(name == "width") attribute_width().set(wnd_, value);
	else if(name == "height") attribute_height().set(wnd_, value);
	else if(name == "caption") attribute_caption().set(wnd_, value);
	return *this;
}

std::string qt_dialog::get_attr(const std::string& name)
{
	if(name == "margin") return attribute_margin().get(top_->layout());
	else if(name == "spacing") return attribute_spacing().get(top_->layout());
	else if(name == "width") return attribute_width().get(wnd_);
	else if(name == "height") return attribute_height().get(wnd_);
	else if(name == "caption") return attribute_caption().get(wnd_);
	return "";
}

widget& qt_dialog::insert(widget *w)
{
	qt_widget *q = dynamic_cast<qt_widget *>(w);
	if(q) top_->layout()->add(q->get_widget());
	return *this;
}

//
// ------------
//

widget& qt_widget::insert(widget *)
{
	return *this;
}

void qt_widget::on_click(void)
{
	qApp->exit(get_event("on-click"));
}

void qt_widget::on_change(const QString&)
{
	qApp->exit(get_event("on-change"));
}

void qt_widget::on_return(void)
{
	qApp->exit(get_event("on-return"));
}

void qt_widget::on_select(void)
{
	qApp->exit(get_event("on-select"));
}

//
// ------------
//

qt_button::qt_button(QWidget *parent): wnd(new QPushButton(parent))
{
	connect(wnd, SIGNAL(clicked()), SLOT(on_click()));
}

widget& qt_button::set_attr(const std::string& name, const std::string& value)
{
	if(name == "text") attribute_text().set(wnd, value);
	else if(name == "default") attribute_default().set(wnd, value);
	else if(name == "flat") attribute_default().set(wnd, value);
	else if(name == "font") attribute_font().set(wnd, value);
	else if(name == "pixmap") attribute_pixmap().set(wnd, value);
	return *this;
}

std::string qt_button::get_attr(const std::string& name)
{
	if(name == "text") return attribute_text().get(wnd);
	else if(name == "default") return attribute_default().get(wnd);
	else if(name == "flat") return attribute_flat().get(wnd);
	else if(name == "font") attribute_font().get(wnd);
	else if(name == "pixmap") attribute_pixmap().get(wnd);
	return "";
}

QWidget *qt_button::get_widget(void) const
{
	return wnd;
}

//
// ------------
//

qt_label::qt_label(QWidget *parent): wnd(new QLabel(parent))
{
}

widget& qt_label::set_attr(const std::string& name, const std::string& value)
{
	if(name == "text") attribute_text().set(wnd, value);
	else if(name == "font") attribute_font().set(wnd, value);
	else if(name == "pixmap") attribute_pixmap().set(wnd, value);
	else if(name == "align") attribute_align().set(wnd, value);
	return *this;
}

std::string qt_label::get_attr(const std::string& name)
{
	if(name == "text") return attribute_text().get(wnd);
	if(name == "font") return attribute_font().get(wnd);
	else if(name == "align") return attribute_align().get(wnd);
	else if(name == "pixmap") return attribute_pixmap().get(wnd);
	return "";
}

QWidget *qt_label::get_widget(void) const
{
	return wnd;
}

//
// ------------
//

qt_textbox::qt_textbox(QWidget *parent): wnd(new QLineEdit(parent))
{
	connect(wnd, SIGNAL(textChanged(const QString&)),
			SLOT(on_change(const QString&)));
	connect(wnd, SIGNAL(returnPressed()), SLOT(on_return()));
}

widget& qt_textbox::set_attr(const std::string& name, const std::string& value)
{
	if(name == "text") attribute_text().set(wnd, value);
	else if(name == "align") attribute_align().set(wnd, value);
	else if(name == "echo") attribute_echo().set(wnd, value);
	return *this;
}

std::string qt_textbox::get_attr(const std::string& name)
{
	if(name == "text") return attribute_text().get(wnd);
	else if(name == "align") return attribute_align().get(wnd);
	else if(name == "echo") return attribute_echo().get(wnd);
	return "";
}

QWidget *qt_textbox::get_widget(void) const
{
	return wnd;
}

//
// ------------
//

qt_listbox::qt_listbox(QWidget *parent): wnd(new QListBox(parent))
{
	connect(wnd, SIGNAL(selectionChanged()), SLOT(on_select()));
}

widget& qt_listbox::set_attr(const std::string& name, const std::string& value)
{
	if(name == "font") attribute_font().set(wnd, value);
	else if(name == "current") attribute_current().set(wnd, value);
	return *this;
}

std::string qt_listbox::get_attr(const std::string& name)
{
	if(name == "text") return attribute_curtext().get(wnd);
	else if(name == "count") attribute_count().get(wnd);
	else if(name == "font") return attribute_font().get(wnd);
	else if(name == "current") return attribute_current().get(wnd);
	return "";
}

QWidget *qt_listbox::get_widget(void) const
{
	return wnd;
}

widget& qt_listbox::insert(widget *w)
{
	qt_label *q = dynamic_cast<qt_label *>(w);
	if(q)
	{
		QLabel *l = dynamic_cast<QLabel *>(q->get_widget());
		l->hide();
		if(l->pixmap()) wnd->insertItem(*l->pixmap());
		else wnd->insertItem(l->text());
	}
	return *this;
}
//
// ------------
//

qt_vbox::qt_vbox(QWidget *parent): wnd(new QFrame(parent))
{
	new QVBoxLayout(wnd);
}

widget& qt_vbox::set_attr(const std::string& name, const std::string& value)
{
	if (name == "margin") attribute_margin().set(wnd->layout(), value);
	else if (name == "spacing") attribute_spacing().set(wnd->layout(), value);
	return *this;
}

std::string qt_vbox::get_attr(const std::string& name)
{
	if(name == "margin") return attribute_margin().get(wnd->layout());
	else if(name == "spacing") return attribute_spacing().get(wnd->layout());
	return "";
}

QWidget *qt_vbox::get_widget(void) const
{
	return wnd;
}

widget& qt_vbox::insert(widget *w)
{
	qt_widget *q = dynamic_cast<qt_widget *>(w);
	if(q)
	{
		q->get_widget()->reparent(wnd, QPoint());
		((QBoxLayout*)wnd->layout())->addWidget(q->get_widget());
	}
	return *this;
}

//
// ------------
//

qt_hbox::qt_hbox(QWidget *parent): wnd(new QFrame(parent))
{
	new QHBoxLayout(wnd);
}

widget& qt_hbox::set_attr(const std::string& name, const std::string& value)
{
	if(name == "margin") attribute_margin().set(wnd->layout(), value);
	else if(name == "spacing") attribute_spacing().set(wnd->layout(), value);
	return *this;
}

std::string qt_hbox::get_attr(const std::string& name)
{
	if(name == "margin") return attribute_margin().get(wnd->layout());
	else if(name == "spacing") return attribute_spacing().get(wnd->layout());
	return "";
}

QWidget *qt_hbox::get_widget(void) const
{
	return wnd;
}

widget& qt_hbox::insert(widget *w)
{
	qt_widget *q = dynamic_cast<qt_widget *>(w);
	if(q)
	{
		q->get_widget()->reparent(wnd, QPoint());
		((QBoxLayout*)wnd->layout())->addWidget(q->get_widget());
	}
	return *this;
}
