#!/usr/bin/guile -s
!#
(use-modules (ice-9 debug) (ice-9 string-fun) (srfi srfi-1))
(define datadir "./lists/")

(define (directory-files dir)
  (let ((dir-stream (opendir dir)))
    (let loop ((new (readdir dir-stream))
	       (acc '()))
      (if (eof-object? new)
	(begin
	  (closedir dir-stream)
	  acc)
	(loop (readdir dir-stream)
	      (if (or (string=? "."  new)             ;;; ignore
		      (string=? ".." new))            ;;; ignore
		acc
		(cons new acc)))))))

(define (list-lists)
  (for-each (lambda (x) (format #t "~A d~%" x))
	    (directory-files datadir)))


(define (read-items lstname)
  (with-input-from-file (string-append datadir lstname)
    (lambda ()
      (let loop ((line (read-line)))
	(if (eof-object? line) '()
	    (cons (with-input-from-string line
		    (lambda ()
		      (let ((name (read))
			    (attrs (read)))
			(cons name
			      (if (eof-object? attrs) '()
				  attrs)))))
		  (loop (read-line))))))))

(define (list-items lstname)
  (for-each (lambda (x) (format #t "~A r~%" (car x)))
	    (read-items lstname)))


(define (read-item lstname itemname)
  (let ((result (assoc itemname (read-items lstname))))
    (if result
	(for-each (lambda (x) (format #t "~A:~A~%" (car x) (cdr x)))
		  (cdr result)))))

(define (make-list lstname)
  (with-output-to-file (string-append datadir lstname)
    (lambda () #f)))

(define (remove-list lstname)
  (if (access? (string-append datadir lstname) F_OK)
      (delete-file (string-append datadir lstname))))

(define (write-items lstname items)
  (with-output-to-file (string-append datadir lstname)
    (lambda ()
      (for-each (lambda (x)
		  (if (null? (cdr x))
		      (format #t "~S~%" (car x))
		      (format #t "~S ~S~%" (car x) (cdr x))))
		items))))

(define (write-item lstname itemname)
  (let ((attrs (read-attrs)))
    (write-items lstname
		 (let* ((items (read-items lstname))
			(result (assoc itemname items)))
		   (if result
		       (set-cdr! result attrs)
		       (set! items (cons (cons itemname attrs) items)))
		   items))))

(define (remove-item lstname itemname)
  (write-items lstname
	       (let loop ((items (read-items lstname)))
		 (if (null? items) '()
		     (if (equal? itemname (caar items))
			 (loop (cdr items))
			 (cons (car items) (loop (cdr items))))))))

(define (print-list lstname)
  (if (access? (string-append datadir lstname) R_OK)
      (format #t "d ~A~%" lstname)))

(define (print-item lstname itemname)
  (if (and (access? (string-append datadir lstname) R_OK)
	   (assoc itemnomo (read-items lstname)))
      (format #t "r ~A/~A~%" lstname itemname)))

(define (usage)
  (display "-h       display help screen")(newline)
  (display "-v       display version information")(newline)
  (display "-l <obj> list of available objects")(newline)
  (display "-f <obj> check object status")(newline)
  (display "-d <obj> remove object")(newline)
  (display "-r <obj> read object info")(newline)
  (display "-w <obj> write object info")(newline))

(define (version)
  (display "lists.scm 1.99")(newline))

(define (act path action lstaction itemaction)
  (separate-fields-discarding-char
    #\/
    path
    (lambda args
      (let ((args (filter (lambda (x) (not (zero? (string-length x)))) args)))
	(cond ((null? args) (action))
	      ((null? (cdr args)) (lstaction (car args)))
	      (else (itemaction (car args) (cadr args))))))))

(define (ignore-action . args) #f)

(define (legi-atributojn)
  (let loop ((line (read-line)))
    (if (eof-object? line) '()
	(split-discarding-char
	  #\:
	  line
	  (lambda (x y) (cons (cons x y) (loop (read-line))))))))

(define (opt=? args opt)
  (string=? (car args) opt))

(define (opt-arg args)
  (if (null? (cdr args)) "" (cadr args)))

(define (main args)
  (let loop ((args (cdr args)))
    (if (null? args) 0
	(let ((arg (string->symbol (car args))))
	  (cond ((opt=? args "-h") (usage))
		((opt=? args "-v") (version))
		((opt=? args "-l")
		 (act (opt-arg args) list-lists list-items ignore-action))
		((opt=? args "-w")
		 (act (opt-arg args) ignore-action make-list write-item))
		((opt=? args "-r")
		 (act (opt-arg args) ignore-action ignore-action read-item))
		((opt=? args "-d")
		 (act (opt-arg args) ignore-action remove-list remove-item))
		((opt=? args "-f")
		 (act (opt-arg args) ignore-action print-list print-item))
		(else (error "invalid option")))))))

(main (command-line))
