;description: alterator abstract model support, implemented as two woobus modules - admiral and rear-admiral

(define-module (alterator admiral)
	       :use-module (srfi srfi-1) 
	       :use-module (srfi srfi-13) 
	       :use-module (ice-9 regex)
	       :use-module (ice-9 rdelim) ; for read-line


	       :use-module (alterator glob)
	       :use-module (alterator str)
	       :use-module (alterator algo)
	       :use-module (alterator pipe)
	       :use-module (alterator command)
	       :use-module (alterator compat command) ;for non-native external filters

	       :use-module (alterator admiral pattern)
	       :use-module (alterator admiral common)

	       :export (admiral rear-admiral line-from-process))


;;helper utility
(define (line-from-process . cmdline)
  (with-input-from-pipe
    cmdline
    (lambda()
      (let ((line (read-line)))
	(if (eof-object? line)
	  ""
	  line)))))

;;;;;;;;;;;;;;;;;;;;; admiral

;execute non-native filter
(define (execute-admiral-filter cmd filename)
  (let ((process (create-process 'read-write filename "convert")))
    (write-command cmd (caddr process))
    (force-output (caddr process))
    (begin-1 (read-command-list (cadr process))
	     (stop-process 'wait process))))

(define (make-admiral-adapted-filter filename)
  (make-adapted-filter filename 
		       execute-admiral-filter))


;create filter function from readed description (native filter)
(define (make-admiral-filter instructions)
  (let ((ctxt (gensym))
	(result (gensym)))
    (eval
      `(lambda (,ctxt)
	 (let ((,result (list #f)))
	   (define (hoo-command name . args)
	     (append! ,result (list `(,(sure-symbol name) ,@args))))
	   (define (all-attrs) (cddr ,ctxt))
	   (define (match num) (admiral-pattern-subexpr (car ,ctxt) num))
	   ,@instructions
	   (cdr ,result)))
     (current-module))))

(define (make-admiral-native-filter filename)
  (define (valid-expr? expr)
    (and (pair? expr) (command? (car expr))))
  (define (precompile-pattern item)
    (let ((pattern (car item))
	  (actions (cdr item)))
      (cons (make-admiral-pattern pattern)
	    (make-admiral-filter actions))))
  (map precompile-pattern
       (admiral-read-script filename valid-expr?)))

;load all admiral (forward) filters into single list
(define (load-admiral-filters directory)
  (define (filter-or-adapter filename)
    (if (admiral-native-filter? filename)
      (make-admiral-native-filter filename)
      (make-admiral-adapted-filter filename)))
  (fold (lambda (filename lst)
	  (format #t "admiral: loading ~A~%" filename)
	  ;or load native script or simulate this action
          (append lst (list (filter-or-adapter filename))))
	'()
	(glob (string-append directory "/*.adm"))))


;try to find in single filter first pattern and eval
(define (admiral-find-and-eval filter cmd)
  (define (sure-list elt) (and elt (list elt)))
  (sure-list
    (find-and-eval cmd filter
		   (lambda (curr cmd)
		     (admiral-pattern-match? curr cmd))
		   (lambda (match-result cmd action)
		     (action (cons match-result cmd))))))

;collect result from filters
(define (admiral-filter-command cmd admirals)
  ;append only non-empty lists to first one
  (fold (lambda (filter result)
	  (append-not-empty result
			    (admiral-find-and-eval filter cmd)))
	'()
	admirals))


;;;;;;;;;;;;;; rear-admiral

;second pass: run commands throw backward filter
(define (pass-throw-rear-admiral-filter back-cmd process)
  (define (read-command-until-stop port)
    (let loop ((result '())
	       (cmd (read-command port)))
      (if (or(null? cmd)
	    (string=?
	      "/rcommander/stop"
	      (sure-string (command-name cmd))))
	result
	(loop (append result (list cmd))
	      (read-command port)))))
  (if (eq? back-cmd 'stop)
    (stop-process 'terminate process)
    (begin (write-command back-cmd (caddr process))
	   (force-output (caddr process))
	   (read-command-until-stop (cadr process)))))

;first pass: backward filter was executed
(define (execute-rear-admiral-filter fwd-cmd filename)
  (let ((process (create-process 'read-write filename "unconvert")))
    (write-command fwd-cmd (caddr process))
    (force-output (caddr process))
    (lambda (cmd) (pass-throw-rear-admiral-filter cmd process))))

(define (make-rear-admiral-adapted-filter filename)
  (make-adapted-filter filename 
		       execute-rear-admiral-filter))

(define (rear-admiral-made-choise item back-cmd)
  (let ((pattern (car item))
	(actions (cdr item))
	(match-result (gensym)))
    `(let* ((,match-result (admiral-pattern-match?
			   ',(make-admiral-pattern pattern)
			   ,back-cmd))
	    (match2 (and ,match-result
			 (lambda (num) (admiral-pattern-subexpr ,match-result num)))))
       (if ,match-result
	 (begin
	   ,@actions)))))

(define (make-rear-admiral-filter choises)
  (let ((ctxt (gensym))
	(back-cmd (gensym))
	(result (gensym)))

    (eval
      `(lambda (,ctxt)
	 (define (match num) (admiral-pattern-subexpr (car ,ctxt) num))
	 (lambda (,back-cmd)
	   (if (not (eq? ,back-cmd 'stop))
	     (let ((,result (list #f)))
	       (define (woo-command name . args)
		 (append! ,result (list `(,(sure-symbol name) ,@args))))
	       (define (all-attrs) (cdr ,back-cmd))
	       ,@(map (lambda (item)
			(rear-admiral-made-choise item back-cmd))
		      choises)
	       (cdr ,result)))))
	 (current-module))))

(define (make-rear-admiral-native-filter filename)
  (define (valid-expr? expr)
    (and (pair? expr)
	 (command? (car expr))
	 (let valid-actions? ((lst (cdr expr)))
	   (cond
	     ((null? lst) #t)
	     ((and (pair? (car lst)) (command? (caar lst)))
	      (valid-actions? (cdr lst)))
	     (else #f)))))
  (define (precompile-pattern item)
    (let ((pattern (car item))
	  (choises (cdr item)))
      (cons (make-admiral-pattern pattern)
	    (make-rear-admiral-filter choises))))
  (map precompile-pattern
       (admiral-read-script filename valid-expr?)))

;load all rear-admiral (backward) filters into single list
(define (load-rear-admiral-filters directory)
  (define (filter-or-adapter filename)
    (if (admiral-native-filter? filename)
      (make-rear-admiral-native-filter filename)
      (make-rear-admiral-adapted-filter filename)))
  (fold (lambda (filename lst)
         (format #t "rear-admiral: loading ~A~%" filename)
         (append lst
                 (list (filter-or-adapter filename))))
        '()
       (glob (string-append directory "/*.radm"))))

(define (rear-admiral-cleanup-filters filters)
  (for-each (lambda (process) (process 'stop))
	    filters))
    
(define (rear-admiral-filter-command cmd  filters)
  (fold (lambda (current lst)
	  (append-not-empty lst (current cmd)))
	'()
	filters))

(define (rear-admiral-drop-errors lst)
  (define (cond-error-command? cmd)
    (and (command? cmd)
	 (string-starts-with? (sure-string (command-name cmd)) "/error?/")))

  (define (extract-object-name name)
    (let ((str (sure-string name)))
	  (substring
	    str
	    (string-index str #\/ 1)
	    (string-length str))))

  ;extract name from error command and search in list for same object
  (define (same-object-exists? name lst)
    (let ((real-name (sure-symbol (extract-object-name name))))
      (find (lambda (x) (eq? real-name (command-name x))) lst)))

  (define (cond-error->error cmd)
    `( ,(sure-symbol (string-append "/error"
				    (extract-object-name (command-name cmd))))
       ,@(cdr cmd)))

  (fold (lambda (cmd result)
	  (if (cond-error-command? cmd)
	    (if (same-object-exists? (command-name cmd) lst)
	      result
	      (append result (list (cond-error->error cmd))))
	    (append result (list cmd))))
	'()
	lst))

;;;;;;;;;;;;;;;;;;; woobus components

(define (admiral-single-query cmd admirals)
  (let ((pre-answer (admiral-filter-command cmd admirals)))
    (if (null? pre-answer)
      (list cmd)
      (admiral-merge pre-answer))))

(define (admiral-main cmd-pair next admirals)
  (let* ((first (car cmd-pair))
    	 (cmds (cdr cmd-pair))
	 (answer (concatenate (map 
				(lambda (cmd)
					(admiral-single-query cmd admirals))
				cmds))))
    (if (find error-command? answer);return up on errors
      (cons first (filter error-command? answer))
      (next (cons first answer)))))


(define (admiral directory)
  (let ((admirals (load-admiral-filters directory)))
    (lambda (cmd-pair next)
      (admiral-main cmd-pair next admirals))))


(define (rear-admiral-main cmd-pair rear-admirals)
  (let* ((first (car cmd-pair))
	 (cmds (cdr cmd-pair))
	 (avail-filters (admiral-filter-command 
			   first
			   rear-admirals)))
    (if (null? avail-filters)
      cmd-pair
      (cons first
	    (rear-admiral-drop-errors
	      (admiral-merge
		(map
		  (lambda (cmd)
		    (rear-admiral-filter-command cmd avail-filters))
		  cmds)))))))

(define (rear-admiral directory)
  (let ((rear-admirals (load-rear-admiral-filters directory)))
    (lambda (cmd-pair next)
      ;process only answers
      (rear-admiral-main (next cmd-pair) rear-admirals))))

