;description : wrapper objects over real widgets

;there are two levels of abstraction
;1. wrapper over real widget (Qt,Gtk,NDK, etc.)
;2. scheme object for this widget to join with scheme event callbacks

(define-module (alterator lookout widgets)
	       :use-module (alterator sandbox)
	       :use-module (alterator lookout events)
	       :use-module (alterator lookout common)
	       :export (on-destroy
			 make-widget
			 declare-widget
			 widget-parse-args
			 pseudo-widget

			 load-widgets))

;;;;;;;;;;;;;;;;;;;;;;;;
(declare-event on-destroy) ;special widget-specific event

(define (pseudo-widget callbacks)
  (lambda (command . args) #f))

;create widget object: container of events, subwidgets and internal widget object
(define (make-widget widget-constructor)
  ;support for declare attribute
  (define (undeclare-attribute field) (if (procedure? field) (field) field))
  (define (undeclare-attrs args) (map undeclare-attribute args))
  ;intellect - set-or-get command
  (define (set-or-get widget command args)
    ;made single list from command and args
    (let ((full-args (if (and (pair? command)
			     (eq? (car command) 'attr))
		      (append (cdr command) args)
		      (append (list command) args))))
      (if (= (length full-args) 1)
	(apply widget 'get full-args)
	(apply widget 'set full-args))))

  (let* ((callbacks (make-event-holder))
	 (subwidgets (list #f))
	 (widget (widget-constructor callbacks)))
    (lambda (command . args) ; dispatcher function
      (cond
	    ;may be preprocessed attribute was received - set or get
	    ((pair? command) (and (eq? (car command) 'attr) (set-or-get widget
									(cadr command)
									(cddr command))))
	    ;attribute was received - set or get
	    ((procedure? command) (set-or-get widget (command) args))
	    ;symbol - normal message
	    ((symbol? command)
	     (case command
	       ((insert) (append! subwidgets (list (car args)))
			 (car args))
	       ((delete) (callbacks 'call on-destroy)
		         (for-each (lambda (x)
			 	(x 'delete)) (cdr subwidgets))
			 (set! subwidgets (list #f))
			 (widget 'delete))
	       ((delete-subwidgets) (widget 'set 'clear-layout #t)
				    (for-each (lambda (x) (x 'delete)) (cdr subwidgets))
				    (set! subwidgets (list #f)))
	       ((show-subwidgets) (for-each (lambda (x) (x 'set 'hidden #f)) (cdr subwidgets)))
	       ((add-event delete-event call set-context get-context) (apply callbacks command args))
	       ;pass other messages to lower object
	       ((set get) (apply widget command (undeclare-attrs args)))
	       ((set-as-is) (apply widget 'set args))
	       (else (apply widget command args))))))))

(define (parse-pair-arg widget arg)
  (case (car arg)
    ((widget) ((widget 'insert ((caddr arg) widget)) 'post-new))
    ((attr)  (apply widget 'set-as-is (cdr arg))) ; we don't need second undeclare now
    ((event) (widget 'add-event arg))
    ;go to sublist
    (else (widget-parse-args widget arg))))

(define (valid-attr? attr) (or (symbol? attr) (string? attr)))

;parse widget args during widget creation
;ignore all unknown argument types
(define (widget-parse-args widget arglist)
  (let loop ((arglist arglist))
    (and (not (null? arglist))
	 (let ((arg (car arglist)))
	   (cond
	     ((pair? arg)
	      (parse-pair-arg widget arg)
	      (widget-parse-args widget (cdr arglist)))
	     ((and (valid-attr? arg) (pair? (cdr arglist)))
	      (widget 'set arg (cadr arglist))
	      (widget-parse-args widget (cddr arglist)))
	     ;support for declare-attribute, convert procedure into symbol by running it
	     ((procedure? arg)
	      (widget-parse-args widget (cons (arg) (cdr arglist))))
	     (else (loop (cdr arglist)))))))
  widget)


;creates function that parse it's args (there are possible some default args) and return
; a list started with widget symbol
(define-macro (declare-widget name create . attrs)
	      (let ((names (map (lambda (x) (gensym)) attrs))
		    (args (gensym)))
		`(define (,name ,@names . ,args)
		   (let ((widget (make-widget ,create)))
		     (list 'widget
			   widget
			   (lambda (parent)
			     (widget 'new parent)
			     (widget 'set-context (fluid-ref lookout-context))
			     ,@(map (lambda (x y)
				      `(widget 'set ',x ,y))
				    attrs names)
			     (widget-parse-args widget ,args)
			     widget))))))

;(define *lookout-known-macros* '(declare-keyword
;				  declare-attribute
;				  declare-event
;				  use-translation))
;;include widget definitions from filename, return list of evaluated expressions
;(define (load-widgets filename)
;  (define (known-macros? item)
;    (and (pair? item)
;	 (member (car item) *lookout-known-macros*)))
;  (let* ((macros (list-extract 
;		   (lookout-process-insertions (read-file filename))
;		   known-macros?))
;	 (defs (sandbox-extract-definitions (cdr macros))))
;    (eval
;      `(begin
;	 ,@(car macros)
;	 ,(sandbox-transform-definitions (car defs)
;					 `((list ,@(cdr defs)))))
;      (current-module))))
;
;(define (lookout-process-insertions lst)
;  (define (insertion? x) (and (pair? x) (eq? (car x) 'include-as-is)))
;  (fold (lambda (x y)
;	  (if (insertion? x)
;	    (append y (read-file (cadr x)))
;	    (append y (list x))))
;	'()
;	lst))

;old and insecure version for Compact release
(define (load-widgets filename)
  (map
    (lambda (x) (eval x (current-module)))
    (read-file filename)))



