;description: set of functions to work with woobus from interface descriptions

(define-module (alterator lookout woo)
	       :use-module (srfi srfi-1)
	       :use-module (alterator algo)
	       :use-module (alterator str)
	       :use-module (alterator command)
	       :export (woo-get-option
			 woo-read-names
			 woo-extract-name
			 woo-first-command
			 woo-error?

			 woo
			 woo-try
			 woo-catch
			 set-woo-query!

			 woo-read
			 woo-list
			 woo-delete
			 woo-new
			 woo-write

			 woo-cache-reset
			 woo-cache-ignore
			 woo-cache-unignore))

(define woo-query #f)
(define (set-woo-query! func) (set! woo-query func))


(define (woo-get-option answer option)
   (or (cond-cdr (command-arg-ref answer option)) ""))

(define (woo-error? cmd) (error-command? cmd))

(define (woo-find-error cmds) (find woo-error? cmds))

;version with predefinded action field
(define (woo action name . args)
  (woo-query `(,(sure-symbol name) action ,(symbol->string action) ,@args)))

;throw on error
(define (woo-try-cmds cmds)
  (let ((error-answer (woo-find-error cmds)))
    (and error-answer
	 (throw 'woo-error (woo-get-option error-answer 'reason)))
    cmds))

;version with exception support
(define (woo-try action name . args)
  (woo-try-cmds (apply woo action name args)))

;wrappers over standard actions
(define (woo-read name) (woo-try 'read name))
(define (woo-list name) (woo-try 'list name))
(define (woo-delete name . args) (apply woo-try 'delete name args))
(define (woo-new name . args) (apply woo-try 'new name args))
(define (woo-write name . args) (apply woo-try 'write name args))

;try-catch cycle
(define (woo-catch run-code catch-code . finally)
  (let ((first-retcode
	  (catch 'woo-error
		 run-code
		 (lambda (key . args) (catch-code (car args))))))
    (if (not (null? finally)) ((car finally)))
    first-retcode))

(define (woo-extract-name name cmd)
  (let ((str (sure-string (command-name cmd))))
    (if (> (string-length str) (string-length name))
      (substring str (+ (string-length name) 1) (string-length str))
      str)))


(define (woo-read-names name cmdlist)
  (define (extract-name cmd) (woo-extract-name name cmd))
  (map extract-name cmdlist))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (woo-cache-common type possible . args)
  (if (and (not (null? args))
           (= (length args) 2)
           (member (car args) possible))
    (woo-try-cmds 
      (woo-query `(/ctrl/cache action ,type ,(car args) ,(sure-string (cadr args)))))
    (woo-try-cmds
      (woo-query `(/ctrl/cache action ,type)))))

(define (woo-cache-reset . args)
  (apply woo-cache-common "reset" '(id) args))

(define (woo-cache-unignore . args)
  (apply woo-cache-common "unignore" '(id) args))

(define (woo-cache-ignore . args)
  (apply woo-cache-common "ignore" '(id count) args))

(define (woo-first-command cmdlist)
  (and (pair? cmdlist) (car cmdlist)))

