;user's session in generic look of alterator
(define-module (alterator generic session)
  :use-module (ice-9 threads)
  :use-module (srfi srfi-1)
  :use-module (alterator algo)
  :use-module (alterator object)
  :use-module (alterator lookout context)
  :use-module (alterator lookout woo)
  :export (generic-session
  	   generic-post-event

           make-generic-session
           find-generic-session
           close-generic-session))
           
;;;;;;;;;;;;;;;;;;; internal definitions

(define generic-session-list '()) ;internal alist of sessions

;TODO: made more strong generator
(define random-state (seed->random-state (current-time)))

(define (make-session-id) (random 100000000 random-state))

(define (make-widget-id-generator)
  (let ((counter 0))
    (lambda ()
      (begin-1
	(string->symbol (string-append "w" (number->string counter)))
	(set! counter (+ counter 1))))))

;;;;;;;;;;;;;;;;;;;; extrenal definitions


(define generic-session (make-fluid))
(fluid-set! generic-session '())

(define (generic-post-event . args)
  (define-operation write-output)
  (let ((session (fluid-ref generic-session)))
    (and session (apply write-output session args))))


(define (make-session-object id username language)
  (define-operation reset-output)
  (define-operation signal-output)

  (let ((session-id id)
        (input-mutex (make-mutex))
        (output-mutex (make-mutex))
        (input-condvar (make-condition-variable))
        (output-condvar (make-condition-variable))

        (input-queue #f)
        (output-queue (list #f))
        
        (main-thread #f)
        (widgets '())
        (id-generator (make-widget-id-generator))
        (user username)
        (lang language))

        (object
         #f
         ;simple properties
         ((get-session-id self) id)
         ((get-user self) user)
         ((get-lang self) lang)
         ((make-widget-id self) (id-generator))

         ;add remove widgets to list
         ((add-widget self id w)
          (set! widgets (alist-set id w widgets)))
         ((remove-widget self id)
          (set! widgets (alist-delete id widgets)))
         ((get-widget self id)
	 	(cond-cdr (assq (sure-symbol id) widgets)))

         ;main internal thread
         ((start-thread self proc)
          (let ((l-context (fluid-ref lookout-context))
	        (w-gate (fluid-ref woo-gate)))
            (set! main-thread (begin-thread ;bind to thread both lookout-context and session-context
                               (with-fluids ((generic-session self)
			                     (lookout-context l-context)
					     (woo-gate w-gate))
                                            (proc)
                                            (close-generic-session) ;notify about end of work only here to avoid races
					    (signal-output self)))))) ;last message (we still exists in fluids)
         
         ;input-output queues
         ((write-input self msg)
          (with-mutex input-mutex
                      (set! input-queue msg)
                      (signal-condition-variable input-condvar)))
         ((read-input self)
          (with-mutex input-mutex
                      (wait-condition-variable input-condvar input-mutex)
                      (begin-1 input-queue
                               (set! input-queue #f))))
         ((signal-output self)
          (with-mutex output-mutex
                      (signal-condition-variable output-condvar)))
         ((write-output self . command )
          (with-mutex output-mutex
                      (append! output-queue (list (cons 'command command)))))
         ((reset-output self)
	  (set! output-queue (list #f)))
         ((read-output self)
          (with-mutex output-mutex
                      (wait-condition-variable output-condvar output-mutex)
                      (begin-1 (cdr output-queue)
		               (reset-output self)))))))
         
(define (make-generic-session username language)
  (let* ((id (make-session-id))
         (session (make-session-object id username language)))
    (set! generic-session-list (alist-set id session generic-session-list))
    session))

(define (find-generic-session username session-id)
  (define-operation get-user)
  
  (let ((session (cond-cdr (assv (string->number session-id) generic-session-list))))
    (and session 
         (string=? (get-user session) username)
         session)))

(define (close-generic-session)
  (define-operation get-session-id)
  (set! generic-session-list (alist-delete (get-session-id (fluid-ref generic-session)) generic-session-list)))
