/*
 *  ALTerator - ALT Linux configuration project
 *
 *  Copyright (c) 2004,2005 ALT Linux Ltd.
 *  Copyright (c) 2004,2005 Alexey Voinov
 *  Copyright (c) 2004,2005 Stanislav Ievlev
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 *  USA.
 */

namespace attribute_text
{
	template <typename W>
	void set(W *widget,SCM value)
	{
		SCM_ASSERT(SCM_STRINGP(value),value,SCM_ARG1,"attribute_text");
		widget->setText(str2qstr(scm2str(value)));
	}

	template <typename W>
	SCM get(W *widget)
	{
		return str2scm(qstr2str(widget->text()));
	}
};

namespace attribute_title
{
	template <typename W>
	void set(W *widget,SCM value)
	{
		SCM_ASSERT(SCM_STRINGP(value),value,SCM_ARG1,"attribute_text");
		widget->setTitle(str2qstr(scm2str(value)));
	}

	template <typename W>
	SCM get(W *widget)
	{
		return str2scm(qstr2str(widget->title()));
	}
};

namespace attribute_widget_id
{
	template <typename W>
	void set(W *widget,SCM value)
	{
		SCM_ASSERT(SCM_SYMBOLP(value),value,SCM_ARG1,"attribute_widget_id");
		widget->setName(scm2symbol(value));
	}

	template <typename W>
	SCM get(W *widget)
	{
		return symbol2scm(widget->name());
	}
};

namespace attribute_mask
{
	template <typename W>
	void set(W *widget, SCM value)
	{
		SCM_ASSERT(SCM_STRINGP(value),value,SCM_ARG1,"attribute_mask");
		widget->setInputMask(str2qstr(scm2str(value)));
	}

	template <typename W>
	SCM get(W *widget)
	{
		return str2scm(qstr2str(widget->inputMask()));
	}
};

namespace attribute_margin
{
	template <typename L>
	void set(L *layout,SCM value)
	{
		SCM_ASSERT(SCM_NUMBERP(value),value,SCM_ARG1,"attribute_margin");
		QBoxLayout *l = dynamic_cast<QBoxLayout *>(layout);
		if(l) l->setMargin(percent_to_height(scm2num(value)));
	}

	template <typename L>
	SCM get(L *layout)
	{
		QBoxLayout *l = dynamic_cast<QBoxLayout *>(layout);
		if(l) return num2scm(height_to_percent(l->margin()));
		return SCM_UNSPECIFIED;
	}
};

namespace attribute_spacing
{
	template <typename L>
	void set(L *layout,SCM value)
	{
	        SCM_ASSERT(SCM_NUMBERP(value),value,SCM_ARG1,"attribute_spacing");
		QBoxLayout *l = dynamic_cast<QBoxLayout *>(layout);
		if(l) l->setSpacing(percent_to_height(scm2num(value)));
	}

	template <typename L>
	SCM get(L *layout)
	{
		QBoxLayout *l = dynamic_cast<QBoxLayout *>(layout);
		if(l) return num2scm(height_to_percent(l->spacing()));
		return SCM_UNSPECIFIED;
	}
};

namespace attribute_caption
{
	template <typename W>
	void set(W *widget,SCM value)
	{
	        SCM_ASSERT(SCM_STRINGP(value),value,SCM_ARG1,"attribute_caption");
		widget->setCaption(str2qstr(scm2str(value)));
	}

	template <typename W>
	SCM get(W *widget)
	{
		return str2scm(qstr2str(widget->caption()));
	}
};

namespace attribute_align
{
	namespace
	{
		template <typename W>
		void _set(W *widget, Qt::AlignmentFlags f)
		{
			widget->setAlignment((widget->alignment()
				     & (~Qt::AlignHorizontal_Mask)) | f);
		}
	}


	template <typename W>
	void set(W *widget,SCM obj)
	{
	        SCM_ASSERT(SCM_SYMBOLP(obj),obj,SCM_ARG1,"attribute_align");
		const std::string value=scm2symbol(obj);
		if(value == "left") _set(widget, Qt::AlignLeft);
		else if(value == "right") _set(widget, Qt::AlignRight);
		else if(value == "center") _set(widget, Qt::AlignHCenter);
		else if(value == "justify") _set(widget, Qt::AlignJustify);
		else if(value == "auto") _set(widget, Qt::AlignAuto);
	}

	template <typename W>
	SCM get(W *widget)
	{
		int f = widget->alignment();
		if(f & Qt::AlignLeft) return symbol2scm("left");
		else if(f & Qt::AlignRight) return symbol2scm("right");
		else if(f & Qt::AlignHCenter) return symbol2scm("center");
		else if(f & Qt::AlignJustify) return symbol2scm("justify");
		else return symbol2scm("auto");
	}
};

namespace attribute_width
{
	template <typename W>
	void set(W *widget,SCM value)
	{
	        SCM_ASSERT(SCM_NUMBERP(value),value,SCM_ARG1,"attribute_width");
		widget->setFixedWidth(percent_to_width(scm2num(value)));
	}

	template <typename W>
	SCM get(W *widget)
	{
		return num2scm(width_to_percent(widget->width()));
	}
};

namespace attribute_height
{
	template <typename W>
	void set(W *widget,SCM value)
	{
	        SCM_ASSERT(SCM_NUMBERP(value),value,SCM_ARG1,"attribute_height");
		widget->setFixedHeight(percent_to_height(scm2num(value)));
	}

	template <typename W>
	SCM get(W *widget)
	{
		return num2scm(height_to_percent(widget->height()));
	}
};

namespace attribute_font
{
	template <typename W>
	void set(W *widget,SCM value)
	{
	        SCM_ASSERT(SCM_STRINGP(value),value,SCM_ARG1,"attribute_font");
		QFont font;
		font.setRawName(str2qstr(scm2str(value)));
		widget->setFont(font);
	}

	template <typename W>
	SCM get(W *widget)
	{
		QFont font = widget->font();
		return str2scm(qstr2str(font.rawName()));
	}
};

namespace attribute_default
{
	template <typename W>
	void set(W *widget,SCM value)
	{
		widget->setDefault(scm2bool(value));
	}

	template <typename W>
	SCM get(W *widget)
	{
		return bool2scm(widget->isDefault());
	}
};

namespace attribute_readonly
{
	template <typename W>
	void set(W *widget,SCM value)
	{
		widget->setReadOnly(scm2bool(value));
	}

	template <typename W>
	SCM get(W *widget)
	{
		return bool2scm(widget->isReadOnly());
	}
};

namespace attribute_focus
{
	template <typename W>
	void set(W *widget,SCM value)
	{
		(scm2bool(value))?widget->setFocus(): widget->clearFocus();
	}

	template <typename W>
	SCM get(W *widget)
	{
		return bool2scm(widget->hasFocus());
	}
};

namespace attribute_editable
{
	template <typename W>
	void set(W *widget,SCM value)
	{
		widget->setEditable(scm2bool(value));
	}

	template <typename W>
	SCM get(W *widget)
	{
		return bool2scm(widget->editable());
	}
};

namespace attribute_enabled
{
	template <typename W>
	void set(W *widget,SCM value)
	{
		widget->setEnabled(scm2bool(value));
	}

	template <typename W>
	SCM get(W *widget)
	{
		return bool2scm(widget->isEnabled());
	}
};

namespace attribute_hidden
{
	template <typename W>
	void set(W *widget,SCM obj)
	{
		bool value = scm2bool(obj);
		if(value) widget->hide();
		else widget->show();
	}

	template <typename W>
	SCM get(W *widget)
	{
		return bool2scm(widget->isHidden());
	}
};


namespace attribute_flat
{
	template <typename W>
	void set(W *widget,SCM value)
	{
		widget->setFlat(scm2bool(value));
	}

	template <typename W>
	SCM get(W *widget)
	{
		return bool2scm(widget->isFlat());
	}
};

namespace attribute_echo
{
	template <typename W>
	void set(W *widget,SCM obj)
	{
	        SCM_ASSERT(SCM_SYMBOLP(obj),obj,SCM_ARG1,"attribute_echo");
		const std::string value = scm2symbol(obj);
		if(value == "yes") widget->setEchoMode(QLineEdit::Normal);
		else if(value == "no") widget->setEchoMode(QLineEdit::NoEcho);
		else if(value == "stars") widget->setEchoMode(QLineEdit::Password);
	}

	template <typename W>
	SCM get(W *widget)
	{
		int f = widget->echoMode();
		if(f == QLineEdit::Normal) return symbol2scm("yes");
		else if(f == QLineEdit::NoEcho) return symbol2scm("no");
		else if(f == QLineEdit::Password) return symbol2scm("stars");
		return SCM_UNSPECIFIED;
	}
};


namespace attribute_sizepolicy
{
	namespace
	{
		QSizePolicy::SizeType name_to_type(SCM obj)
		{
			const std::string value = scm2symbol(obj);
			if (value == "fixed") return QSizePolicy::Fixed;
			if (value == "expanding") return QSizePolicy::Expanding;
			return QSizePolicy::Preferred;
		}
	}
	
	template <typename W>
	void set(W *widget,SCM obj)
	{
	        SCM_ASSERT(SCM_SYMBOLP(obj) ||
		           (SCM_NFALSEP(scm_pair_p(obj)) &&
			      SCM_SYMBOLP(SCM_CAR(obj)) && SCM_SYMBOLP(SCM_CDR(obj))),
			   obj,SCM_ARG1,"attribute_sizepolicy");
		if (SCM_NFALSEP(scm_pair_p(obj)))
			widget->setSizePolicy(name_to_type(SCM_CAR(obj)),
					      name_to_type(SCM_CDR(obj)));
		else
		{
			QSizePolicy::SizeType type = name_to_type(obj);
			widget->setSizePolicy(type,type);
		}
	}

	template <typename W>
	SCM get(W *widget)
	{
		return SCM_UNDEFINED;
	}
};

namespace attribute_pixmap
{
	template <typename W>
	void set(W *widget,SCM value)
	{
	        SCM_ASSERT(SCM_STRINGP(value),value,SCM_ARG1,"attribute_pixmap");
		QPixmap pix(str2qstr(scm2str(value)));
		widget->setPixmap(pix);
//     		widget->setBackgroundColor(lbl1->pixmap()->convertToImage().pixel(0, 0));
		widget->setMinimumSize(pix.size());
//       		lbl1->setSizePolicy(QSizePolicy::Minimum,QSizePolicy::Fixed);

	}

	template <typename W>
	SCM get(W *widget)
	{
		const QPixmap *pixmap = widget->pixmap();
		return bool2scm(pixmap != 0);
	}
};

namespace attribute_backroundpixmap
{
	template <typename W>
	void set(W *widget,SCM value)
	{
	        SCM_ASSERT(SCM_STRINGP(value),value,SCM_ARG1,"attribute_backgroundpixmap");
		widget->setPaletteBackgroundPixmap(QPixmap(str2qstr(scm2str(value))));
	}

	template <typename W>
	SCM get(W *widget)
	{
		const QPixmap *pixmap = widget->paletteBackgroundPixmap();
		return bool2scm( pixmap!=0 );
	}
};

namespace attribute_count
{
	template <typename W>
	SCM get(W *widget)
	{
		return num2scm(widget->count());
	}
};

namespace attribute_curtext
{
	template <typename W>
	SCM get(W *widget)
	{
		return str2scm(qstr2str(widget->currentText()));
	}
};

namespace attribute_current
{
	template <typename W>
	void set(W *widget,SCM value)
	{
	        SCM_ASSERT(SCM_NUMBERP(value),value,SCM_ARG1,"attribute_current");
		widget->setCurrentItem(scm2num(value));
	}

	template <typename W>
	SCM get(W *widget)
	{
		return num2scm(widget->currentItem());
	}
};

namespace attribute_checked
{
	template <typename W>
	void set(W *widget,SCM value)
	{
		widget->setChecked(scm2bool(value));
	}

	template <typename W>
	SCM get(W *widget)
	{
		return bool2scm(widget->isChecked());
	}
};

namespace attribute_checkable
{
	template <typename W>
	void set(W *widget,SCM value)
	{
		widget->setCheckable(scm2bool(value));
	}

	template <typename W>
	SCM get(W *widget)
	{
		return bool2scm(widget->isCheckable());
	}
};
