;;;
;;; ALTerator - ALT Linux configuration project
;;;
;;; Copyright (c) 2004,2005 ALT Linux Ltd.
;;; Copyright (c) 2004,2005 Alexey Voinov
;;; Copyright (c) 2004,2005 Stanislav Ievlev
;;;
;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2 of the License, or
;;; (at your option) any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
;;; USA.
;;;

;description: main lookout mapper between dialogs and it's ids

(define-module (alterator lookout mapper)
	       :use-module (srfi srfi-1)
	       :use-module (srfi srfi-13)
	       :use-module (ice-9 regex)
	       :use-module (alterator str)
	       :use-module (alterator algo)
	       :use-module (alterator glob)
	       :use-module (alterator lookout context)
	       :export (later
			with-mapper
	       
	       		load-mapper
	                mapper-view
			mapper-view-all))

(define-macro (with-mapper <mapper> <instruction> . <instructions>)
    `(with-changed-context 'mapper ,<mapper> ,<instruction> . ,<instructions>))

(define-macro (later <instruction> . <instructions>)
	      `(lambda () ,<instruction> . ,<instructions>))

(define (cond-cons x y) (and x (cons x y)))

(define (load-mapper filename)
  (with-input-from-file filename
    (lambda () 
      (let loop ((e (read)))
	(if (eof-object? e) '()
	    (fold-right
	      (lambda (x y) (or (cond-cons (parse-mapper-item x) y) y))
	      (loop (read))
	      (perform-unquote e)))))))

(define (parse-mapper-item item)
  (and (pair? item)
       (cond-cons (convert-key (car item))
		  (convert-command (cdr item)))))

(define (convert-key key)
  (cond ((string? key) (lambda (x) (string=? key x)))
	((regexp? key) (lambda (x) (regexp-exec key x)))
	((null? key) (lambda (x) #t))
	((pair? key) (apply disjoin (map convert-key key)))
	((symbol? key) (convert-key (symbol->string key)))
	((procedure? key) key)
	(else #f)))

(define (convert-command cmd)
  (cond ((null? cmd)
	 '())
	((eq? 'file (car cmd))
	 (if (null? (cdr cmd)) '()
	     (cons* '/ctrl/look 'action 'file 'name (cadr cmd) (cddr cmd))))
	((eq? 'view (car cmd))
	 (if (null? (cdr cmd)) '()
	     (cons* '/ctrl/look 'action 'view 'id (cadr cmd) (cddr cmd))))
	(else
	  cmd)))

(define (mapper-view id . rest)
  (let loop ((mapper (if (null? rest) (from-context 'mapper) (car rest))))
    (and (pair? mapper)
	 (or (test-and-eval ((caar mapper) (sure-string id)) (cdar mapper))
	     (loop (cdr mapper))))))

(define (mapper-view-all id . rest)
  (let ((mapper (if (null? rest) (from-context 'mapper) (car rest))))
    (define (cond-append result lst)
      (or
	(and lst (append result (list lst)))
	result))
    (let loop ((mapper mapper)
	       (result '()))
      (if (null? mapper)
	result
	(loop (cdr mapper)
	      (cond-append result
			   (test-and-eval ((caar mapper) (sure-string id)) (cdar mapper))))))))

(define (test-and-eval ctxt answer)
  (and ctxt
       (map (lambda (x) (if (procedure? x) (x) x))
	    answer)))

(define (perform-unquote e)
  (if (pair? e)
      (case (car e)
	((unquote) (list (eval (cadr e) (current-module))))
	((unquote-splicing) (eval (cadr e) (current-module)))
	((include) (fold (lambda (x y) (append y x)) '()
			 (map load-mapper (glob (cadr e)))))
	(else (list (fold-right append '() (map perform-unquote e)))))
      (list e)))

(define (disjoin . funcs)
  (lambda args (any? (lambda (f) (apply f args)) funcs)))

(define (any? pred lst)
  (if (null? lst) #f
      (or (pred (car lst))
	  (any? pred (cdr lst)))))
