// Corpora.cpp : Implementation of CCorpora

#include "../common/string_socket.h"
#include "../ConcordLib/ConcHolder.h"
#include "Corpora.h"
#include "Morphan.h"
#include "../common/util_classes.h"

#ifdef WIN32
	#include <process.h>  /* _beginthread, _endthread */
#else
#endif


/*
#define _CRTDBG_MAP_ALLOC
#include <stdlib.h>
#include <crtdbg.h>
*/

#ifdef DETECT_MEMORY_LEAK
	#ifdef _DEBUG
	#define new DEBUG_NEW
	#undef THIS_FILE
	static char THIS_FILE[] = __FILE__;
	#endif
#endif

const char* AdditionalHitDelimeter = "\x1";

class CDDCCorpusListenHost  : public CHost
{
public:
	CConcHolder*					m_pHolder;
	
	CDDCCorpusListenHost (bool bDoubleMessage, ProtocolDensityEnum pdProtocolDensity);
	virtual ~CDDCCorpusListenHost ();
	string		ProcessSocketString(const string& S, SOCKET rConnectedSocket);
};

struct CQueryStatistics 
{
	DwordVector		m_HitsEnds;
	DWORD			m_FreqCount;
};

struct CHitToSort {
	BYTE	m_HostNo;
	DWORD	m_HitNo;
	DWORD	m_OrderId;
	bool operator < (const CHitToSort& X) const {
		if (m_OrderId != X.m_OrderId)
			return m_OrderId < X.m_OrderId;
		return m_HitNo < X.m_HitNo;
	};
};


class CDDCServerListenHost  : public CHost
{
public:
	struct CFirstHitsQueryResult 
	{
		DWORD				m_AllHitsCount;
		DWORD				m_AllRelevantDocsCount; 
		string				m_HitsDistributionStr;
		string				m_RelevantDocsDistributionStr;
		int					m_InternalError;
		bool				m_bSort;
		NetworkErrorsEnum	m_NetworkError;
		vector<CHitToSort>  m_Hits;
		vector<SOCKET>		m_Sockets;

		bool HasErrors() const
		{
			return		(m_NetworkError != neSuccess ) 
					||	(m_InternalError != errUnknown);
		};

		void CloseSockets()
		{
			// closing sockets, which were opened before
			for (size_t j=0; j<m_Sockets.size(); j++)
				if (m_Sockets[j] != -1)
				{
					CloseSocket(m_Sockets[j]);
					m_Sockets[j] = -1;
				};
		};
	};

	map<string, CQueryStatistics>	m_QueryToStatis;
	vector<CHost>					m_Hosts;	
	string							m_QueryResultString;

	CDDCServerListenHost (bool bDoubleMessage, ProtocolDensityEnum pdProtocolDensity);
	virtual ~CDDCServerListenHost ();

	string				ProcessSocketString(const string& S, SOCKET rConnectedSocket);
	void				GetFirstHitsFromCorpora(const DwordVector& PossibleHosts, const string Query,  const DWORD ResultLimit, const int TimeOut, CFirstHitsQueryResult& Result);
	NetworkErrorsEnum	GetHitStringsFromOneCorpora(CHost& Host,  const string& ResultType, DWORD StartHitNo, DWORD ResultLimit, int TimeOut, SOCKET& CorpusSocket, int& InternalError, string& Result );
	NetworkErrorsEnum	RunDistributed(string Query, const string& ResultType, DWORD StartHitNo, DWORD ResultLimit, int& iInternalError, DWORD& EndHitNo, DWORD& HitsCount,  int TimeOut, string& ResultString, DWORD& RelevantDocsCount);
	bool				ReadPossibleHosts(const string& InputQuery, string& CleanQuery, DwordVector& PossibleHosts) const;
};


// all sockets which are opened by this daemon 
vector<CDDCCorpusListenHost> LocalCorpora;

CDDCServerListenHost* pServer = 0;


CDDCCorpusListenHost::CDDCCorpusListenHost (bool bDoubleMessage, ProtocolDensityEnum pdProtocolDensity) : 
		CHost(bDoubleMessage, pdProtocolDensity)
{
	m_pHolder = 0;
	m_LogFunction = concord_daemon_log;
	
};


CDDCCorpusListenHost::~CDDCCorpusListenHost ()
{
	StopAcceptingThread();
};




bool SaveTrigger(const string& S, DWORD LParam)
{
	string ErrorStr;
	if (!SendString(LParam, S.c_str(), S.length(), ErrorStr))
	{
		concord_daemon_log(Format("cannot send a string for GetOccurrences: %s", ErrorStr.c_str()));
		return false;
	};
	return true;
};



string CDDCCorpusListenHost::ProcessSocketString(const string& InputSocket, SOCKET rConnectedSocket) 
{
	string  Result;
	DWORD  StartHitNo;
	char sResultType[100];

	//sResultType\x1StartHitNo ResultLimit

	int CountItems = sscanf (InputSocket.c_str(), "get_hit_strings %[^\x1]\x1%i %i", sResultType,  &StartHitNo,&m_pHolder->m_ResultLimit);
	if(CountItems == 3) 
	{
		m_pHolder->SetResultFormat(sResultType);
		int  Error = m_pHolder->GenerateHitStrings(StartHitNo);

		if (Error == errUnknown) 
			if (m_pHolder->m_QueryResultStr.empty())
				if (m_pHolder->GetResultFormatStr() == "html")
					if (m_pHolder->m_ResultLimit > 0)
						m_pHolder->m_QueryResultStr = "no entries found";


		//InternalError\x1Result
		Result = Format ("%i\x1", Error);
		Result += m_pHolder->m_QueryResultStr;
		m_pHolder->m_QueryResultStr= "";
	}
	else
	{
		// getting hits
		DWORD  EndHitNo = 0;
		int	TimeOut;
		char sQuery[3000];
		const char* strInputSocket = InputSocket.c_str();
	
		int CountItems = sscanf (strInputSocket, "get_first_hits %[^\x1]\x1 %i %i", sQuery, &TimeOut,&m_pHolder->m_ResultLimit);
		if(CountItems == 3) 
		{
			m_pHolder->SetTimeOut(TimeOut);
			int Error = m_pHolder->GetHits(sQuery, EndHitNo);
			Result = Format ("%i %i %i %i %i", Error, EndHitNo, m_pHolder->m_AllHitsCount, m_pHolder->m_RelevantDocumentCount, (m_pHolder->HitsShouldBeSorted()?1:0));
			Result += m_pHolder->GetHitIds();
		}
		else
		{
			concord_daemon_log("Error:Cannot parse socket input");
			return "";
		};
	};
		
	return Result;
};




//==========================================================
//========  CDDCServerListenHost  ==========================
//==========================================================

CDDCServerListenHost::CDDCServerListenHost (bool bDoubleMessage, ProtocolDensityEnum pdProtocolDensity) : 
CHost(bDoubleMessage, pdProtocolDensity)
{
	m_LogFunction = concord_daemon_log;
};


CDDCServerListenHost::~CDDCServerListenHost ()
{
	StopAcceptingThread();
};


// processing input for one daemon
//  there are two possibilities:
//  1. "run_query" returns results for DDC
//  2. "get_paradigm" returns a paradigm from morphology 
string CDDCServerListenHost::ProcessSocketString(const string& InputSocket, SOCKET rConnectedSocket) 
{
	string  Result;

	concord_daemon_log(Format("%s recv \"%s\"", m_CorporaName.c_str(), InputSocket.c_str()));


	DWORD StartHitNo, ResultLimit;
	int TimeOut;
	char sQuery[3000];
	char sCorporaName[1000];
	char sResultType[100];
	int CountItems = sscanf (InputSocket.c_str(), 
				"run_query %[^\x1]\x1%[^\x1]\x1%[^\x1]\x1%i %i %i", 
							sCorporaName, sQuery, sResultType, &StartHitNo, &ResultLimit, &TimeOut);

	Result.reserve(10000);

	if(CountItems == 6) 
	{

		int InternalError;
		int iNetworkError;
		DWORD EndHitNo, HitsCount, RelevantDocsCount;
		string ResultType = sResultType;
		string strQuery = sQuery;
		Trim(ResultType);
		Trim(strQuery);
		if ( strQuery.empty() ) 
		{
			concord_daemon_log("Error:Cannot parse socket input");
			return "";
		}
		string ResultString;
		iNetworkError = RunDistributed(strQuery, ResultType, StartHitNo, ResultLimit, InternalError, EndHitNo, HitsCount, TimeOut, ResultString, RelevantDocsCount);
		Result = ResultString + Format ("\x1 %i %i %i %i %i", InternalError, iNetworkError, EndHitNo, HitsCount, RelevantDocsCount);
	}
	else
	{
		char sWord[255];
		int iNurLemmatize, iLangua;
		int CountItems = sscanf (InputSocket.c_str(), "get_paradigm %[^\x1]\x1%i %i", sWord, &iNurLemmatize, &iLangua);
		if (CountItems != 3)
		{
			concord_daemon_log("Error:Cannot parse socket input");
			return "";
		};
		int LemmasCount;
		Result = Lemmatize(sWord, (MorphLanguageEnum)iLangua, LemmasCount);
		if (!iNurLemmatize)
		{
			for (int i = 0; i < LemmasCount; i++)
			{
				Result += "\n";
				Result += GetParadigm(sWord, i, (MorphLanguageEnum)iLangua);
			};
		};
	};

	return Result;
};


/*
	return false only when a network error occurs
*/
const size_t MaxSubhostsSize = 256;
bool CDDCServerListenHost::ReadPossibleHosts(const string& InputQuery, string& CleanQuery, 
									DwordVector& PossibleHosts) const
{
	PossibleHosts.reserve(m_Hosts.size());
	int Starter;
	for (Starter  = InputQuery.length()-1;   Starter >= 0; Starter--)
		if ((BYTE)InputQuery[Starter] == ':')
			break;
		else
			if (		(		!is_english_alpha((BYTE)InputQuery[Starter])
							&&	!strchr("_-1234567890,# \t", (BYTE)InputQuery[Starter])
						)
					||	(Starter == 0)
				)
			{
				for (size_t i=0; i<m_Hosts.size(); i++)
					PossibleHosts.push_back(i);
				CleanQuery = InputQuery;
				return true;
			}


	assert (Starter != string::npos);

	string s = InputQuery.substr(Starter+1);
	Trim(s);
	if (s.empty()) return false;
	StringTokenizer tok(s.c_str(),", ");
	while (tok())
	{
		string CorporaName = tok.val();
		size_t j=0;
		for (; j<m_Hosts.size(); j++)
			if (m_Hosts[j].m_CorporaName == CorporaName)
				break;
		if ( j == m_Hosts.size()) return false;
		PossibleHosts.push_back(j);
	};
	CleanQuery = InputQuery.substr(0, Starter);
	if (PossibleHosts.size() >= MaxSubhostsSize)
	{
		return false;
	};
	return true;
};


NetworkErrorsEnum CDDCServerListenHost::GetHitStringsFromOneCorpora(CHost& Host,  const string& ResultType,  
												DWORD StartHitNo, DWORD ResultLimit, int TimeOut, SOCKET& CorpusSocket, int& InternalError, string& Result )
{
	if (CorpusSocket == -1)
	{
		return neNetworkError;
	};

	//Query\x1StartHitNo ResultLimit bGetText
	string S = Format ("get_hit_strings %s\x1%i %i",  ResultType.c_str(), StartHitNo,  ResultLimit);
	string ErrorStr;
	if (!SendString(CorpusSocket, S.c_str(), S.length(), ErrorStr)) 
	{
		ErrorMessage (Format("cannot send a string to socket %s : %s", Host.GetAddressStr().c_str(), ErrorStr.c_str() ));

		if (!CloseSocket(CorpusSocket))
			ErrorMessage (Format("cannot close socket %s", Host.GetAddressStr().c_str()));
		CorpusSocket =-1;

		return neCouldNotSendData;
	};

	NetworkErrorsEnum Res = RecieveString(CorpusSocket, S, TimeOut);
	if (!CloseSocket(CorpusSocket))
		ErrorMessage (Format("cannot close socket %s", Host.GetAddressStr().c_str()));
	CorpusSocket =-1;

	if (Res  != neSuccess)
	{
		concord_daemon_log ( " Cannot recieve a resulting string from a client  " );
		return Res;
	};

	//InternalError EndHitNo Result	
	int z = S.find('\x1');
	if ( (z == -1) || (z == 0) ) 
	{
		return neProtocolError;
	};

	Result = S.substr(z+1);
	S.erase(z, S.length() - z);
	int iInternalError;
	if(sscanf(S.c_str(), "%i", &iInternalError) != 1)
	{
		return neProtocolError;
	};
	InternalError = (DDCErrorEnum)iInternalError;

	return neSuccess;
};


void CDDCServerListenHost::GetFirstHitsFromCorpora(const DwordVector& PossibleHosts,
													   const string Query,  
													   const DWORD ResultLimit, 
													   const int TimeOut,
													   CFirstHitsQueryResult& Result)
{
	Result.m_Sockets.resize(PossibleHosts.size(),-1);
	Result.m_AllHitsCount = 0;
	Result.m_AllRelevantDocsCount = 0;
	
	for (int i=0; i<PossibleHosts.size(); i++)	
	{
		BYTE HostNo = PossibleHosts[i];
		CHost& Host = m_Hosts[HostNo];
		string strError;
		Result.m_Sockets[i] = Host.CreateAndConnectSocket(strError);
		if (Result.m_Sockets[i] == -1)
		{
			Result.m_NetworkError = neCouldNotOpenSocket;
			return;
		};
		string S = Format ("get_first_hits %s\x1%i %i", Query.c_str(), TimeOut,  ResultLimit);
		string ErrorStr;
		if (!SendString(Result.m_Sockets[i], S.c_str(), S.length(), ErrorStr)) 
		{
			ErrorMessage (Format("cannot send a string to socket %s : %s", Host.GetAddressStr().c_str(), ErrorStr.c_str() ));
			Result.m_NetworkError = neCouldNotSendData;
			return;
		};
	};

	vector<size_t>  HitsDistribution(m_Hosts.size(), 0);
	vector<size_t>  RelevantDocsDistribution(m_Hosts.size(), 0);
	for (int i=0; i<PossibleHosts.size(); i++)	
	{
		BYTE HostNo = PossibleHosts[i];
		CHost& Host = m_Hosts[HostNo];

		string S;
		Result.m_NetworkError = RecieveString(Result.m_Sockets[i], S, TimeOut);
		if (Result.m_NetworkError != neSuccess)
		{
			concord_daemon_log ( " Cannot recieve a resulting string from a client  " );
			return;
		};

		DWORD  CorpusHitsCount,RelevantDocumentsCount, EndHitNo;
		int iInternalError, iSort;

		if(sscanf(S.c_str(), "%i %i %i %i %i", &iInternalError, &EndHitNo, &CorpusHitsCount, &RelevantDocumentsCount, &iSort) != 5)
		{
			Result.m_NetworkError = neProtocolError;
			return;
		};
		Result.m_InternalError = (DDCErrorEnum)iInternalError;
		Result.m_AllHitsCount += CorpusHitsCount;
		HitsDistribution[HostNo] = CorpusHitsCount;

		Result.m_AllRelevantDocsCount += RelevantDocumentsCount;
		RelevantDocsDistribution[HostNo] = RelevantDocumentsCount;

		Result.m_bSort = (iSort!=0);
		// reading order ids, whihc would be used to sort results
		int f = S.find(";");
		if (f != string::npos)
		{
			StringTokenizer tok(S.c_str() + f +1,";");
			while (tok())
			{
				CHitToSort HS;
				if(sscanf(tok.val(), "%i %i", &HS.m_HitNo, &HS.m_OrderId) != 2)
				{
					Result.m_NetworkError = neProtocolError;
					return;
				};
				HS.m_HostNo = HostNo;
				Result.m_Hits.push_back(HS);
			};
		};
	};

	// collecting hits distribution in the right order for all possible corpus threads
	// the previous cycle was only for "possible" host (=user-defined corpus threads)
	Result.m_HitsDistributionStr = "Corpora Distribution:";
	Result.m_RelevantDocsDistributionStr = "Relevant Documents Distribution:";
	for (size_t i=0; i< HitsDistribution.size();  i++)
	{
		Result.m_HitsDistributionStr += Format("%i",HitsDistribution[i]);
		Result.m_RelevantDocsDistributionStr += Format("%i",RelevantDocsDistribution[i]);

		if (i+1 == HitsDistribution.size())
		{
			Result.m_HitsDistributionStr += Format("=%i;\n",Result.m_AllHitsCount);
			Result.m_RelevantDocsDistributionStr += Format("=%i;\n",Result.m_AllRelevantDocsCount);
		}
		else
		{
			Result.m_HitsDistributionStr += "+";
			Result.m_RelevantDocsDistributionStr += "+";
		};
	};
	
	Result.m_NetworkError = neSuccess;
};


NetworkErrorsEnum CDDCServerListenHost::RunDistributed(string Query, const string& ResultType, DWORD StartHitNo, 
											DWORD ResultLimit, int& InternalError, DWORD& EndHitNo, DWORD& HitsCount, 
											int TimeOut, string& ResultString,
											DWORD& RelevantDocsCount)
{
	DwordVector PossibleHosts;
	string CleanQuery;
	//  reading possible hosts, using query where these hosts can be specified
	// PossibleHosts is an index of m_Hosts
	if (!ReadPossibleHosts(Query, CleanQuery, PossibleHosts))
		return neSubcorpusNotFound;

	EndHitNo = 0;
	HitsCount = 0;
	ResultString = "";

	// 
	time_t start_get_hits, end_get_hits; 
	if  (TimeOut != -1)
		time(&start_get_hits);
	

	CFirstHitsQueryResult FHResult;
	// reading first StartHiNo+ResultLimit  hits from each host and storing them to Hits

	GetFirstHitsFromCorpora(PossibleHosts, CleanQuery, StartHitNo+ResultLimit, TimeOut, FHResult );
	HitsCount = FHResult.m_AllHitsCount;
	RelevantDocsCount = FHResult.m_AllRelevantDocsCount;
	InternalError = FHResult.m_InternalError;
	if (FHResult.HasErrors())
	{
		FHResult.CloseSockets();
		return FHResult.m_NetworkError;
	}

	if  (TimeOut != -1)
	{
		time(&end_get_hits);
		TimeOut -= end_get_hits-start_get_hits;
		if (TimeOut < 0)
		{
			// if the provious steps took all time we add 10 seconds in order to 
			// recieve at least some results 
			TimeOut = 10;
		};
	};

	// sorting and trimming
	if (FHResult.m_bSort)
		sort (FHResult.m_Hits.begin(), FHResult.m_Hits.end());

	if (StartHitNo+ResultLimit < FHResult.m_Hits.size())
		FHResult.m_Hits.erase(FHResult.m_Hits.begin()+StartHitNo+ResultLimit, FHResult.m_Hits.end());

	if (StartHitNo < FHResult.m_Hits.size())
		FHResult.m_Hits.erase(FHResult.m_Hits.begin(), FHResult.m_Hits.begin()+StartHitNo);
	else
		FHResult.m_Hits.clear();


	//  creating map from HostNo to  Hits, SortedHits[i] contains all reference to hits from host i
	// in the right order
	vector<int> SortedHits[MaxSubhostsSize];
	for (size_t i=0; i < FHResult.m_Hits.size(); i++)
		SortedHits[FHResult.m_Hits[i].m_HostNo].push_back(i);


	//  getting strings from the hosts:
	// Results[HostNo] - contains all hit strings in the right order
	vector<string> Results[MaxSubhostsSize];
	for (int i=0; i<PossibleHosts.size(); i++)	
	{
		BYTE HostNo = PossibleHosts[i]; 

		const vector<int>&  V = SortedHits[HostNo];
		string S;
		int FirstHitNo = (V.empty()) ? 0 : FHResult.m_Hits[V[0]].m_HitNo;
		NetworkErrorsEnum Res = GetHitStringsFromOneCorpora
					(m_Hosts[HostNo], ResultType, FirstHitNo, V.size(), TimeOut,FHResult.m_Sockets[i], InternalError, S);


		if (FHResult.HasErrors())
		{
			FHResult.CloseSockets();
			return FHResult.m_NetworkError;
		}


		StringTokenizer tok(S.c_str(), AdditionalHitDelimeter);
		while (tok())
			Results[HostNo ].push_back(tok.val());

		if (Results[HostNo].size() != V.size())
		{
			FHResult.CloseSockets();
			return neProtocolError;
		};
	};

	// Creating m_QueryResultStr
	for (size_t i=0; i < FHResult.m_Hits.size(); i++)
	{
		BYTE HostNo = FHResult.m_Hits[i].m_HostNo;
		const vector<int>&  V = SortedHits[HostNo];
		if (V.empty()) continue;
		ResultString += Results[ HostNo ][ FHResult.m_Hits[i].m_HitNo - FHResult.m_Hits[V[0]].m_HitNo];
	};

	CConcHolder::DecorateQueryResults(ResultType, ResultString);

	if (CConcHolder::GetResultFormatByString(ResultType) == CConcHolder::DDC_ResultText)
	{
		// we always write relevant document numbers (RelevantDocsDistributionStr) to output, 
		// even though it is not specified in the option file of the coprus threads.
		// The problem is, that daemon doesn't know anything about options files of their coprus threads.
		ResultString = FHResult.m_HitsDistributionStr + FHResult.m_RelevantDocsDistributionStr+ResultString;
	};

	EndHitNo = StartHitNo + FHResult.m_Hits.size();

	return neSuccess;	
};



//=====================================================================
//===========================   global functions  =====================
//=====================================================================


void LoadLocalCorpora (ProtocolDensityEnum ProtocolDensity, string ConfigFile)
{
	string Error;
	if (!IsRmlRegistered(Error)) 
	{
		concord_daemon_log ( Error  );
		return;
	};
	string IniPath = GetIniFilePath();

	{
		int ii = ConfigFile.find("$RML");
		if (ii != string::npos)
				ConfigFile.replace(ii, 4, IniPath);
	};

	concord_daemon_log ( Format ("Start loading corpora from %s", ConfigFile.c_str()) );
	vector<CHost> Hosts;
	LoadHosts(ConfigFile, Hosts);


	concord_daemon_log ( Format("  Found %i Hosts ", Hosts.size()) );
	LocalCorpora.clear();
	
	for (int i = 0; i<Hosts.size(); i++)
	{

		CDDCCorpusListenHost H(true, ProtocolDensity);
		H.CopyAddressParametersFrom(Hosts[i]);
		H.m_pHolder = new CConcHolder;
		H.m_LogFunction = concord_daemon_log;;

		H.m_pHolder->m_pIndexator = new CConcIndexator;
		concord_daemon_log ( Format ("Loading corpora index for %s", H.GetLocalPathOfIndex().c_str()) );
		if (!H.m_pHolder->m_pIndexator->LoadProject(H.GetLocalPathOfIndex())) 
		{
			concord_daemon_log ( Format("Error! Cannot load project %s! ", H.GetLocalPathOfIndex().c_str()) );
			return;
		};
		concord_daemon_log ( "  init graphan " );
		H.m_pHolder->m_AdditionalHitDelimeter = AdditionalHitDelimeter;
		concord_daemon_log ( "  ok " );
			
		LocalCorpora.push_back(H);
	};

	
	try {

		for (int i = 0; i<LocalCorpora.size(); i++)
		{
			CDDCCorpusListenHost& H = LocalCorpora[i];
			LocalCorpora[i].CreateListener();
		};
	}
	catch (...)
	{
		concord_daemon_log ( "Error! An exception occured while creating listeners! ");
		return;
	};
};

void LoadDDCServer (ProtocolDensityEnum ProtocolDensity, string ServerConfigFile)
{
	string Error;
	if (!IsRmlRegistered(Error)) 
	{
		concord_daemon_log ( Error  );
		return;
	};
	string IniPath = GetIniFilePath();
	{
		int ii = ServerConfigFile.find("$RML");
		if (ii != string::npos)
				ServerConfigFile.replace(ii, 4, IniPath);
	};

	if (pServer)
		delete pServer;
	pServer = new CDDCServerListenHost(false, ProtocolDensity);
	if (!pServer)
	{
		concord_daemon_log ( "Error! cannot allocate memory for a server instance");
		return;
	};
	
	LoadHosts(ServerConfigFile, pServer->m_Hosts);
	if (pServer->m_Hosts.empty())
	{
		delete pServer;
		pServer = 0;
		return;
	};
	bool bServerIsFound = false;	
	for (int i = 0; i<pServer->m_Hosts.size(); i++)
	if (pServer->m_Hosts[i].m_CorporaName == "server")
	{
		pServer->CopyAddressParametersFrom(pServer->m_Hosts[i]);
		pServer->m_LogFunction = concord_daemon_log;;
		pServer->CreateListener();
		pServer->m_Hosts.erase(pServer->m_Hosts.begin() + i);
		bServerIsFound = true;
		break;
	}

	if (!bServerIsFound)
	{
		concord_daemon_log ( Format("Error! there is no server socket description in %s!", ServerConfigFile.c_str()));
		return;

	};
	if (pServer->m_Hosts.empty())
	{
		concord_daemon_log ( Format("Error! there is no ouptut server  sockets in %s!", ServerConfigFile.c_str()));
		return;

	};

};


void UnloadLocalCorporaAndServer ()
{
	for (int i =0 ;  i<LocalCorpora.size(); i++)
	{
		delete LocalCorpora[i].m_pHolder->m_pIndexator;
		delete LocalCorpora[i].m_pHolder;
	};

	LocalCorpora.clear();

	// stop server
	delete pServer;
	pServer = 0;
};



