// QNode.cpp: implementation of the CQNode class.
//
//////////////////////////////////////////////////////////////////////

#include "StdConc.h"
#include "QueryNode.h"
#include "../common/DwdsThesaurus.h"
#include "../common/util_classes.h"

extern CDwdsThesaurus* pDwdsThesaurus;

#ifdef DETECT_MEMORY_LEAK
	#ifdef _DEBUG
	#define new DEBUG_NEW
	#undef THIS_FILE
	static char THIS_FILE[] = __FILE__;
	#endif
#endif

//#pragma intrinsic(memcpy, memcmp, memset, strlen,strcpy)


const size_t MaxDistanceForNear = 32;
#define GetOccurrencesSize() m_Occurrences.size()  

//======================== CQueryNode =====================

CQueryNode::CQueryNode(bool bUseNodeIndices)
{
	m_bAtomic = false;
	m_bNegated = false;
	m_bUseNodeIndices = bUseNodeIndices;

};

CQueryNode::~CQueryNode()
{
	ClearVector(m_Occurrences);
	ClearVector(m_OccurrenceNodeIndices);
};

size_t CQueryNode::GetNodeFrequenceByNodeIndex(size_t NodeIndex) const
{
	assert (false);
	return 0;
}

/*
	EvaluateWithoutHits is the first stage of  the evaluation of one query.
	The main task it to build CQueryNode::m_Occurrences, which is written by chunks,
	sorted by the first item of a chunk. 

*/
void	CQueryNode::EvaluateWithoutHits() 
{
};	

void	CQueryNode::Evaluate() 
{
	throw  CExpc ("not implemented!");
};	



void	CQueryNode::SetNegation(bool Value)
{
	m_bNegated = Value;
}
bool	CQueryNode::GetNegation() const
{
	return m_bNegated;
};


void	CQueryNode::AddOccurrences(const CQueryNode& FromNode, int start, int end)
{
	m_Occurrences.insert(m_Occurrences.end(), FromNode.m_Occurrences.begin() + start,  FromNode.m_Occurrences.begin() + end);
	if (m_bUseNodeIndices)
		m_OccurrenceNodeIndices.insert(m_OccurrenceNodeIndices.end(), FromNode.m_OccurrenceNodeIndices.begin() + start,  FromNode.m_OccurrenceNodeIndices.begin() + end);
		
}

void	CQueryNode::ClearAndReserveOccurrences(int size)
{
	m_Occurrences.clear();
	m_Occurrences.reserve(size);
	m_OccurrenceNodeIndices.clear();
}
void	CQueryNode::SwapOccurrences(CQueryNode& Node)
{
	Node.m_Occurrences.swap(m_Occurrences);
	Node.m_OccurrenceNodeIndices.swap(m_OccurrenceNodeIndices);
	
}

void CQueryNode::SetHolder(const CConcHolder*	pHolder)
{
	m_pHolder = pHolder;
	m_bUseNodeIndices = m_pHolder->HasRankOrderOperator();
	//m_bUseNodeIndices = false;
}

void CQueryNode::ConvertOccurrencesToHits (bool bUniqueHits)
{
	const vector<CTokenNo>& Breaks = m_pHolder->GetBreaks();
	int TextAreasCount = m_pHolder->m_pIndexator->m_Bibl.GetTextAreasCount();
	int TextAreaNo = m_pHolder->GetTextArea();
	bool bTextAreaNo =  TextAreaNo != UnknownTextAreaNo;
	m_Hits.clear();
	m_Hits.reserve(GetOccurrencesSize());

	int cnt = GetOccurrencesSize();

	bool bLongSequence = GetOccurrencesSize()*5 > Breaks.size() ;
	vector<CTokenNo>::const_iterator break_it = Breaks.begin();

	CQueryNode NewNode (m_bUseNodeIndices);
	if (bTextAreaNo)
		NewNode.ClearAndReserveOccurrences(GetOccurrencesSize());


	for (int i = 0;  i < cnt; i++)
	{
		CTokenNo Occurrence = m_Occurrences[i];

		if (!bLongSequence)
		{
			break_it = lower_bound (break_it, Breaks.end(), Occurrence);
			assert (break_it != Breaks.end());
			if (*break_it == m_Occurrences[i])
				break_it++;
		}
		else
		{
			// optimization for the very long sequence of m_Occurrences, lower_bound is too slow
			while(m_Occurrences[i] >= *break_it) break_it++;
		};

		
		DWORD BreakNo = break_it - Breaks.begin();
		if (bTextAreaNo)
			if ( BreakNo%TextAreasCount != TextAreaNo ) // if it is the right text area
				continue;
			else
				NewNode.AddOccurrences(*this, i, i+1);
				


		if (	m_Hits.empty() 
			|| ( m_Hits.back().m_BreakNo != BreakNo)
			|| !bUniqueHits
		   )
   		{
			CHit H(BreakNo);
			H.m_BreakNo = BreakNo;
			H.m_HighlightOccurrenceEnd = i+1;
			m_Hits.push_back(H);
		}
		else
			m_Hits.back().m_HighlightOccurrenceEnd = i+1;

	};

	if (bTextAreaNo)
		SwapOccurrences(NewNode);
		
};



//void CQueryNode::ConvertOccurrencesToHitsForPatterns (bool bUniqueHits)	
//{
//	const vector<CTokenNo>& Breaks = m_pHolder->GetBreaks();
//	m_Hits.clear();
//	int TextAreasCount = m_pHolder->m_pIndexator->m_Bibl.GetTextAreasCount();
//	int TextAreaNo = m_pHolder->GetTextArea();
//	bool bTextAreaNo =  TextAreaNo != UnknownTextAreaNo;
//
//	m_Hits.reserve(m_ChunkLengths.size());
//
//	int cnt = GetOccurrencesSize();
//
//	CQueryNode NewNode(m_bUseNodeIndices);
//	NewNode.ClearAndReserveOccurrences(cnt);
//	NewNode.m_ChunkLengths.reserve(m_ChunkLengths.size());
//	size_t NewOccurrencesCount = 0;
//
//	vector<CTokenNo>::const_iterator break_it = Breaks.begin();
//	bool bLongSequence = m_ChunkLengths.size()*5 > Breaks.size() ;
//	
//	size_t ChunkNo = 0;
//	for (int i = 0;  i < cnt; i += m_ChunkLengths[ChunkNo], ChunkNo++)
//	{
//		size_t ItemsCount = m_ChunkLengths[ChunkNo];
//
//		if (!bLongSequence)
//		{
//			break_it = lower_bound (break_it, Breaks.end(), m_Occurrences[i]);
//			assert (break_it != Breaks.end());
//			if (*break_it == m_Occurrences[i]) 
//				break_it++; // then Break is the first position of the next break
//		}
//		else
//		{
//			while(m_Occurrences[i] >= *break_it) break_it++;
//		}
//
//		if (m_Occurrences[i + ItemsCount -1] >= *break_it )
//		{
//			 //this period of m_Occurrences should be ignored since 
//			 // it belongs two or more breaks simultaneously
//			continue;
//		};
//
//		DWORD BreakNo = break_it - Breaks.begin();
//		if (bTextAreaNo)
//			if ( BreakNo%TextAreasCount != TextAreaNo )
//				continue;
//		
//		NewNode.m_ChunkLengths.push_back(ItemsCount);
//		NewOccurrencesCount += i + ItemsCount;
//		NewNode.AddOccurrences(*this, i,  i + ItemsCount);
//
//		if (	m_Hits.empty() 
//			|| ( m_Hits.back().m_BreakNo != BreakNo)
//			|| !bUniqueHits
//		   )
//   		{
//			CHit H(BreakNo);
//			H.m_BreakNo = BreakNo;
//			H.m_HighlightOccurrenceEnd = NewNode.m_Occurrences.size();
//			m_Hits.push_back(H);
//		}
//		else
//		{
//			m_Hits.back().m_HighlightOccurrenceEnd += ItemsCount;
//			assert (m_Hits.back().m_HighlightOccurrenceEnd == NewNode.m_Occurrences.size()); 
//		};
//		
//	};
//	SwapOccurrences(NewNode);
//	NewNode.m_ChunkLengths.swap(m_ChunkLengths);
//};


void CQueryNode::ConvertOccurrencesToHitsForPatterns (bool bUniqueHits)	
{
	const vector<CTokenNo>& Breaks = m_pHolder->GetBreaks();
	m_Hits.clear();
	int TextAreasCount = m_pHolder->m_pIndexator->m_Bibl.GetTextAreasCount();
	int TextAreaNo = m_pHolder->GetTextArea();
	bool bTextAreaNo =  TextAreaNo != UnknownTextAreaNo;

	m_Hits.reserve(m_ChunkLengths.size());

	int cnt = GetOccurrencesSize();

	CQueryNode NewNode(m_bUseNodeIndices);
	NewNode.ClearAndReserveOccurrences(cnt);
	NewNode.m_ChunkLengths.reserve(m_ChunkLengths.size());
	size_t InputStarter = 0;
	size_t NewOccurrencesCount = 0;

	vector<CTokenNo>::const_iterator break_it = Breaks.begin();
	bool bLongSequence = m_ChunkLengths.size()*5 > Breaks.size() ;
	
	size_t ChunkNo = 0;
	int i = 0;
	for (;  i < cnt; i += m_ChunkLengths[ChunkNo], ChunkNo++)
	{
		size_t ItemsCount = m_ChunkLengths[ChunkNo];

		if (!bLongSequence)
		{
			break_it = lower_bound (break_it, Breaks.end(), m_Occurrences[i]);
			assert (break_it != Breaks.end());
			if (*break_it == m_Occurrences[i]) 
				break_it++; // then Break is the first position of the next break
		}
		else
		{
			while(m_Occurrences[i] >= *break_it) break_it++;
		}

		if (m_Occurrences[i + ItemsCount - 1] >= *break_it )
		{
			if ( i > InputStarter)
				NewNode.AddOccurrences(*this, InputStarter,  i);
			InputStarter = i + ItemsCount;
			 //this period of m_Occurrences should be ignored since 
			 // it belongs two or more breaks simultaneously
			continue;
		};

		DWORD BreakNo = break_it - Breaks.begin();
		if (bTextAreaNo)
			if ( BreakNo%TextAreasCount != TextAreaNo )
			{
				if ( i > InputStarter)
					NewNode.AddOccurrences(*this, InputStarter,  i);
				InputStarter = i + ItemsCount;

				continue;
			}
		
		NewNode.m_ChunkLengths.push_back(ItemsCount);
		NewOccurrencesCount += ItemsCount;
		

		if (	m_Hits.empty() 
			|| ( m_Hits.back().m_BreakNo != BreakNo)
			|| !bUniqueHits
		   )
   		{
			CHit H(BreakNo);
			H.m_BreakNo = BreakNo;
			H.m_HighlightOccurrenceEnd = NewOccurrencesCount;
			m_Hits.push_back(H);
		}
		else
		{
			m_Hits.back().m_HighlightOccurrenceEnd += ItemsCount;
			assert (m_Hits.back().m_HighlightOccurrenceEnd == NewOccurrencesCount); 
		};
		
	};
	if ( i > InputStarter)
		NewNode.AddOccurrences(*this, InputStarter,  i);

	SwapOccurrences(NewNode);
	NewNode.m_ChunkLengths.swap(m_ChunkLengths);
};




// ================== CQueryBinaryOperationNode ==================

CQueryBinaryOperationNode::CQueryBinaryOperationNode():CQueryNode(false)
{
	m_pChild1 = 0;
	m_pChild2 = 0;
};

CQueryBinaryOperationNode::~CQueryBinaryOperationNode()
{
	if (m_pChild1)
		delete m_pChild1;
	if (m_pChild2)
		delete m_pChild2;
}


void CQueryBinaryOperationNode::Create(const CConcHolder*	pHolder, CQueryNode* child1, CQueryNode* child2, string Operation)
{
	SetHolder(pHolder);
	m_pChild1 = child1;
	m_pChild2 = child2;
	m_Source = child1->m_Source + Operation + child2->m_Source;
}

size_t CQueryBinaryOperationNode::GetNodeFrequenceByNodeIndex(size_t NodeIndex) const
{
	size_t Result = 0;
	if (m_pChild1) Result += m_pChild1->GetNodeFrequenceByNodeIndex(NodeIndex);
	if (m_pChild2) Result += m_pChild2->GetNodeFrequenceByNodeIndex(NodeIndex);
	return Result;
}




// ================== CQueryAndOperation ==================
CQueryAndOperation::CQueryAndOperation()
{
};

void CQueryBinaryOperationNode::hits_add(const CQueryNode& NodeFrom, vector<CHit>::const_iterator _First) 
{
	
	size_t prev_end =  0;
	if (_First !=  NodeFrom.m_Hits.begin() )
		prev_end = (_First-1)->m_HighlightOccurrenceEnd;

	vector<CHit>::const_iterator _Last = NodeFrom.m_Hits.end();

	for (; _First != _Last; _First++)
	{
		{
			size_t end = _First->m_HighlightOccurrenceEnd;
			AddOccurrences(NodeFrom, prev_end, end);
			prev_end = end;
		}
		{
			CHit  H(_First->m_BreakNo);
			H.m_HighlightOccurrenceEnd = GetOccurrencesSize();
			m_Hits.push_back(H);
		}
	};
};

void CQueryBinaryOperationNode::hits_and_positions_union() 
{	
	vector<CHit>::const_iterator _First1 = m_pChild1->m_Hits.begin();
	vector<CHit>::const_iterator _Last1 = m_pChild1->m_Hits.end();
	vector<CHit>::const_iterator _First2 = m_pChild2->m_Hits.begin();
	vector<CHit>::const_iterator _Last2 = m_pChild2->m_Hits.end();
	m_Hits.clear();
	m_Hits.reserve(m_pChild1->m_Hits.size() + m_pChild2->m_Hits.size());
	ClearAndReserveOccurrences(m_pChild1->GetOccurrencesSize() + m_pChild2->GetOccurrencesSize());
	

	size_t  prev_end1 = 0;
	size_t  prev_end2 = 0;
	for (; _First1 != _Last1 && _First2 != _Last2; )
		if (_First1->m_BreakNo < _First2->m_BreakNo)
		{
			
			AddOccurrences(*m_pChild1, prev_end1, _First1->m_HighlightOccurrenceEnd);
			prev_end1 = _First1->m_HighlightOccurrenceEnd;

			{
				CHit  H(_First1->m_BreakNo);
				H.m_HighlightOccurrenceEnd = GetOccurrencesSize();
				m_Hits.push_back(H);
			}
			_First1++;
		}
		else 
		if (_First2->m_BreakNo < _First1->m_BreakNo)
		{
			AddOccurrences (*m_pChild2, prev_end2,  _First2->m_HighlightOccurrenceEnd);
			prev_end2 = _First2->m_HighlightOccurrenceEnd;
			{
				CHit  H(_First2->m_BreakNo);
				H.m_HighlightOccurrenceEnd = GetOccurrencesSize();
				m_Hits.push_back(H);
			}
			_First2++;
		}
		else
		{
			{	// inserting highlight occurrences from the first and the second child
				AddOccurrences(*m_pChild1, prev_end1, _First1->m_HighlightOccurrenceEnd);
				AddOccurrences(*m_pChild2, prev_end2, _First2->m_HighlightOccurrenceEnd);
				{
					CHit  H(_First1->m_BreakNo);
					H.m_HighlightOccurrenceEnd = GetOccurrencesSize();
					m_Hits.push_back(H);
				}
				prev_end1 = _First1->m_HighlightOccurrenceEnd;
				prev_end2 = _First2->m_HighlightOccurrenceEnd;
			}

			_First1++;
			_First2++;
		};

	hits_add(*m_pChild1, _First1);
	hits_add(*m_pChild2, _First2);
}


void CQueryBinaryOperationNode::hits_and_positions_intersection() 
{	
	vector<CHit>::const_iterator _First1 = m_pChild1->m_Hits.begin();
	vector<CHit>::const_iterator _Last1 = m_pChild1->m_Hits.end();
	vector<CHit>::const_iterator _First2 = m_pChild2->m_Hits.begin();
	vector<CHit>::const_iterator _Last2 = m_pChild2->m_Hits.end();
	m_Hits.clear();
	m_Hits.reserve(min(m_pChild1->m_Hits.size(),m_pChild2->m_Hits.size()) );
	ClearAndReserveOccurrences( min(m_pChild1->GetOccurrencesSize(),m_pChild2->GetOccurrencesSize()));

	size_t  prev_end1 = 0;
	size_t  prev_end2 = 0;
	for (; _First1 != _Last1 && _First2 != _Last2; )
		if (_First1->m_BreakNo < _First2->m_BreakNo)
		{
			prev_end1 = _First1->m_HighlightOccurrenceEnd;
			_First1++;
		}
		else 
		if (_First2->m_BreakNo < _First1->m_BreakNo)
		{
			prev_end2 = _First2->m_HighlightOccurrenceEnd;
			_First2++;
			
		}
		else
		{

			{	// inserting highlight occurrences from the first and the second child
				AddOccurrences(*m_pChild1, prev_end1, _First1->m_HighlightOccurrenceEnd);
				AddOccurrences(*m_pChild2, prev_end2, _First2->m_HighlightOccurrenceEnd);
				{
					CHit  H (_First1->m_BreakNo);
					H.m_HighlightOccurrenceEnd = GetOccurrencesSize();
					m_Hits.push_back(H);
				}
				prev_end1 = _First1->m_HighlightOccurrenceEnd;
				prev_end2 = _First2->m_HighlightOccurrenceEnd;
			}


			_First1++;
			_First2++;
		};

}

void CQueryBinaryOperationNode::hits_and_positions_difference() 
{	
	vector<CHit>::const_iterator _First1 = m_pChild1->m_Hits.begin();
	vector<CHit>::const_iterator _Last1 = m_pChild1->m_Hits.end();
	vector<CHit>::const_iterator _First2 = m_pChild2->m_Hits.begin();
	vector<CHit>::const_iterator _Last2 = m_pChild2->m_Hits.end();
	m_Hits.clear();
	m_Hits.reserve(min(m_pChild1->m_Hits.size(),m_pChild2->m_Hits.size()) );
	ClearAndReserveOccurrences( min(m_pChild1->GetOccurrencesSize(), m_pChild2->GetOccurrencesSize()) );

	size_t  prev_end1 = 0;
	for (; _First1 != _Last1 && _First2 != _Last2; )
		if (_First1->m_BreakNo < _First2->m_BreakNo)
		{
			AddOccurrences(*m_pChild1, prev_end1, _First1->m_HighlightOccurrenceEnd);
			prev_end1 = _First1->m_HighlightOccurrenceEnd;

			{
				CHit  H (_First1->m_BreakNo);
				H.m_HighlightOccurrenceEnd = GetOccurrencesSize();
				m_Hits.push_back(H);
			}
			_First1++;
		}
		else 
		if (_First2->m_BreakNo < _First1->m_BreakNo)
		{
			_First2++;
		}
		else
		{
			prev_end1 = _First1->m_HighlightOccurrenceEnd;
			_First1++;
			_First2++;
		};

	hits_add(*m_pChild1, _First1);
}


void CQueryAndOperation::Evaluate()
{
	m_pChild1->Evaluate();
	m_pChild2->Evaluate();

	

	if (!m_pChild1->GetNegation() && !m_pChild2->GetNegation())
	{
		hits_and_positions_intersection ();
	}
	else
	if (m_pChild1->GetNegation() && m_pChild2->GetNegation())
	{
		// !a && !b == !(a||b)
		hits_and_positions_union ();
		SetNegation(true);
	}
	else
	if (!m_pChild1->GetNegation() && m_pChild2->GetNegation())
	{
		hits_and_positions_difference ();
	}
	else
	{
		hits_and_positions_difference ();
	}
	
};

//=============   CQueryOrOperation =====================
CQueryOrOperation::CQueryOrOperation()
{
};

void CQueryOrOperation::Evaluate()
{
	m_pChild1->Evaluate();
	m_pChild2->Evaluate();

	
	

	if (!m_pChild1->GetNegation() && !m_pChild2->GetNegation())
	{
		hits_and_positions_union ();
	}
	else
	if (m_pChild1->GetNegation() && m_pChild2->GetNegation())
	{
		// !a || !b ==> !(a&&b)
		hits_and_positions_intersection ();
		SetNegation(true);
	}
	else
	if (m_pChild1->GetNegation() != m_pChild2->GetNegation())
	{
		assert (false);
	}

	
};


// ================ CQueryTokenNode ==================

CQueryTokenNode::CQueryTokenNode(int NodeIndex):CQueryNode(false)
{
	m_bAtomic = true;
	m_bChunk = false;
	m_NodeIndex = NodeIndex;
};

CQueryTokenNode::~CQueryTokenNode()
{
};


bool CQueryTokenNode::BuildRegExp(string RegExpStr, vector<DWORD>& IndexItems)
{
//	RML_RE re(RegExpStr, pcrecpp::UTF8()); // if the input texts are in UTF-8
	RML_RE re(RegExpStr, m_pHolder->m_pIndexator->GetRegExpTables());
	
	const CStringIndexSet*  IndexSet =  m_pHolder->m_pIndexator->GetIndexByName(m_IndexName);
	if (!IndexSet) return false;
	IndexSet->QueryTokenListUsingRegExp(re, IndexItems);
	return true;
};

bool CQueryTokenNode::CreateMorphAnnotationPattern(const CConcHolder* pHolder,const char* src, const char* IndexName)
{
	SetHolder(pHolder);

    string s = src;


	string RegExpStr;
	
	{
		StringTokenizer tok(s.c_str(), ", ;");
		vector<string> Items;
		while (tok())
		{
			Items.push_back(tok.val());
		};
		sort(Items.begin(), Items.end());
		RegExpStr = m_pHolder->m_pIndexator->GetIndexItemSetByVectorString(Items, true);
	};

	m_Source = src;
	m_IndexName = IndexName;
		
	m_IndexItems.clear();
	return BuildRegExp(RegExpStr, m_IndexItems);
}

bool CQueryTokenNode::CreateThesPattern(const CConcHolder* pHolder, const char* src)
{
	SetHolder(pHolder);
	m_Source = src;
    string s = src;
	RmlMakeUpper(s, pHolder->m_pIndexator->m_Language);	
	m_IndexName = "Thes";
	m_IndexItems.clear();
	return BuildRegExp(":"+s+":", m_IndexItems);
}

bool	CQueryTokenNode::CreateChunkPattern(const CConcHolder* pHolder, const char* ChunkTypeStr)
{
	m_bChunk = true;
	m_bAtomic = true;
	SetHolder(pHolder);
	m_Source = ChunkTypeStr;
	if (!pHolder->m_pIndexator->m_pChunkIndex) return false;
	m_IndexName = ChunkIndexName;
	return BuildRegExp(m_Source+",", m_IndexItems);
};

size_t CQueryTokenNode::GetNodeFrequenceByNodeIndex(size_t NodeIndex) const
{
	if (m_NodeIndex == NodeIndex)
		return m_Occurrences.size();
	else
		return 0;
}


void GetWordForms (const MorphLanguageEnum Langua, const string& src, set<string>& WordForms)
{
	if ( src.empty() ) return;
	WordForms.insert(src);
	if (Langua == morphUnknown)  return;

	vector<CFormInfo> Paradigms;
	const CLemmatizer* pLemmatizer = GetLemmatizerByLanguage(Langua);
	if (!pLemmatizer) return;
	const bool bCapital = is_upper_alpha((BYTE)src[0],Langua);

	try
	{
		string h = src;
		pLemmatizer->CreateParadigmCollection(false, h, bCapital, Paradigms);
	}
	catch(...)
	{
		throw CExpc (Format("Morphology has crushed on %s ", src.c_str()), errProcessMorphology);
	}

	//	print out all paradigms
	for (int i = 0; i < Paradigms.size(); i++)
	{
		const CFormInfo& F = Paradigms[i];
		//  print all forms of one paradigm
		for (int j = 0; j < F.GetCount(); j++)
		{
			string form  = F.GetWordForm(j);
			RmlMakeLower(form, Langua);
			// set register of the first char of "form" equal to the  register of "src"
			if (bCapital)
			{
				//  making the first word upper
				form[0] = ReverseChar((BYTE)form[0], Langua);
			};
			WordForms.insert(form);
		}
	}
};	

// for each string "aab" adds also "Aab" and "AAB"
// for each string "Aab" adds also "aab" and "AAB"
void AddFormsWithDifferentRegisters (const MorphLanguageEnum Langua, set<string>& WordForms)
{
	set<string> NewWordForms;
	for (set<string>::const_iterator it = WordForms.begin(); it != WordForms.end(); it++)
	{
		const string& q = *it;
		if (q.empty())  continue;
		NewWordForms.insert(q);

		if (is_lower_alpha((BYTE)q[0],  Langua))
		{
			string upp = q;
			upp[0] = ReverseChar((BYTE)upp[0], Langua);
			NewWordForms.insert(upp);
		}
		else
		{
			string lwr = q;
			RmlMakeLower(lwr, Langua);
			NewWordForms.insert(lwr);

		};
		string upr = q;
		RmlMakeUpper(upr, Langua);
		NewWordForms.insert(upr);
	};

	NewWordForms.swap(WordForms);
};


bool	CQueryTokenNode::CreateTokenPattern(const CConcHolder* pHolder, const char* src, bool bRegularExpession)
{
	string str = src;
	if (		(str.length() >= 2)
			&&	(str[0] == '\\')
			&&	ispunct((BYTE)str[1])
		)
	{
		str = str.erase(0, 1);
	};

	SetHolder(pHolder);
	m_Source = str;
	m_IndexName = "Token";


	bool bExact = false;;
	if	(		!str.empty()
			&& (str[0] == '@')
		)
	{
		bExact = true;
		str.erase(0,1);
		m_Source.erase(0,1);
	};
			

	m_IndexItems.clear();

	if	(		(str.size() > 1)
			&&	(str[0] == '%')
		)
	{
		string q = '^'+ str.substr(1) + '@';
		RmlMakeLower(q, pHolder->m_pIndexator->m_Language);
		m_IndexName = "MorphAnnot";
		return BuildRegExp(q, m_IndexItems);
	}
	else
	if (bRegularExpession)
	{
		if (str.length() < 3)  return  false;
		if (str[0] != '/')return  false;
		if (str[str.length()-1] != '/')return  false;
		str = str.substr(1, str.length() - 2);

		return BuildRegExp(str, m_IndexItems);
	}
	else
	{
		const CStringIndexSet*  IndexSet =  m_pHolder->m_pIndexator->GetIndexByName(m_IndexName);
		if (!IndexSet) return false;
	
		StringTokenizer tok(str.c_str(), "/");
		while (tok())
		{
			string word = tok.val();
			if (word.empty()) return false;
			if (word[word.length() - 1] == '*')
			{
				set<string> Prefixes;

				Prefixes.insert(word.substr(0,word.length() - 1));

				if (!pHolder->m_pIndexator->m_bCaseSensitive)
					AddFormsWithDifferentRegisters(m_pHolder->m_pIndexator->m_Language,Prefixes);

				for (set<string>::const_iterator it = Prefixes.begin(); it != Prefixes.end(); it++)
					IndexSet->QueryTokenListWithRightTruncation(*it, m_IndexItems);

			}
			else
			if (word[0] == '*')
			{
				if (!BuildRegExp(word.substr(1)+"$", m_IndexItems))
					return false;
			}
			else
			{
				set<string> WordForms;

				if (bExact || m_pHolder->m_pIndexator->m_bDisableDefaultQueryLexicalExpansion)
					WordForms.insert(word);
				else
					GetWordForms (m_pHolder->m_pIndexator->m_Language, word, WordForms);

				if (!pHolder->m_pIndexator->m_bCaseSensitive)
					AddFormsWithDifferentRegisters(m_pHolder->m_pIndexator->m_Language,WordForms);

				for (set<string>::const_iterator it = WordForms.begin(); it != WordForms.end(); it++)
					IndexSet->QueryTokenList(*it, m_IndexItems);
			};
			
		};
		sort(m_IndexItems.begin(),  m_IndexItems.end());
		vector<DWORD>::iterator endit = unique(m_IndexItems.begin(),  m_IndexItems.end());
		m_IndexItems.erase(endit,m_IndexItems.end());


	};

	
	return true;	
}



bool	CQueryTokenNode::CreateNodeByIndexName(const CConcHolder* pHolder, const char* IndexName, const char* Value, bool bRegularExpession)
{
	SetHolder(pHolder);
	m_Source = Value;
	
	CStringIndexSet* pIndexSet;
	{
		string name = IndexName;
		if (name.size() < 2) return false;
		if (name[0] != '$') return false;
		
		pIndexSet = pHolder->m_pIndexator->GetIndexByNameOrShortName(name.substr(1));
		if (!pIndexSet) return false;
		m_IndexName = pIndexSet->m_Name;
	};
	m_IndexItems.clear();
	
	string str = Value;

	if (bRegularExpession)
	{
		if (str.length() < 3)  return  false;
		if (str[0] != '/')return  false;
		if (str[str.length()-1] != '/')return  false;
		str = str.substr(1, str.length() - 2);
		return BuildRegExp(str, m_IndexItems);
	}
	else
		pIndexSet->QueryTokenList(str, m_IndexItems);

	
	return true;
};


bool	CQueryTokenNode::CreateFileList(const CConcHolder* pHolder,const char* src)
{
	SetHolder(pHolder);
	m_Source = src;
    string s = src;
	if (access(src,04) != 0) return false;
	m_IndexName = "Token";
	const CStringIndexSet*  IndexSet =  m_pHolder->m_pIndexator->GetIndexByName(m_IndexName);
	if (!IndexSet) return false;

	m_IndexItems.clear();
	set<string> WordForms;
	{
		FILE* fp = fopen (src,"r");
		if (!fp) return false;
		char buffer[1000];
		while (fgets(buffer,1000,fp))
		{
			string s = buffer;
			Trim(s);
			if (!s.empty())
				WordForms.insert(s);
		};
		fclose(fp);
	}


	if (!pHolder->m_pIndexator->m_bCaseSensitive)
			AddFormsWithDifferentRegisters(m_pHolder->m_pIndexator->m_Language,WordForms);

	for (set<string>::const_iterator it = WordForms.begin(); it != WordForms.end(); it++)
			IndexSet->QueryTokenList(*it, m_IndexItems);

	
	return true;
};

void CQueryTokenNode::EvaluateWithoutHits()
{
	ClearAndReserveOccurrences(5000);
	if (!m_bChunk)
	{
		m_pHolder->m_pIndexator->GetIndexByName(m_IndexName)->FindOccurrences(
				m_IndexItems, 
				m_pHolder->m_CurrentSearchPeriodNo, 
				m_Occurrences,
				m_pHolder->m_Profiler, 
				&m_pHolder->m_ShortOccurCaches,
				m_CacheIds);

		m_ChunkLengths.resize(GetOccurrencesSize(), 1);
	}
	else
	{
		m_pHolder->m_pIndexator->m_pChunkIndex->FindChunkOccurrences(
				m_IndexItems, 
				m_Occurrences, 
				m_ChunkLengths,
				m_pHolder->m_CurrentSearchPeriodNo, 			
				m_pHolder->m_Profiler, 
				&m_pHolder->m_ShortOccurCaches,
				m_CacheIds);
	};

	if (m_bUseNodeIndices)
		m_OccurrenceNodeIndices.resize(m_Occurrences.size(), m_NodeIndex);

};



void CQueryTokenNode::Evaluate()
{
	EvaluateWithoutHits();
	if (m_bChunk)
		ConvertOccurrencesToHitsForPatterns(true);
	else
		ConvertOccurrencesToHits (true);
};


// ========= CQuerySequenceNode ======================
CQuerySequenceNode::CQuerySequenceNode():CQueryNode(false)
{
	m_bAtomic = true;
};

CQuerySequenceNode::~CQuerySequenceNode()
{
	for (int i = 0; i < m_Items.size(); i++)
		delete m_Items[i];

};

bool CQuerySequenceNode::AddDistance(const char* s)
{
	if (!strcmp(s, "0")) 
	{
		m_Distances.push_back(0);
		return true;
	}
	int distance = atoi(s);
	if (distance == 0)
		return false;
	if (distance > MaxDistanceForNear)
		return false;
	m_Distances.push_back(distance);
	return true;
};

size_t CQuerySequenceNode::GetNodeFrequenceByNodeIndex(size_t NodeIndex) const
{
	size_t Result = 0;
	for (int i = 0; i < m_Items.size(); i++)
		Result += m_Items[i]->GetNodeFrequenceByNodeIndex(NodeIndex);
	return Result;
}


bool	CQuerySequenceNode::Create(const CConcHolder* pHolder, const vector<const CQueryNode*>& SequenceObj, const vector<string>& Distances)
{
	if (SequenceObj.size() != Distances.size() + 1)
	{
		// internal parse error
		return false;
	};
	

	for (int i = 0; i < Distances.size(); i++)
		if (!AddDistance(Distances[i].c_str()))
			return false;

	for (int i = 0; i < SequenceObj.size(); i++)
	{
		CQueryTokenNode* it = (CQueryTokenNode*)(void*) (SequenceObj[i]);
		m_Items.push_back(it);
	};

	SetHolder(pHolder);

	return true;	
};

void CQuerySequenceNode::EvaluateWithoutHits()
{	
	
	size_t MinOccurSize = UINT_MAX;
	//  first get occurrences for each child
	int ItemsCount = m_Items.size();
	for (size_t i = 0; i < ItemsCount; i++)
	{
		m_Items[i]->EvaluateWithoutHits();
		MinOccurSize = min (m_Items[i]->GetOccurrencesSize(), MinOccurSize);
	}

	ClearAndReserveOccurrences(MinOccurSize*ItemsCount);

	DwordVector S;	
	S.resize(ItemsCount, 0);

	// Path is currently observed path from an item of the first  list to an item of the last list
	// for each i > 0, 
	// if Path[i] != -1,  
	// then m_Items[i]->m_Occurrences[[Path[i]] - m_Items[i-1]->m_Occurrences[Path[i-1]] <= m_Distances[i-1]

	int* Path = new int[ItemsCount];	
	for (size_t i=0; i<ItemsCount; i++) Path[i] = -1;


	// depth-first search  from Path[0] 
	size_t Occur0Count = m_Items[0]->GetOccurrencesSize();
	for (Path[0] = 0; Path[0] < Occur0Count; Path[0]++)
	{
		if (Path[0]>0) // ignoring this line if it has the same value
			if (m_Items[0]->m_Occurrences[Path[0]-1] ==  m_Items[0]->m_Occurrences[Path[0]])
				continue;

		// i is the level of depth-first search
		size_t i = 1;
		while (i > 0)
		{
			// take the previos token no
			CTokenNo T = m_Items[i-1]->m_Occurrences[Path[i-1]];

			// get the number of postions in the current column
			size_t OccurCount = m_Items[i]->GetOccurrencesSize();

			// if we are for the first time  on this  level, then  Path[i]!= -1
			if (Path[i] == -1)
			{
				// position  S[i] and  Path[i] on the  first child after T
				for (Path[i]=S[i]; Path[i] < OccurCount; Path[i]++)
					if (m_Items[i]->m_Occurrences[Path[i]] >  T)
						break;
				S[i] = Path[i]; // save the starting position of Path[i] 
			}
			else
				Path[i]++; // get the next position in this column and try if it suits


			BYTE Distance = m_Distances[i-1] + 1; // get the possible distance
			bool bPositionIsFound = false;

			// searching for the position, which is inside the distance, starting from Path[i]
			for (; Path[i] < OccurCount; Path[i]++)
			{
				if (Path[i]>0) // ignoring equal occurrences
					if (m_Items[i]->m_Occurrences[Path[i]-1] ==  m_Items[i]->m_Occurrences[Path[i]])
						continue;

				
				if (m_Items[i]->m_Occurrences[Path[i]] >  T + Distance)
					break; // this is the end position, because we are outside the distance
				else
					// a suitable position is found
					if (i+1 == ItemsCount) // if it is the end  of pattern then save this occurrence
					{
						// adding current path from Path[0] to Path[ItemsCount-1] to occurrences
						for (int j=0; j < ItemsCount; j++)
							AddOccurrences(*m_Items[j], Path[j], Path[j]+1);

					}
					else
					{
						// Path[i] is a current node we should search for a child of Path[i]
						// it means we should go to the  next level (i+1)
						bPositionIsFound = true;
						break;
					};
			};

			if (bPositionIsFound)
				i++; // going to depth 
			else
			{
				// leaving this level, clearing Path[i] 
				Path[i] = -1;
				i--;
			};

		};
		// after this cycle Path[1]...Path[ItemsCount-1] must be -1
		// we are waiting  for the next value of Path[0]
		// S[1] ... S[ItemCount-1] were updated 
		// S[i] is a lower bound of a possible children of S[i-1], 
	};
	delete Path;

	// this procedure produces list of occurrences are stored by chunks, and sorted by the first item. 
	// The length of chunk is equal to the length of the  pattern,
	// Each chunk is a found occurrence of the pattern.
	m_ChunkLengths.resize(GetOccurrencesSize()/m_Items.size(), m_Items.size());


	
};

void CQuerySequenceNode::Evaluate()
{
	EvaluateWithoutHits();
	ConvertOccurrencesToHitsForPatterns(true);
};

//======================== CQueryNearNode =========================


CQueryNearNode::CQueryNearNode():CQueryNode(false)
{
	m_pChild1 = 0;
	m_pChild2 = 0;
	m_pMiddleChild = 0;
};

CQueryNearNode::~CQueryNearNode()
{
	if (m_pChild1) delete m_pChild1;
	if (m_pChild2) delete m_pChild2;
	if (m_pMiddleChild) delete m_pMiddleChild;
};

size_t CQueryNearNode::GetNodeFrequenceByNodeIndex(size_t NodeIndex) const
{
	size_t Result = 0;
	if (m_pChild1) Result += m_pChild1->GetNodeFrequenceByNodeIndex(NodeIndex);
	if (m_pChild2) Result += m_pChild2->GetNodeFrequenceByNodeIndex(NodeIndex);
	if (m_pMiddleChild) Result += m_pMiddleChild->GetNodeFrequenceByNodeIndex(NodeIndex);
	return Result;
}


bool CQueryNearNode::ReadDistanceFromString(const char* s)
{
	if (!strcmp(s, "0")) 
	{
		m_Distance = 0;
		return true;
	}

	int distance = atoi(s);
	if (distance == 0)
		return false;
	if (distance > MaxDistanceForNear)
		return false;
	m_Distance = distance;
	return true;
};

bool CQueryNearNode::Create(const CConcHolder* pHolder, const char* distance_str, CQueryNode* child1, CQueryNode* child2, CQueryNode* child3)
{
	if (!ReadDistanceFromString(distance_str)) return false;

	if (!child1->m_bAtomic) return false;
	if (!child2->m_bAtomic) return false;
	if (child3 != 0)
		if (!child3->m_bAtomic) return false;

	if (!child3)
	{
		m_pChild1 = child1;
		m_pChild2 = child2;
	}
	else
	{
		m_pChild1 = child1;
		m_pMiddleChild = child2;
		m_pChild2 = child3;
	};
	SetHolder(pHolder);
	return  true;
}

//void CQueryNearNode::EvaluateWithMiddleChild()
//{
//	m_pMiddleChild->EvaluateWithoutHits();
//	int cnt1 = m_pChild1->GetOccurrencesSize();
//	int cnt2 = m_pChild2->GetOccurrencesSize();
//	int cnt_middle = m_pMiddleChild->GetOccurrencesSize();
//	// Start2 is a index in m_pChild2->m_Occurrences
//	int Start2 = 0;
//
//	// Start3 is a index in m_pChild3->m_Occurrences
//	int Start_middle = 0;
//
//	ClearAndReserveOccurrences(min(m_pChild1->GetOccurrencesSize(),  m_pChild2->GetOccurrencesSize())*2);
//	size_t  pos1 = 0, pos2=0, pos_middle = 0; 
//
//	size_t PatternSizeMiddle;
//
//	for (int i = 0; i < cnt1; i+=m_pChild1->m_ChunkLengths[pos1], pos1++)
//	{
//		CTokenNo  T  = m_pChild1->m_Occurrences[i];
//
//		for (; Start2 < cnt2; Start2 += m_pChild2->m_ChunkLengths[pos2], pos2++)
//			if ( T <= m_pChild2->m_Occurrences[Start2+m_pChild2->m_ChunkLengths[pos2]-1] + m_Distance+1) 
//				break;
//
//		
//
//		// T is the position of the last word in chunk from the first list
//		size_t PatternSize1 = m_pChild1->m_ChunkLengths[pos1];
//		T  = m_pChild1->m_Occurrences[i + PatternSize1 - 1];
//
//	
//		size_t _pos2 = pos2;
//		for (int k = Start2; k < cnt2; k+=m_pChild2->m_ChunkLengths[_pos2++])
//			if ( T + m_Distance+1 < m_pChild2->m_Occurrences[k]) 
//				  //  == " if m_pChild2->m_Occurrences[k] is outside [T-m_Distance, T+m_Distance] then break"
//				break;
//			else
//			{
//				size_t PatternSize2 = m_pChild2->m_ChunkLengths[_pos2];
//
//				if (T < m_pChild2->m_Occurrences[k])
//				{
//					bool bCheckMiddle = false;
//					for (; Start_middle < cnt_middle; Start_middle+=m_pMiddleChild->m_ChunkLengths[pos_middle++])
//						if ( m_pMiddleChild->m_Occurrences[Start_middle] > T) 
//						{
//							bCheckMiddle = (m_pMiddleChild->m_Occurrences[Start_middle] < m_pChild2->m_Occurrences[k]);
//							break;
//						};
//					if (!bCheckMiddle)
//						continue;
//					// adding chunk from the first vector
//					AddOccurrences(*m_pChild1, i, i+PatternSize1);
//
//					
//					PatternSizeMiddle = m_pMiddleChild->m_ChunkLengths[pos_middle];
//					AddOccurrences(*m_pMiddleChild,  Start_middle, Start_middle+PatternSizeMiddle);
//
//					// adding chunk from the second vector
//					AddOccurrences(*m_pChild2, k, k+PatternSize2);
//
//					m_ChunkLengths.push_back(PatternSize1+PatternSizeMiddle+PatternSize2);
//
//				}
//				else
//				if (T > m_pChild2->m_Occurrences[k])
//				{
//					bool bCheckMiddle = false;
//					for (; Start_middle < cnt_middle; Start_middle+=m_pMiddleChild->m_ChunkLengths[pos_middle++])
//						if ( m_pMiddleChild->m_Occurrences[Start_middle] > m_pChild2->m_Occurrences[k]) 
//						{
//							bCheckMiddle = (m_pMiddleChild->m_Occurrences[Start_middle] < T);
//							break;
//						};
//					if (!bCheckMiddle)
//						continue;
//
//					// adding chunk from the second vector
//					AddOccurrences(*m_pChild2, k, k+PatternSize2);
//
//					PatternSizeMiddle = m_pMiddleChild->m_ChunkLengths[pos_middle];
//					AddOccurrences(*m_pMiddleChild, Start_middle, Start_middle+PatternSizeMiddle);
//
//					// adding chunk from the first vector
//					AddOccurrences(*m_pChild1, i, i+PatternSize1);
//
//					m_ChunkLengths.push_back(PatternSize1+PatternSizeMiddle+PatternSize2);
//				};
//			};
//
//		// if the middle child is finished than stop processing
//		if (Start_middle == cnt_middle)
//			break;
//
//	};
//
//	ConvertOccurrencesToHitsForPatterns(true);
//}

void CQueryNearNode::Evaluate()
{
	m_ChunkLengths.clear();
	m_pChild1->EvaluateWithoutHits();
	m_pChild2->EvaluateWithoutHits();

	if (m_pMiddleChild) 
	{
		m_pMiddleChild->EvaluateWithoutHits();
	};

	int cnt1 = m_pChild1->GetOccurrencesSize();
	int cnt2 = m_pChild2->GetOccurrencesSize();
	int cnt_middle = 0;
	if (m_pMiddleChild) 
		cnt_middle = m_pMiddleChild->GetOccurrencesSize();
	if (cnt1 < cnt2)
	{
		swap(m_pChild1, m_pChild2);
		swap(cnt1, cnt2);
	}

	// Start2 is a index in m_pChild2->m_Occurrences
	int Start2 = 0;

	// Start3 is a index in m_pChild3->m_Occurrences
	int Start_middle = 0;

	ClearAndReserveOccurrences(min(m_pChild1->GetOccurrencesSize(),  m_pChild2->GetOccurrencesSize())*2);
	size_t  pos1 = 0, pos2=0, pos_middle = 0; 
	size_t PatternSize2, PatternSize1, PatternSizeMiddle=0;
	const bool bMiddleChild  = m_pMiddleChild != 0;
	for (int i = 0; i < cnt1; i+=m_pChild1->m_ChunkLengths[pos1], pos1++)
	{
		CTokenNo  T  = m_pChild1->m_Occurrences[i];

		for (; Start2 < cnt2; Start2 += m_pChild2->m_ChunkLengths[pos2], pos2++)
			if ( T <= m_pChild2->m_Occurrences[Start2+m_pChild2->m_ChunkLengths[pos2]-1] + m_Distance+1) 
				break;

		

		// T is the position of the last word in chunk from the first list
		PatternSize1 = m_pChild1->m_ChunkLengths[pos1];
		T  = m_pChild1->m_Occurrences[i + PatternSize1 - 1];

	
		size_t _pos2 = pos2;
		for (int k = Start2; k < cnt2; k+=m_pChild2->m_ChunkLengths[_pos2++])
			if ( T + m_Distance+1 < m_pChild2->m_Occurrences[k]) 
				  //  == " if m_pChild2->m_Occurrences[k] is outside [T-m_Distance, T+m_Distance] then break"
				break;
			else
			{
				PatternSize2 = m_pChild2->m_ChunkLengths[_pos2];

				if (T < m_pChild2->m_Occurrences[k])
				{
					if (bMiddleChild)
					{
						bool bCheckMiddle = false;
						for (; Start_middle < cnt_middle; Start_middle+=m_pMiddleChild->m_ChunkLengths[pos_middle++])
							if ( m_pMiddleChild->m_Occurrences[Start_middle] > T) 
							{
								bCheckMiddle = (m_pMiddleChild->m_Occurrences[Start_middle] < m_pChild2->m_Occurrences[k]);
								break;
							};
						if (!bCheckMiddle)
							continue;
					};
					

					// adding chunk from the first vector
					AddOccurrences(*m_pChild1, i, i+PatternSize1);

					if (bMiddleChild)
					{
						PatternSizeMiddle = m_pMiddleChild->m_ChunkLengths[pos_middle];
						AddOccurrences(*m_pMiddleChild,  Start_middle, Start_middle+PatternSizeMiddle);
					};

					// adding chunk from the second vector
					AddOccurrences(*m_pChild2, k, k+PatternSize2);

					m_ChunkLengths.push_back(PatternSize1+PatternSizeMiddle+PatternSize2);

				}
				else
				if (T > m_pChild2->m_Occurrences[k])
				{
					if (bMiddleChild)
					{
						bool bCheckMiddle = false;
						for (; Start_middle < cnt_middle; Start_middle+=m_pMiddleChild->m_ChunkLengths[pos_middle++])
							if ( m_pMiddleChild->m_Occurrences[Start_middle] > m_pChild2->m_Occurrences[k]) 
							{
								bCheckMiddle = (m_pMiddleChild->m_Occurrences[Start_middle] < T);
								break;
							};
						if (!bCheckMiddle)
							continue;
					};

					// adding chunk from the second vector
					AddOccurrences(*m_pChild2, k, k+PatternSize2);

					if (bMiddleChild)
					{
						PatternSizeMiddle = m_pMiddleChild->m_ChunkLengths[pos_middle];
						AddOccurrences(*m_pMiddleChild, Start_middle, Start_middle+PatternSizeMiddle);
					};

					// adding chunk from the first vector
					AddOccurrences(*m_pChild1, i, i+PatternSize1);

					m_ChunkLengths.push_back(PatternSize1+PatternSizeMiddle+PatternSize2);
				};
			};

		// if the middle child is finished than stop processing
		if (bMiddleChild)
			if (Start_middle == cnt_middle)
				break;

	};

	ConvertOccurrencesToHitsForPatterns(true);

};

//void CQueryNearNode::Evaluate()
//{
//	m_ChunkLengths.clear();
//	m_pChild1->EvaluateWithoutHits();
//	m_pChild2->EvaluateWithoutHits();
//
//	if (m_pMiddleChild) 
//	{
//		EvaluateWithMiddleChild();
//		return;
//	};
//
//	int cnt1 = m_pChild1->GetOccurrencesSize();
//	int cnt2 = m_pChild2->GetOccurrencesSize();
//	if (cnt1 < cnt2)
//	{
//		swap(m_pChild1, m_pChild2);
//		swap(cnt1, cnt2);
//	}
//
//	// Start2 is a index in m_pChild2->m_Occurrences
//	int Start2 = 0;
//
//
//	ClearAndReserveOccurrences(min(m_pChild1->GetOccurrencesSize(),  m_pChild2->GetOccurrencesSize())*2);
//	size_t  pos1 = 0, pos2=0, pos_middle = 0; 
//	for (int i = 0; i < cnt1; i+=m_pChild1->m_ChunkLengths[pos1], pos1++)
//	{
//		CTokenNo  T  = m_pChild1->m_Occurrences[i];
//
//		for (; Start2 < cnt2; Start2 += m_pChild2->m_ChunkLengths[pos2], pos2++)
//			if ( T <= m_pChild2->m_Occurrences[Start2+m_pChild2->m_ChunkLengths[pos2]-1] + m_Distance+1) 
//				break;
//
//		
//
//		// T is the position of the last word in chunk from the first list
//		size_t PatternSize1 = m_pChild1->m_ChunkLengths[pos1];
//		T  = m_pChild1->m_Occurrences[i + PatternSize1 - 1];
//
//	
//		size_t _pos2 = pos2;
//		for (int k = Start2; k < cnt2; k+=m_pChild2->m_ChunkLengths[_pos2++])
//			if ( T + m_Distance+1 < m_pChild2->m_Occurrences[k]) 
//				  //  == " if m_pChild2->m_Occurrences[k] is outside [T-m_Distance, T+m_Distance] then break"
//				break;
//			else
//			{
//				size_t PatternSize2 = m_pChild2->m_ChunkLengths[_pos2];
//
//				if (T < m_pChild2->m_Occurrences[k])
//				{
//
//					// adding chunk from the first vector
//					AddOccurrences(*m_pChild1, i, i+PatternSize1);
//
//
//					// adding chunk from the second vector
//					AddOccurrences(*m_pChild2, k, k+PatternSize2);
//
//					m_ChunkLengths.push_back(PatternSize1+PatternSize2);
//
//				}
//				else
//				if (T > m_pChild2->m_Occurrences[k])
//				{
//					// adding chunk from the second vector
//					AddOccurrences(*m_pChild2, k, k+PatternSize2);
//					// adding chunk from the first vector
//					AddOccurrences(*m_pChild1, i, i+PatternSize1);
//
//					m_ChunkLengths.push_back(PatternSize1+PatternSize2);
//				};
//			};
//
//	};
//
//	ConvertOccurrencesToHitsForPatterns(true);
//
//};

CQueryWithNode::CQueryWithNode()
{
	m_bAtomic = true;
};

bool CQueryWithNode::Create(const CConcHolder*	pHolder, CQueryNode* child1, CQueryNode* child2)
{
	CQueryBinaryOperationNode::Create(pHolder, child1, child2, string(" with ") );
	return true;
};

void CQueryWithNode::EvaluateWithoutHits()
{
	((CQueryTokenNode*)m_pChild1)->EvaluateWithoutHits();
	((CQueryTokenNode*)m_pChild2)->EvaluateWithoutHits();

	m_Occurrences.resize( m_pChild1->GetOccurrencesSize() + m_pChild2->GetOccurrencesSize() );

	vector<CTokenNo>::iterator it = set_intersection (m_pChild1->m_Occurrences.begin(), m_pChild1->m_Occurrences.end(),
				   m_pChild2->m_Occurrences.begin(), m_pChild2->m_Occurrences.end(),
				   m_Occurrences.begin());

	m_Occurrences.resize(it - m_Occurrences.begin());

	if (m_bUseNodeIndices)
		m_OccurrenceNodeIndices.resize(m_Occurrences.size(), ((CQueryTokenNode*)m_pChild1)->m_NodeIndex);

	m_ChunkLengths.resize(GetOccurrencesSize(), 1);

};

void CQueryWithNode::Evaluate()
{
	EvaluateWithoutHits();
	ConvertOccurrencesToHits (true);
};



