// QNode.h: interface for the CQNode class.
//
//////////////////////////////////////////////////////////////////////

#ifndef QUERY_NODE_H
#define QUERY_NODE_H

#include "../common/utilit.h"
#include "ConcHolder.h"
#include "../PCRE/pcre_rml.h"


/*! \brief
	CQueryNode is an abstract class for any node in a query parse tree. A parse tree is built by 
	YACC during CQueryParser::ParseQuery. This class contains all members 
	which are common to atomic and operation nodes.
*/
class CQueryNode  
{
	//! is the node negated
	bool				m_bNegated;
	

				
public:
	//! true if this node is a description of one token or a sequence of adjacent tokens
	bool				m_bAtomic;  

	//! the string from which this node was created
	string				m_Source;
	//! all occurrences of this node in the current \ref period_def "subcorpus", which should be highlighted
	vector<CTokenNo>	m_Occurrences;
	//! query node indices for each occurrence (the origin for each occurrence)
	vector<BYTE>		m_OccurrenceNodeIndices;
	//! should DDC uses m_OccurrenceNodeIndices (m_OccurrenceNodeIndices is necessary only for #less_by_rank)
	bool				m_bUseNodeIndices;
	//! all hits of this node in the current \ref period_def "subcorpora"
	vector<CHit>		m_Hits;
	//! a reference to the parent holder
	const CConcHolder*	m_pHolder;
	//! the division of m_Occurrences by chunks , so m_ChunkLengths[0]+m_ChunkLengths[1]... = m_Occurrences.size()
	vector<size_t>		m_ChunkLengths;
	//!
	vector<int>			m_CacheIds;


	CQueryNode(bool bUseNodeIndices);
	virtual ~CQueryNode();

	virtual void	Evaluate();	
	virtual void	EvaluateWithoutHits();	

	//! set m_bNegated
	void	SetNegation(bool Value);
	//! get m_bNegated
	bool	GetNegation() const;
	void	AddOccurrences(const CQueryNode& FromNode, int start, int end);
	void	ClearAndReserveOccurrences(int size);
	void	SwapOccurrences(CQueryNode& Node);

	//! convert occurrences to hits  using m_pHolder->GetBreaks()
	void	ConvertOccurrencesToHits (bool bUniqueHits);
	//! convert occurrences to hits for pattern query construction (like "mother likes father")
	void	ConvertOccurrencesToHitsForPatterns (bool bUniqueHits);
	//! set  m_pHolder
	void	SetHolder(const CConcHolder*	pHolder);
	virtual size_t GetNodeFrequenceByNodeIndex(size_t NodeIndex) const;
};

/*! An abtsract class for binary logical operations on nodes (and, or). It contains instances of three major 
logical operation (union, intersection and difference) which were slightly modified to deal with 
CQueryNode::m_Hits and CQueryNode::m_Occurrences simultaneously.
*/
class CQueryBinaryOperationNode : public CQueryNode 
{
	//! add the hit list of "NodeFrom" to the back of "First"
	void hits_add(const CQueryNode& NodeFrom, vector<CHit>::const_iterator First);

protected:
	//! the first operation member
	CQueryNode*	m_pChild1;  
	//! the second operation member
	CQueryNode*	m_pChild2;  
public:
	CQueryBinaryOperationNode();
	void Create(const CConcHolder*	pHolder, CQueryNode* child1, CQueryNode* child2, string Operation);
	virtual ~CQueryBinaryOperationNode();

	//! a modified version of std::union which unites simultaneously  two pairs of vectors
	void hits_and_positions_union();
	//! a modified version of std::intersection which simultaneously  deals  with two pairs of vectors
	void hits_and_positions_intersection();
	//! a modified version of std::difference which simultaneously  deals  with two pairs of vectors
	void hits_and_positions_difference();

	size_t GetNodeFrequenceByNodeIndex(size_t NodeIndex) const;
};

//!  CQueryAndOperation implements  the hit intersection or "and" operator: for example, "Mother&&father" 
class CQueryAndOperation	: public CQueryBinaryOperationNode  
{
public:
	CQueryAndOperation();
	void Evaluate();
};

//!  CQueryAndOperation implements  the hit union or "or" operator: for example, "Mother||father" 
class CQueryOrOperation	: public CQueryBinaryOperationNode 
{
public:
	CQueryOrOperation();
	void Evaluate( );
};

//!  CQueryTokenNode is a class which corresponds to one token or it's singular property (for example lemma)
class CQueryTokenNode : public CQueryNode
{
	//!  find index items  in the index which is called m_IndexName
	vector<DWORD>		m_IndexItems;
	// the name  of index, on which this node is built (see  CQueryChunkNode and  CQueryTokenNode nodes)
	string				m_IndexName;
	// true, if it is a chunk (NP, VP and so on)
	bool				m_bChunk;


	bool BuildRegExp(string RegExpStr, vector<DWORD>& IndexItems);
public:
	// a unique index of the node in query tree
	BYTE				m_NodeIndex;
	
	CQueryTokenNode(int NodeIndex);
	virtual ~CQueryTokenNode();

	//! create pattern which is associated with "Token" index
	bool	CreateTokenPattern(const CConcHolder* pHolder, const char* src, bool bRegularExpession);
	//! create pattern which is associated with "Thes" index
	bool	CreateThesPattern(const CConcHolder* pHolder,const char* src);
	//! create pattern which is associated with "MorphPattern" or "Lemma" index
	bool	CreateMorphAnnotationPattern(const CConcHolder* pHolder,const char* src, const char* IndexName);
	//! create pattern which is associated with "Token" index, loading search tokens from a file
	bool	CreateFileList(const CConcHolder* pHolder,const char* src);
	//! create pattern which is associated with "Chunk" index 
	bool	CreateChunkPattern(const CConcHolder* pHolder, const char* ChunkTypeStr);
	//! create pattern which is associated with "IndexName" index ( a generalized version)
	bool	CreateNodeByIndexName(const CConcHolder* pHolder, const char* IndexName, const char* Value, bool bRegularExpession);

	void	EvaluateWithoutHits();
	void	Evaluate();
	size_t	GetNodeFrequenceByNodeIndex(size_t NodeIndex) const;

};


/*! \brief  CQuerySequenceNode is an implementation of sequence  operator, for example, "Mother and father".
	This class holds a sequence of nodes of type CQueryTokenNode (slot m_Items).
	Between m_Items[i] 	and  m_Items[i+1] must be no more than m_Distances[i] tokens.
*/ 
class CQuerySequenceNode : public CQueryNode
{
	vector<BYTE>				m_Distances;	
	vector<CQueryTokenNode*>	m_Items;

	//! add a distance to m_Distances, return false if the value is too large
	bool AddDistance(const char* s);
	
public:
	
	CQuerySequenceNode();
	virtual ~CQuerySequenceNode();
	//! create the node during yacc parsing
	bool	Create(const CConcHolder* pHolder, const vector<const CQueryNode*>& SequenceObj, const vector<string>& Distances);

	void	EvaluateWithoutHits();
	void	Evaluate();
	size_t	GetNodeFrequenceByNodeIndex(size_t NodeIndex) const;
};

/*!   \brief CQueryNearNode is a class which is used to search two or three 
nodes, which are inside some text period of length CQueryNearNode::m_Distance. It other words
it is an implementation of "near" operator: for example, NEAR(a,b,2).
*/ 
class CQueryNearNode : public CQueryNode 
{
	//!  the first child 
	CQueryNode*		m_pChild1;  
	//!  the middle  child  (can be null)
	CQueryNode*		m_pMiddleChild;   
	//!  the second  child  
	CQueryNode*		m_pChild2;  
	//!  the distance between the first child and the second child
	BYTE			m_Distance;

	//! read the  text distance from a string, return false if it is to large
	bool ReadDistanceFromString(const char* s);
	

public:
	
	CQueryNearNode();
	virtual ~CQueryNearNode();
	//! create the node during yacc parsing
	bool Create(const CConcHolder* pHolder, const char* distance_str, CQueryNode* child1, CQueryNode* child2, CQueryNode* child3 = 0);
	void Evaluate();
	size_t GetNodeFrequenceByNodeIndex(size_t NodeIndex) const;

};

/*!  \brief CQueryWithNode is a class which is used to search a node which simultaneously has two properties, namely 	m_pChild1 and  m_pChild2.
(implementation of "with"  operator: for example, Mother with [NOUN] )
*/ 
class CQueryWithNode : public CQueryBinaryOperationNode
{
public:
	CQueryWithNode();
	//! create the node during yacc parsing
	bool Create(const CConcHolder*	pHolder, CQueryNode* child1, CQueryNode* child2);
	void EvaluateWithoutHits();
	void Evaluate();
};



#endif 
