// ==========  This file is under  LGPL, the GNU Lesser General Public Licence
// ==========  Dialing Syntax Analysis (www.aot.ru)
// ==========  Copyright by Dmitry Pankratov, Igor Nozhov, Alexey Sokirko


#include "StdSynan.h"
#include "RusSentence.h"
#include "RusFormatCaller.h"

#undef NDEBUG
#include "RusWord.h"

CSentence* NewSentenceRussian (const CSyntaxOpt* pSyntaxOptions)
{
	return new CRusSentence(pSyntaxOptions);
};

CRusSentence::CRusSentence(const CSyntaxOpt* pSyntaxOptions)  : CSentence(pSyntaxOptions)
{
	InitClauseVectorRules();

	m_KOTORYI_INDEX = GetOpt()->m_pOborDic->FindSubConj(""); 
	m_CHEI_INDEX	 = GetOpt()->m_pOborDic->FindSubConj(""); 

}

CRusSentence::~CRusSentence()
{
	m_vectorPrimitiveRules.clear();
};

void CRusSentence::ReadNextFromPlmLinesLanguageSpecific()
{
	TryBuildVerbLemmaWithKa();
};



void CRusSentence::ChooseClauseType(const  vector<SClauseType>& vectorTypes, CMorphVariant& V)
{
	int empty_type = -1; 

	for(int i = 0 ; i < vectorTypes.size() ; i++ )
	{
		if( !vectorTypes[i].m_Root.IsEmpty() )
		{
			int node = vectorTypes[i].m_Root.m_WordNo;
			int hom = vectorTypes[i].m_Root.m_HomonymNo;

			int iUnit = V.UnitNoByWordNo(node);
			assert (iUnit != -1);
			if( V.m_SynUnits[iUnit].m_iHomonymNum == hom )
			{
				V.m_ClauseTypeNo = i;
				return;
			}				
		}
		else
			empty_type = i;

	}
	V.m_ClauseTypeNo = empty_type;

}

bool CRusSentence::RunSyntaxInClauses(ESynRulesSet type)
{
	try
	{
		if( m_pSyntaxOptions == NULL )
			return false;

		CRusFormatCaller FormatCaller(GetOpt());
		bool bRebuildAllGroups = false;
		switch(type)
		{
		case	SimpleSimilarRules :
			{
				FormatCaller.AddSimpleSimilarRules(  );
				break;
			}
		case RulesBeforeSimClauses: 
			{
				FormatCaller.AddRulesBeforeSimClauses(  );
				break;
			}
		case RulesAfterSimClauses : 
			{
				FormatCaller.AddRulesAfterSimClauses( );
				break;
			}
		case AllRules			  :
			{
				FormatCaller.AddAllRules( );
				bRebuildAllGroups = true;
				
				break;
			}

		case GroupRulesForClause			  :
			{
				FormatCaller.AddGroupRulesForClause( );
				break;
			}

		default:
			return false;		
		}		
		

		//     for_each , , ,   4  
		for(int i = 0 ; i < GetClausesCount() ; i++ )
		{

			GetClause(i).BuildGroups(FormatCaller, bRebuildAllGroups);		
			FormatCaller.Reset();
		}

		
		return true;
	}
	catch(...)
	{
		OutputErrorString("Failed RunSyntaxInClause");
		return false;
	}
}





void TryBuildVerbLemmaWithKa(CWord& W)
{
   if ( W.m_strWord.find("-") == string::npos) return;

   for (int i = 0; i < W.m_Homonyms.size(); i++)
   {
     if (W.m_Homonyms[i].HasPos(VERB))
	  if ((W.m_Homonyms[i].HasGrammem(rPlural) &&
		   W.m_Homonyms[i].HasGrammem(rPastTense)) ||
		  ((W.m_Homonyms[i].HasGrammem(rFirstPerson) || 
		    W.m_Homonyms[i].HasGrammem(rSecondPerson)) &&  
		  W.m_Homonyms[i].HasGrammem(rImperative)))
	  {
         CHomonym GoodHom = W.m_Homonyms[i];
		 W.m_Homonyms.clear();
         W.m_Homonyms.push_back(GoodHom);
		 return;
	  }
   }   
}

void CRusSentence::TryBuildVerbLemmaWithKa()
{
  for (int i = 0; i < m_Words.size(); i++)
	  ::TryBuildVerbLemmaWithKa(m_Words[i]);
}

void CRusSentence::TryToAddComparativeTypeToClause()
{
	int i,j,k;
	for (i = 0; i < GetClausesCount(); i++)
	{
		if ( GetClause(i).HasUnambiguousStrongRootWithoutWeakHomonyms()) continue;
		
		for (j = GetClause(i).m_iFirstWord; j <= GetClause(i).m_iLastWord; j++)
			for(k = 0; k < m_Words[j].GetHomonymsCount(); k++)
				if (  (   (m_Words[j].GetHomonym(k).HasPos(ADJ_FULL)) 
					   || (m_Words[j].GetHomonym(k).HasPos(NUMERAL)) // , 
					  )
					&& 
					 (m_Words[j].GetHomonym(k).HasGrammem(rComparative)) 
				   )
				{
					SClauseType NewTyp(COMPARATIVE_T, j, k);	
					GetClause(i).m_vectorTypes.push_back(NewTyp);
				}			
				
	}
}








void CRusSentence::DeleteHomOneToThousand()
{
	for (int i = 0; i < m_Words.size(); i++)
	{
		for (int j = 0; j < m_Words[i].GetHomonymsCount(); j++)
		{
			CHomonym& HomF = m_Words[i].GetHomonym(j);
			for (int k = 0; k < m_Words[i].GetHomonymsCount(); k++)
			{
				CHomonym& HomS = m_Words[i].GetHomonym(k);
				if (HomF.m_lFreqHom > 0 && HomS.m_lFreqHom > 0)
				{
					int iFrq = HomF.m_lFreqHom/HomS.m_lFreqHom;
					if (iFrq >= 1000) HomS.m_bDelete = true;
				}
			}
		}

		m_Words[i].DeleteMarkedHomonymsBeforeClauses();
	}
}

void CRusSentence::AddWeightForSynVariantsInClauses()
{
	for (int i = 0; i < GetClausesCount(); i++)
	{
		CClause&  pClause = GetClause(i);
		
		pClause.AddVariantWeightWithHomOneToFifty();
	}
}

bool FindSpecialParticlesForParenthesis(const CWord& W)
{
	if (W.FindLemma("") ||
		W.FindLemma("") ||
		W.FindLemma("") ||
		W.FindLemma("") ||
		W.FindLemma("") ||
		W.FindLemma("") ||
		W.FindLemma("") ||
		W.FindLemma("") ||
		W.FindLemma("") ||
		W.FindLemma("") ||
		W.FindLemma("") ||
		W.FindLemma("") 
		)
			return true;

	return false;
}


void CRusSentence::DetermineParenthesis()
{

	vector<CPeriod> v_oborots_del;
	int i_obr_parenthesis = -1;

	int i = 0;
	for (; i < m_Words.size(); i++)
	{
		if ( -1 != m_Words[i].GetHomonymByPOS(INP) && 
			 m_Words[i].IsOborot1() && 
			 m_Words[i].IsOborot2() )
		{
			int OborotNo = m_Words[i].GetOborotNo();
			if (		(OborotNo == -1)
					|| !m_Words[i].GetOborotPtr()->HasPartOfSpeech(INP) 
			   ) 
					OutputErrorString("Error in OBOROTS dictionary! Can't find parenthesis!");
		};
		

		if ( m_Words[i].IsOborot1() && 
			 m_Words[i].GetOborotPtr()->HasPartOfSpeech(INP) )
		{
			i_obr_parenthesis = i;

			for ( int j = i-1; j >= 0 && (m_Words[j].m_TokenType != PUNCTUAT); j--)
				if	(	(			!m_Words[j].GetHomonym(0).HasPos(PARTICLE) 
								&&	!m_Words[j].GetHomonym(0).HasPos(CONJ)
						)
						||	FindSpecialParticlesForParenthesis(m_Words[j]) 
					)
				{
					v_oborots_del.push_back( CPeriod(i, -1) );
					break;
				}
		}

		if (m_Words[i].IsOborot2() && i_obr_parenthesis > -1)
		{
			int first_ob = i_obr_parenthesis; 
			i_obr_parenthesis = -1;

			if (v_oborots_del.size() > 0 && 
				-1 == v_oborots_del[v_oborots_del.size()-1].m_iLastWord)
			{
				v_oborots_del.back().m_iLastWord = i;
				continue;
			}

			for ( int j = i+1; j < m_Words.size() && (m_Words[j].m_TokenType != PUNCTUAT); j++)
				if (		(		!m_Words[j].GetHomonym(0).HasPos(PARTICLE) 
								&&	!m_Words[j].GetHomonym(0).HasPos(CONJ) 
							)
						||	FindSpecialParticlesForParenthesis(m_Words[j])
					)
				{
					v_oborots_del.push_back( CPeriod(first_ob, i) );
					break;
				}
		}

	}

	for (i = 0; i < v_oborots_del.size(); i++)
	{
		if ( v_oborots_del[i].m_iLastWord < 0 || 
			 v_oborots_del[i].m_iFirstWord < 0) continue;

		const COborotForSyntax* O = m_Words[v_oborots_del[i].m_iFirstWord].GetOborotPtr();

		if (		(		O->HasPartOfSpeech(PARTICLE) 
						||	O->HasPartOfSpeech(CONJ) 
						||	O->HasPartOfSpeech(ADV) 
					) 
				&&	v_oborots_del[i].m_iFirstWord != v_oborots_del[i].m_iLastWord
			)
		{
			// ,          "", 
			//  ,          
			//     ,       ,        .
			m_Words[v_oborots_del[i].m_iFirstWord].m_bBadParenthesis = true;
			continue;
		};

		if ( v_oborots_del[i].m_iFirstWord == v_oborots_del[i].m_iLastWord)
		{
			for (int k = 0; k < m_Words[v_oborots_del[i].m_iFirstWord].GetHomonymsCount(); k++)
			{

				//example :  .
				CHomonym& HomDel = m_Words[v_oborots_del[i].m_iFirstWord].GetHomonym(k);
				if ( m_Words[v_oborots_del[i].m_iFirstWord].GetHomonymsCount() < 2 )
				{
					string msg_s = "Error in morphology with parenthesis : ";
					msg_s += m_Words[v_oborots_del[i].m_iFirstWord].m_strWord;					//::MessageBox(NULL, msg_s.c_str() ,"SynAn",MB_OK);
					//::MessageBox(NULL, msg_s.c_str() ,"SynAn",MB_OK);
				}
				if ( HomDel.HasPos(INP) && m_Words[v_oborots_del[i].m_iFirstWord].GetHomonymsCount() > 1 ) 
					HomDel.m_bDelete = true; 
			}

			m_Words[v_oborots_del[i].m_iFirstWord].DeleteMarkedHomonymsBeforeClauses();

			if (		!O->HasPartOfSpeech(PARTICLE)
					&&	!O->HasPartOfSpeech(CONJ)
					&&	!O->HasPartOfSpeech(ADV)
				)
				DeleteOborotThatContains(v_oborots_del[i].m_iFirstWord);

			continue;
		}

		for ( int j = v_oborots_del[i].m_iFirstWord; j <= v_oborots_del[i].m_iLastWord; j++ )
		{
			DeleteOborotThatContains(j);

			for(int k = 0 ; k < m_Words[j].GetHomonymsCount() ; k++ )
			{
				if( m_Words[j].GetHomonym(k).HasPos(CONJ) )
				{
					const char* strLemma = m_Words[j].GetHomonym(k).m_strLemma.c_str();
					int iOborNum = GetOpt()->m_pOborDic->FindSimplePrep(strLemma);
					if( iOborNum != -1 )
					{
						CHomonym& H = m_Words[j].GetHomonym(k);
						H.m_bOborot1 = true;
						H.m_bOborot2 = true;
						H.m_bInOb = true;
						H.m_OborotNo = iOborNum;
					}
					break;
				}
			}
		}
	}

}


void CreateHomonymFor_NECHEGO(CHomonym& H, long plPardigmID, string psAncode, long plOldGrammems)
{
	string s;
	H.m_lPradigmID = plPardigmID;
	
	H.m_lFreqHom = 1;
	s = "";
	H.SetLemma(s);
	H.m_iCmpnLen = 6;
	H.m_LemSign = '+';
	H.m_CommonGramCode = "??";
	const CAgramtab* Agramtab = H.GetOpt()->GetGramTab();
	H.m_iGrammems = Agramtab->GetAllGrammems( psAncode.c_str() );
	H.m_iTagID = PRONOUN_PREDK;
	H.m_iPoses = (1 << PRONOUN_PREDK);
	

	H.m_iGrammems &= ~_QM(rGenitiv);
	int lGramm = H.m_iGrammems;
	vector<int> v_gramm;

	if ( (plOldGrammems & _QM(rGenitiv)) != 0 )
	{
		H.m_iGrammems |= _QM(rGenitiv);
		lGramm |= _QM(rGenitiv);
		v_gramm.push_back(lGramm);
		lGramm &= ~_QM(rGenitiv);
	}

	if ( (plOldGrammems & _QM(rDativ)) != 0 )
	{
		H.m_iGrammems |= _QM(rDativ);
		lGramm |= _QM(rDativ);
		v_gramm.push_back(lGramm);
		lGramm &= ~_QM(rDativ);
	}

	if ( (plOldGrammems & _QM(rAccusativ)) != 0 )
	{
		H.m_iGrammems |= _QM(rAccusativ);
		lGramm |= _QM(rAccusativ);
		v_gramm.push_back(lGramm);
		lGramm &= ~_QM(rAccusativ);
	}

	if ( (plOldGrammems & _QM(rInstrumentalis)) != 0 )
	{
		H.m_iGrammems |= _QM(rInstrumentalis);
		lGramm |= _QM(rInstrumentalis);
		v_gramm.push_back(lGramm);
		lGramm &= ~_QM(rInstrumentalis);
	}

	if ( (plOldGrammems & _QM(rLocativ)) != 0 )
	{
		H.m_iGrammems |= _QM( rLocativ);
		lGramm |= _QM(rLocativ);
		v_gramm.push_back(lGramm);
		lGramm &= ~_QM(rLocativ);
	}
	
	for ( int i = 0; i < v_gramm.size(); i++ )
	{
		string strPos = H.GetPartOfSpeechStr();
		string strHelpGr;
		strHelpGr = Agramtab->GrammemsToStr(v_gramm[i]);

		string n_code;
		BYTE iPos;
		QWORD iGrm;
		strPos += " ";
		strPos += strHelpGr;
		bool Result = Agramtab->ProcessPOSAndGrammems(strPos.c_str(), iPos, iGrm); 
		assert (Result);
 		Result = Agramtab->GetGramCodeByGrammemsAndPartofSpeechIfCan(iPos, iGrm, n_code);
		assert (Result);

		H.m_GramCodes += n_code; 
	}
	
	
}


void CRusSentence::DisruptPronounPredik()
{
	int i, j;
	for ( i = 0, j = 2; i < m_Words.size() && j < m_Words.size(); i++, j++ )
		if ( m_Words[i].FindLemma("") && 
			 m_Words[i+1].GetHomonymByPOS(PREP) != -1 && 
			 m_Words[j].FindLemma(""))
			 break;

	if ( j < m_Words.size() )
	{
		int ihom = m_Words[j].GetHomonymByPOS(PRONOUN);
		if ( ihom == -1 ) return;
		QWORD lOldGrammems = m_Words[j].m_Homonyms[ihom].m_iGrammems;
		//m_Words[j].SetWordStr("");
		m_Words[j].m_Homonyms.clear();
		CHomonym Hom_nechego (this);
		CreateHomonymFor_NECHEGO(Hom_nechego, GetOpt()->m_lPradigmID_NECHEGO, GetOpt()->m_Gramcode_NECHEGO, lOldGrammems);
		m_Words[j].m_Homonyms.push_back(Hom_nechego);
		MarkWordAsDeleted( i );
	}
		
}


const long  CommonNounPrefixesCount = 2;
const string CommonNounPrefixes[CommonNounPrefixesCount] = {"-", "-"};


void CreateHomonymFor_EksVice(CHomonym& H,long plPardigmID, string psAncode, string sLem, string TypeAncode)
{
	const CAgramtab* Agramtab = H.GetOpt()->GetGramTab();
	H.m_lPradigmID = plPardigmID;
	H.m_GramCodes = psAncode;
	H.m_lFreqHom = 1;
	H.SetLemma(sLem);
	H.m_iCmpnLen = sLem.length();
	H.m_LemSign = '+';
	H.m_CommonGramCode = TypeAncode; 
	H.m_iGrammems = Agramtab->GetAllGrammems( psAncode.c_str() );
	H.m_iTagID = Agramtab->GetTagId( psAncode.c_str() );
	H.m_iPoses = (1 << H.m_iTagID);
	
}

void CRusSentence::CutPrefixEksAndVize()
{

	for ( int ii = 0; ii < m_Words.size(); ii++ )
	{
		string Word = m_Words[ii].m_strUpperWord;
		vector<int> Prefixes;
		for (long i=0; i < CommonNounPrefixesCount; i++)
		 if (CommonNounPrefixes[i] == Word.substr(0, CommonNounPrefixes[i].length()))
		 {
			 Prefixes.push_back(i);
			 Word.erase(0, CommonNounPrefixes[i].length());
			 i = -1;
			 if (Word.empty()) break;
		 };
		if (Prefixes.empty()) continue;


	
		vector<CFormInfo> Paradigms;
		GetOpt()->GetLemmatizer()->CreateParadigmCollection(false, Word, false, Paradigms);

		if ( Paradigms.size() < 1 ) continue;

		vector<CHomonym> vec_Homonyms;
		for (long k=0; k < Paradigms.size(); k++)
		{
			long lParadigm = Paradigms[k].GetParadigmId();
			string TypeAncode = Paradigms[k].GetCommonAncode();
			string sGramatCode = Paradigms[k].GetSrcAncode();
			string sLemma = Paradigms[k].GetWordForm(0); 
			if ( !binary_search(m_pSyntaxOptions->m_pProfessions->begin(), m_pSyntaxOptions->m_pProfessions->end(), sLemma) )
				continue;
			CHomonym NewHom (this);
			CreateHomonymFor_EksVice(NewHom, lParadigm, sGramatCode, sLemma, TypeAncode);
			vec_Homonyms.push_back(NewHom);
		}

		if ( vec_Homonyms.size() > 0 )
		{
			for (long i=0; i <  Prefixes.size(); i++)
			{
				m_Words[ii].m_GraphemDescriptors +=  " " + CommonNounPrefixes[Prefixes[i]];
			};

			m_Words[ii].m_Homonyms.clear();
			for ( int j = 0; j < vec_Homonyms.size(); j++ )
				m_Words[ii].m_Homonyms.push_back(vec_Homonyms[j]);
		}

	}
}

//    ,       ,
//     (     ), ..
//    .
//     - CFormatCaller::find_subj_for_verb
//: ",     ,    "
//     "",     "",
//  .  "".    
void ChangeSubjAndItsGroupGrammemsInClause(CClause& C, SVI pSynVar)
{
	if ( pSynVar->m_Subjects.size() != 1) return;


	int iSbj = pSynVar->GetUnitLastWord(pSynVar->GetFirstSubject());
	assert(iSbj != -1);
	int iPrdk = pSynVar->GetUnitLastWord(pSynVar->m_iPredk);
	assert(iPrdk != -1);

	CGroup* GrpWithSubj = const_cast<CGroup*>( C.GetLastHost(iSbj, pSynVar) );
	const CGroup* GrpWithPredk = C.GetLastHost(iPrdk, pSynVar);

	QWORD ipredk_gram = 0;
	if ( !GrpWithPredk )
		ipredk_gram = pSynVar->m_SynUnits[pSynVar->m_iPredk].m_iGrammems;
	else
		ipredk_gram = GrpWithPredk->GetGrammems();

	QWORD isubj_gram = 0;	
	if ( !GrpWithSubj )
		isubj_gram = pSynVar->m_SynUnits[pSynVar->GetFirstSubject()].m_iGrammems;
	else
		isubj_gram = GrpWithSubj->GetGrammems();

	string debug = C.GetOpt()->GetGramTab()->GrammemsToStr(isubj_gram);
	debug = C.GetOpt()->GetGramTab()->GrammemsToStr(ipredk_gram);

    QWORD new_subj_grammems = (ipredk_gram & isubj_gram & rAllNumbers) | _QM(rNominativ); 
	debug = C.GetOpt()->GetGramTab()->GrammemsToStr(new_subj_grammems);


	// setting grammems  for group
	if ( GrpWithSubj )
	{
		size_t breaks = rAllCases | rAllNumbers;	
		GrpWithSubj->SetGrammems( GrpWithSubj->GetGrammems() & (new_subj_grammems | ~breaks) );
		for ( int i = 0; i < GrpWithSubj->m_Relations.size(); i++ )
		{
			GrpWithSubj->m_Relations[i].m_iGrammems = GrpWithSubj->GetGrammems();
			if (GrpWithSubj->m_GroupType == NOUN_ADJ)
			{
				int AdjUnitNo = pSynVar->UnitNoByWordNo(GrpWithSubj->m_Relations[i].m_iLastWord);
				assert (AdjUnitNo < pSynVar->m_SynUnits.size());
				pSynVar->m_SynUnits[AdjUnitNo].ModifyGrammems(C.GetOpt(), new_subj_grammems );	
			};
		}
	}

	pSynVar->m_SynUnits[pSynVar->GetFirstSubject()].ModifyGrammems(C.GetOpt(), new_subj_grammems);	
	

}

void CRusSentence::ChangeSubjAndItsGroupGrammems()
{
	for ( int i = 0; i < GetClausesCount(); i++ )
	{
		CClause& PrCl = GetClause(i);
		for ( SVI pSynVar = PrCl.m_SynVariants.begin(); pSynVar != PrCl.m_SynVariants.end(); pSynVar++ )
		{
			if ( pSynVar->HasSubj() )
				ChangeSubjAndItsGroupGrammemsInClause(PrCl, pSynVar);
		}
	}
}

bool	CRusSentence::IsRelativSentencePronoun(int ClauseStartWordNo, int WordNo, int& HomonymNo) const
{
	int SubordConjNo = m_Words[WordNo].m_SubordinateConjNo;
	HomonymNo = 0;

	// "-"    ,      ,    ""
	if (m_Words[WordNo].m_strWord.find('-') != string::npos) return false;
	

	if (m_Words[WordNo].m_Homonyms.empty()) return false;
	return 	(SubordConjNo == m_KOTORYI_INDEX)
		||	(SubordConjNo == m_CHEI_INDEX);
};


EClauseType CRusSentence::GetClauseTypeByAncodePattern (const CAncodePattern& Pattern) const
{
	BYTE PartOfSpeech = Pattern.m_iTagID;

	switch (PartOfSpeech) {
		case VERB :  return VERB_PERS_T;
		case ADVERB_PARTICIPLE :  return ADVERB_PARTICIPLE_T;
		case PARTICIPLE_SHORT :  return PARTICIPLE_SHORT_T;
		case ADJ_SHORT :  return ADJ_SHORT_T;
		case PREDK :  return PREDK_T;
		case PARTICIPLE :  return PARTICIPLE_T;
		case INFINITIVE :  return INFINITIVE_T;
		case INP :  return INP_T;
	};
	if	(			(PartOfSpeech == ADJ_FULL)
			&&		Pattern.HasGrammem(rComparative)
		)
		return COMPARATIVE_T;
	//    
	return UnknownPartOfSpeech;
};



//      ,        1-2,
//      
//, ""  ".."    .

void CRusSentence::ProcessFio1Fio2()
{
	bool bInFIO = false;
	for (int i = 0; i < m_Words.size();i++)
	{
		if (m_Words[i].m_bFio1)
			bInFIO = true;

		if (bInFIO)
			for (int k =0; k<m_Words[i].m_Homonyms.size(); k++)
				m_Words[i].m_Homonyms[k].m_CoordConjNo = GetOpt()->GetCoordConjCount();

		if (m_Words[i].m_bFio2)
			bInFIO = false;
	};
		


};


//     ""  "e",            
void CRusSentence::DeleteSomeTypesInRelativeClauses()
{
	for (size_t ClauseNo =0; ClauseNo < m_Clauses.size(); ClauseNo++)
	{
		CClause& C = m_Clauses[ClauseNo];
		if (C.IsRelative() )
		{
			for (int i=C.m_vectorTypes.size()-1; i>=0; i--)
			if (		(C.m_vectorTypes[i].m_Type == PARTICIPLE_T)
					||	(C.m_vectorTypes[i].m_Type == ADVERB_PARTICIPLE_T)
				)
					if (m_Words[C.m_vectorTypes[i].m_Root.m_WordNo].m_Homonyms.size() > 1)
						C.DeleteHomonym(C.m_vectorTypes[i].m_Root.m_WordNo,C.m_vectorTypes[i].m_Root.m_HomonymNo);
		};
	};
};

bool	CRusSentence::IsProfession(const CHomonym& H) const
{
	return binary_search(GetOpt()->m_pProfessions->begin(), GetOpt()->m_pProfessions->end(), H.m_strLemma);
};

bool CRusSentence::BuildClauses()
{
	//  log  
	FILE* log_fp = 0;
	// log_fp = fopen("current_sentence.log", "w");
	
	ProcessFio1Fio2();

	SolveAmbiguityUsingRuleForShortAdj();
	DeleteHomonymsIfTooManyPredictedWords();
	m_bPanicMode = IsPanicSentence();
	if (m_bPanicMode)
	{
		rml_TRACE("!!!!!!!!!!  Entering panic mode !!!!!!!!!!!1\n");
	};

	CRusSentence SavedSentence(GetOpt());
	SavedSentence = *this;
	m_bShouldUseTwoPotentialRule = !m_bPanicMode;
TryWithoutTwoPotentialRule:
	bool bRes = true;

	if (log_fp) 
	{
		fprintf (log_fp, "%s....",  GetSentenceBeginStr().c_str());;
		
	};

	assert ( GetClausesCount() == 0 );

	//   : "    "
	DisruptPronounPredik();
	

	//   -  - 
	CutPrefixEksAndVize();

	InitOborotSlots();	

	
	//	   ,        .
	
	DetermineParenthesis();

	//	 ,    1/1000

	DeleteHomOneToThousand();

	FindGraPairs();

	CloneHomonymsForOborots();

	AscribeSimplePrepositionInterpretations(PREP);

	FindAllTermins();

	bool SecondTryOfCoverageKillHomonyms = false;
	vector<CWord> SaveWords = m_Words;
	vector<SFoundTermin> SaveFoundTermins = m_vectorTermins;

	

BuildInitialClausesLabel:

	TraceClauses();

	if(! BuildInitialClauses() )
	{
		fclose(log_fp);
		return false;

	};

	TraceClauses();

	DeleteSomeTypesInRelativeClauses();


	TraceClauses();

	if (log_fp)  fprintf (log_fp, "BuildAnalyticalVerbForms\n");
	BuildAnalyticalVerbForms();


	if (log_fp)  fprintf (log_fp, "TryToAddComparativeTypeToClause\n");
	TryToAddComparativeTypeToClause();

	if (log_fp)  fprintf (log_fp, "TryToRebuildDashInClause\n");
	TraceClauses();
	TryToRebuildDashInClause();
	TraceClauses();

	if (log_fp)  fprintf (log_fp, "RunSyntaxInClause(RulesBeforeSimClauses)\n");
	RunSyntaxInClauses(RulesBeforeSimClauses);

	//    RunSyntaxInClause   ,    
	if (log_fp)  fprintf (log_fp, "DeleteMarkedHomonymsWithClauses\n");
	DeleteMarkedHomonymsWithClauses();

	if (log_fp)  fprintf (log_fp, "InitWordsInTermins\n");
	InitWordsInTermins();

	

	assert ( IsValid() );

	if (!m_bPanicMode)
	{
		if (log_fp)  fprintf (log_fp, "OneRunOfClauseRules(m_vectorPrimitiveRules)\n");
		OneRunOfClauseRules(m_vectorPrimitiveRules);
		assert ( IsValid() );

		if (log_fp)  fprintf (log_fp, "RunSyntaxInClause(RulesAfterSimClauses)\n");
		RunSyntaxInClauses(RulesAfterSimClauses);

	
		if (log_fp)  fprintf (log_fp, "OneRunOfClauseRules(m_vectorMainEncloseRules)\n");
		OneRunOfClauseRules(m_vectorMainEncloseRules);
		assert ( IsValid() );

		if (log_fp)  fprintf (log_fp, "RunSyntaxInClause(GroupRulesForClause)\n");
		RunSyntaxInClauses(GroupRulesForClause);	
	
		if (log_fp)  fprintf (log_fp, "OneRunOfClauseRules(m_vectorDisruptRules)\n");
		OneRunOfClauseRules(m_vectorDisruptRules);
		assert ( IsValid() );
	};
	
	if (log_fp)  fprintf (log_fp, "RunSyntaxInClause(AllRules)\n");
	RunSyntaxInClauses(AllRules);	

	//     _   ",   "     
	// GroupRulesForClause,  OneRunOfClauseRules(m_vectorDisruptRules),      ,   .
	if (log_fp)  fprintf (log_fp, "RunSyntaxInClause(GroupRulesForClause)\n");
	RunSyntaxInClauses(GroupRulesForClause);	

	if (log_fp)  fprintf (log_fp, "BuildClauseRelations()\n");
	BuildClauseRelations();
	assert ( IsValid() );

	if (log_fp)  fprintf (log_fp, "RecalculateRelationsCoordinates()\n");
	RecalculateRelationsCoordinates();	
	AssignClauseNoToWords();
	assert ( IsValid() );

	if (log_fp)  fprintf (log_fp, "AddWeightForSynVariantsInClauses()\n");
	AddWeightForSynVariantsInClauses();
	assert ( IsValid() );

	

	 //    ,    ,     ,
	 //     (     ), ..
	 //    .
	 //     - CFormatCaller::find_subj_for_verb
	 //: ",     ,    "
	 //     "",     "",
	 //  .  "".    
	ChangeSubjAndItsGroupGrammems();


	if (!m_bPanicMode)
	{
		for ( int tt = 0; tt < GetClausesCount(); tt++ )
			if ( 0 == GetClause(tt).GetSynVariantsCount() )
				if (m_bShouldUseTwoPotentialRule)
			{
				*this = SavedSentence;
				m_bShouldUseTwoPotentialRule = false;
				if (log_fp)  fprintf (log_fp, "goto TryWithoutTwoPotentialRule\n");
				goto TryWithoutTwoPotentialRule;
			}



		if (m_pSyntaxOptions->m_KillHomonymsMode == CoverageKillHomonyms)
		{
			
			//	            ( !)
			
			if (log_fp)  fprintf (log_fp, "KillHomonymsInAllSentence\n");
		    if (    KillHomonymsInAllSentence()
			   && !SecondTryOfCoverageKillHomonyms
			   )
		   {
			   for ( int ClauseNo = 0; ClauseNo < GetClausesCount(); ClauseNo++ )
			   {
				   const CClause& Clause = GetClause(ClauseNo);
				   
					//	      c  - 
				   
				   if (			Clause.m_vectorTypes.empty()  
						//||		Clause.m_SynVariants.empty() 
					   )
				   {
					  SecondTryOfCoverageKillHomonyms = true;
					  
						//      (m_Words) ,   ,
						//     BuildAnalyticalVerbForms,     
						//. ,..      .
						//       m_Words = SaveWords,
						//     ,   KillHomonymsInAllSentence.
						//   .
					  
					  vector<CWord> p;
					  for (int i =0; i < SaveWords.size(); i++)
					  {
							int j=0;
							for (; j< m_Words.size(); j++)
							  if (			(SaveWords[i].m_GraphematicalUnitOffset == m_Words[j].m_GraphematicalUnitOffset)
									&&		(SaveWords[i].m_strWord == m_Words[j].m_strWord)
								  )
								  break;
							if (j < m_Words.size())
							{
								p.push_back(m_Words[j]);
								p.back().m_MainVerbs = SaveWords[i].m_MainVerbs;
								p.back().m_TrennbarePraefixWordNo = SaveWords[i].m_TrennbarePraefixWordNo;
							}
							else
								p.push_back(SaveWords[i]);

					  };
					  m_Words = p;
					  m_vectorTermins = SaveFoundTermins;
					  if (log_fp)  fprintf (log_fp, "goto  BuildInitialClausesLabel\n");
					  goto  BuildInitialClausesLabel;
				   };
			   };
		   };


		};
	};


	assert ( IsValid() );

	if (log_fp)  fprintf (log_fp, "fclose(log_fp)\n");
	if (log_fp)  fclose(log_fp);	

	return true;

}





