#!/bin/bash
###
### This file is covered by the GNU General Public License
### version 3 or later.
###
### Copyright (C) 2021-2025, ALT Linux Team

########################
### Common functions ###
########################

# Adds a record to the log file and/or sends a message to the system log
#
log()
{
	: # Does nothing initially
}

# Logs a message in debug mode only
#
dbg()
{
	: # Does nothing initially
}

# Dumps the contents of file "$1" to the log file in debug mode
#
fdump()
{
	: # Does nothing initially
}

# Logs a message about the command that will be executed
#
will_be_run()
{
	: # Does nothing initially
}

# Notify the "alterator browser" frontend
#
notify_alterator()
{
	: # Does nothing initially
}

# Sends a notification to a different type of frontend
#
notify_frontend()
{
	: # Does nothing initially
}

# Runs a command with an optional logging for debugging purposes
#
run()
{
	will_be_run "$@"
	"$@" || return $?
}

# Displays an informational message
#
msg()
{
	local msg fmt="$1"

	shift
	[ -n "$quiet" ] ||
		printf "${CLR_LC1}${fmt}${CLR_NORM}\n" "$@"
	log "$fmt" "$@"

	if [ -p "$frontend_pipe" ]; then
		notify_frontend I "$fmt" "$@"
	elif [ -x "$alterator" ]; then
		msg="$(printf "$fmt" "$@")"
		notify_alterator message "$msg"
	fi
}

# Displays a warning
#
warn()
{
	local fmt="$1"

	shift
	log "%s: $fmt" WARNING "$@"

	if [ -z "$quiet" ]; then
		printf "${CLR_WARN}%s: ${fmt}${CLR_NORM}\n" WARNING "$@"
	elif [ -z "$SYSREST_DEBUG" ]; then
		printf "%s: $fmt\n" WARNING "$@" >&2
	fi

	if [ -p "$frontend_pipe" ]; then
		notify_frontend W "$fmt" "$@"
	elif [ -x "$alterator" ]; then
		msg="$(printf "W: $fmt" "$@")"
		notify_alterator message "$msg"
	fi
}

# Displays a fatal error message and launch the rescue shell
#
fatal()
{
	local rv="$?" fmt="$1"

	trap - ERR; shift
	[ -z "$quiet" ] ||
		exec 1>&3 3>&-
	quiet=
	log "%s #%s: $fmt" FATAL "$rv" "$@"
	printf "${CLR_ERR}%s #%s: ${fmt}${CLR_NORM}\n" FATAL "$rv" "$@"

	if [ -p "$frontend_pipe" ]; then
		notify_frontend E "#%s: $fmt" "$rv" "$@"
	elif [ -x "$alterator" ]; then
		msg="$(printf "$fmt" "$@")"
		notify_alterator error "$msg"
		exit "$rv"
	fi

	if in_array "$action" deploy sysrest fullrest; then
		fmt="Press Ctrl-D to exit the deployment shell..."
		echo -e "${CLR_WARN}${fmt}${CLR_NORM}"
		PS1="(${CLR_LC2}DEPLOY${CLR_NORM}) # " /bin/bash -i 2>&1
	fi

	[ "$rv" != 0 ] ||
		rv=1
	frontend_pipe=
	alterator=

	exit "$rv"
}

# Unexpected Error Handler
#
unexpected_error()
{
	fatal "Unexpected error #%s catched in %s[#%s]." "$?" "$1" "$2"
}

# Tracking custom hooks
#
track_hook()
{
	local rc=0

	[ -z "$disable_hooks" ] ||
		return 0
	dbg "Entering hook '%s'..." "$1"
	"$@" || rc="$?"
	dbg "Leaving hook '%s'..." "$1"

	return "$rc"
}

# Search element "$1" in the array "$@" and return 0 if it found
#
in_array()
{
	local needle="$1"; shift

	while [ "$#" -gt 0 ]; do
		[ "$needle" != "$1" ] ||
			return 0
		shift
	done

	return 1
}

# Return 0 if argument is an integer number
#
is_number()
{
	[ -n "${1##*[!0-9]*}" ] && [ "$1" -ge 0 ] 2>/dev/null
}

# Sets global variables: platform, efi_suffix, uefiboot and hypervisor
#
detect_hw_platform()
{
	uefiboot=
	efi_suffix=
	platform="$(uname -m)"

	case "$platform" in
	e2k*)	platform=elbrus
		;;
	i?86|pentium|athlon)
		platform=i586
		;;
	esac

	case "$platform" in
	aarch64)
		efi_suffix=aa64.efi
		;;
	elbrus)
		no_nvram=1
		pt_scheme=dos
		;;
	i586)
		no_nvram=1
		pt_scheme=dos
		;;
	ppc64le)
		# Do nothing at this point
		;;
	x86_64)
		efi_suffix=x64.efi
		;;
	*)
		fatal "Hardware platform %s is not supported yet." "$platform"
		;;
	esac

	# Determinate the current boot mode
	[ ! -d /sys/firmware/efi ] || uefiboot=1

	# Determinate the hypervisor vendor
	hypervisor="$(lscpu |sed -n -E "s/^Hypervisor vendor:\s+//p")"
}

# Converts human-readable sizes to long integer bytes
#
human2size()
{
	local input="$1" rv=
	local slen="${#input}"
	slen="$(($slen - 1))"
	local data="${input:0:$slen}"
	local lchar="${input:$slen:1}"

	case "$lchar" in
	[0-9]) rv="$input";;
	K) rv="$(( $data * 1024 ))";;
	M) rv="$(( $data * 1024 * 1024 ))";;
	G) rv="$(( $data * 1024 * 1024 * 1024 ))";;
	T) rv="$(( $data * 1024 * 1024 * 1024 * 1024 ))";;
	esac

	is_number "$rv" ||
		exit 1
	echo -n "$rv"
}

# Converts the actual disk size in bytes into a human-readable form
#
size2human()
{
	local real mul=1024
	local frac s suffix="iB"
	local sizes="K M G T P E Z Y"

	if [ "${1-}" = "-w" ]; then
		suffix="B"
		mul=1000
		shift
	fi

	[ "$#" = 1 ] && is_number "${1-}" ||
		exit 1
	real="$1"

	if [ "$real" -lt "$mul" ]; then
		printf "%u bytes" "$real"
		return
	fi

	for s in $sizes; do
		frac="$(( $real * 100 / $mul ))"
		real="$(( $real / $mul ))"
		[ "$real" -ge "$mul" ] ||
			break
	done

	printf "%u.%02u %s%s" "$real" "$((frac % 100))" "$s" "$suffix"
}

# Returns 0 if specified program ($1) exists
#
have_binary()
{
	type -p -- "$1" >/dev/null
}

# Shows a list of network interfaces
#
list_ifaces()
{
	local iface sysfs=/sys/class/net

	find "$sysfs" -mindepth 1 -maxdepth 1 \
		-xtype d -printf '%f\n' |sort |
	while read -r iface; do
		[ "$iface" != lo ] ||
			continue
		[ -f "$sysfs/$iface"/type ] ||
			continue
		printf "%s\n" "$iface"
	done
}

# Shows a list of wired interfaces
#
list_wired_ifaces()
{
	local type iface
	local sysfs=/sys/class/net

	find "$sysfs" -mindepth 1 -maxdepth 1 \
		-xtype d -printf '%f\n' |sort |
	while read -r iface; do
		[ "$iface" != lo ] ||
			continue
		[ -f "$sysfs/$iface"/type ] ||
			continue
		read -r type <"$sysfs/$iface"/type 2>/dev/null ||
			continue
		[ "$type" = 1 ] ||
			continue
		[ ! -d "$sysfs/$iface"/wireless ] ||
			continue
		[ ! -f "$sysfs/$iface"/phy80211/name ] ||
			continue
		printf "%s\n" "$iface"
	done
}

# Determines the wired interface name and future hostname
#
setup_hostname()
{
	local hw="" ip="" iface=""
	local pattern='s/^[0-9]*//g; s|[^[:alnum:]]||g; s/.*/\L&/g'

	if have_binary ip; then
		iface="$(run ip route |
				grep -sE '^default via ' |
				awk '{print $5;}')"
	fi

	[ -n "$iface" ] ||
		iface="$(list_wired_ifaces |head -n1)"
	[ -n "$iface" ] ||
		iface="$(list_ifaces |head -n1)"
	newifname="${newifname:-$iface}"

	if [ -n "$iface" ] && have_binary ip; then
		hw="$(run ip link show dev "$iface" |
			    grep -s ' link/ether ' |
			    head -n1 |
			    awk '{print $2;}' |
			    sed -e 's/[:\.\-]//g')"
		ip="$(run ip addr show dev "$iface" |
			    grep -s ' inet ' |
			    head -n1 |
			    awk '{print $2;}' |
			    cut -f1 -d/)"
	fi

	case "$hostnaming" in
	""|copy)
		# do not modify the original template, just copy it "as is"
		;;

	ip1)	# append the last (4th) part of the IP-address to the base name
		[ -z "$ip" ] ||
			computer="$computer-${ip##*.}"
		;;

	ip2)	# append the two last (3rd and 4th) parts of the IP-address
		[ -z "$ip" ] ||
			computer="$computer-$(echo "$ip" |cut -f3- -d. |tr . _)"
		;;

	ip3)	# append the three last (2nd...4th) parts of the IP-address
		[ -z "$ip" ] ||
			computer="$computer-$(echo "$ip" |cut -f2- -d. |tr . _)"
		;;

	hw6)	# append some hex digits from the Ethernet MAC-address
		case "${#hw}" in
		12|16|40)
			computer="$computer-${hw:6}"
			;;
		esac
		;;

	rnd)	# generate a random part and append it to the base name
		#
		# This was borrowed from installer/preinstall.d/40-autohostname.sh:
		# 1. Remove all digits from beginning [RFC-952], relaxed in [RFC-1123, 2.1]
		# 2. Remove all non-alphanumeric characters from hostname
		#    (base64 generates [:alnum:],[+/=], only [:alnum:][-.] allowed)
		# 3. Convert everything to lowercase.
		#
		if have_binary pwqgen; then
			hw="$(run pwqgen random=24 |
					sed -e "$pattern" |cut -c1-8)"
			computer="${computer:+$computer-}$hw"
		else
			hw="$(run head -c 32 /dev/urandom |base64 |
					sed -e "$pattern" |cut -c1-8)"
			computer="${computer:+$computer-}$hw"
		fi
		;;

	*)	# use a user-defined host naming function
		track_hook "$hostnaming" "$iface" "$hw" "$ip"
		;;
	esac

	run hostname "$computer"
}

