#!/bin/bash
###
### This file is covered by the GNU General Public License
### version 3 or later.
###
### Copyright (C) 2021-2025, ALT Linux Team

#############################
### http protocol handler ###
#############################

# Outputs a list of additional required tools
#
get_proto_requires()
{
	echo "curl"
}

# Returns 0 if write operations are allowed on the remote side
#
have_write_access()
{
	return 1
}

# Sends an HTTP request and captures the HTTP response header
#
grab_http_header()
{
	local opts url="$1"
	local status i=0 rc=0
	local tmpf timeout=30 dir=
	local regex="s/^Content\-Length:\s+([0-9]+)/\1/p"

	if [ "$url" = "-d" ] && [ -n "${2-}" ]; then
		regex="s/^Content\-Type:\s+//p"
		url="$2"
		dir=1
	fi

	opts="-sI --no-buffer --connect-timeout 5 --max-time 7"
	opts="$opts --max-redirs 2 --write-out %{http_code}"
	tmpf="$(run mktemp -qt)"
	filesize=

	while [ "$i" -lt "$timeout" ]; do
		rc=0; run rm -f -- "$tmpf"
		status="$(run curl $opts --output "$tmpf" --url "$url")" ||
			rc="$?"
		[ ! -s "$tmpf" ] ||
			fdump "$tmpf"
		dbg "curl status: %s, protocol status: %s" "$rc" "$status"

		if [ "$rc" = 0 ] && [ "$status" = 200 ]; then
			filesize="$(sed -n -E -- "$regex" "$tmpf" |
					tail -n1 |sed -e 's,\r$,,')"
			[ -z "$filesize" ] || break
		else
			rc=1

			case "$status" in
			301|302|401|403|404)
				# This is unrecoverable client/server error,
				# there is no point in waiting for a fix
				log "unrecoverable http error: %s" "$status"
				break
				;;
			esac
		fi

		i=$((3 + $i))
		sleep 3
	done

	[ -z "$dir" ] ||
		filesize=
	run rm -f -- "$tmpf"

	return "$rc"
}

# Returns 0 if the specified directory exists on the remote side
#
dir_exists()
{
	local filesize=

	grab_http_header -d "$backup/$1/"
}

# Returns 0 if the specified file on the remote side is not empty
#
file_exists()
{
	local filesize=

	grab_http_header "$backup/$1" && [ -n "$filesize" ] &&
		is_number "$filesize" && [ "$filesize" -gt 0 ]
}

# Displays the specified file size (in bytes) if it exists on the remote side
#
get_file_size()
{
	local filesize=

	if grab_http_header "$backup/$1"; then
		printf "%s" "$filesize"
	fi
}

# Creates the specified directory on the remote side
#
create_directory()
{
	local msg="Function %s() is not implemented for '%s' protocol."

	fatal "$msg" create_directory http
}

# Reads the specified file on the remote side and writes it to stdout
#
read_file()
{
	local opts filesize="${2-}"

	opts="--silent --no-buffer --connect-timeout 5 --max-redirs 2"
	[ -z "$filesize" ] ||
		opts="$opts --max-filesize $filesize"
	curl $opts -- "$backup/$1" || printf "%s, curl\n" "$?" >"$workdir"/ERROR
}

# Writes to the specified file on the remote side, reads it from stdin
#
write_file()
{
	local msg="Function %s() is not implemented for '%s' protocol."

	fatal "$msg" write_file http
}

