#!/bin/bash
###
### This file is covered by the GNU General Public License
### version 3 or later.
###
### Copyright (C) 2021-2025, ALT Linux Team

###################################################################
### Setup the logger and user interface for long-life processes ###
###################################################################

# Adds a record to the log file and/or sends a message to the system log
#
log()
{
	local msg fmt

	[ -n "$logprio" ] || [ -n "$logfile" ] ||
		return 0
	fmt="$1"; shift
	msg="$(printf "$fmt" "$@")"
	( fmt="$(env LC_TIME=C date +'%F %T')"
	  [ -z "$logprio" ] ||
		logger -t "$progname" -p "$logprio" -- "$msg" ||:
	  [ -z "$logfile" ] ||
		printf "[%s] %s\n" "$fmt" "$msg" >>"$logfile" ||:
	) 2>/dev/null
}

# Dumps the contents of file "$1" to the log file in debug mode
#
fdump()
{
	[ -n "$SYSREST_DEBUG" ] && [ -n "$logfile" ] ||
		return 0
	( local time="$(env LC_TIME=C date +'%F %T')"
	  printf "[%s] %s contents:\n" "$time" "$1"
	  printf "########################################\n"
	  cat -- "$1"
	  printf "########################################\n"
	) >>"$logfile" 2>/dev/null ||:
}

# Notify the "alterator browser" frontend.
#
# See also:
# 1) string_quote() in alterator-sh-functions
# 2) message_quote() in alterator-pkg/backend3
# 3) alterator-mailbox-send.c in alterator-lookout
#
notify_alterator()
{
	local msg type="$1"

	shift
	msg="$*"
	"$alterator" "$type \"${msg//\"/\\&}\""
}

# Sends a notification to a different type of frontend
#
notify_frontend()
{
	local type="$1" fmt="$2"

	shift 2
	printf "%s: $fmt\n" "$type" "$@" >"$frontend_pipe"
}

# Logs a message in debug mode only
#
dbg()
{
	[ -n "$SYSREST_DEBUG" ] ||
		return 0
	[ -z "$frontend_pipe" ] ||
		notify_frontend D "$@"
	log "$@"
}

# Logs a message about the command that will be executed
#
will_be_run()
{
	[ -n "$SYSREST_DEBUG" ] ||
		return 0
	[ -z "$frontend_pipe" ] ||
		notify_frontend R "%s" "$*"
	log "Executing: %s" "$*"
}

# Setup the logger
#
setup_logger()
{
	local f a

	if [ -n "$logfile" ]; then
		mkdir -p -m 0755 -- "${logfile%/*}"
		[ -w "$logfile" ] ||
			:> "$logfile"
		[ -z "$SYSREST_DEBUG" ] ||
			exec 2>>"$logfile"
	fi

	if [ -p "$frontend_pipe" ]; then
		alterator=
		dbg "Using frontend's pipe: %s" "$frontend_pipe"
	else
		f="$frontend_pipe"
		a="$alterator"
		alterator=

		if [ -n "$a" ] &&
		   have_binary alterator-mailbox-send &&
		   kill -0 alteratord >/dev/null 2>&1
		then
			if [ -p /tmp/alterator/browser-sock ] ||
			   [ -p "$TMPDIR"/alterator/browser-sock ]
			then
				alterator="$(type -p alterator-mailbox-send)"
			fi
		fi

		if [ -n "$f" ]; then
			frontend_pipe=
			fatal "Specified FIFO not found: '%s'." "$f"
		elif [ -n "$alterator" ]; then
			dbg "Using alterator browser frontend"
		elif [ -n "$a" ]; then
			fatal "Can't interact with the alterator browser frontend."
		fi
	fi

	dbg "Started with arguments: %s" "$*"
}

