#!/bin/bash
###
### This file is covered by the GNU General Public License
### version 3 or later.
###
### Copyright (C) 2021-2025, ALT Linux Team

##################################
### Profile creation functions ###
##################################

# Fills a new profile with empty files
#
fill_profile()
{
	local dir="$1"

	mkdir -p -m 0755 -- "$dir"/rootfs
	: >"$dir"/sysrest.ini
	: >"$dir"/hooks.sh
	: >"$dir"/chroot-pre.sh
	: >"$dir"/chroot-post.sh
}

# Creates a new profile and shows the path to it
#
make_profile()
{
	local field x p s
	local d=/sys/class/dmi/id

	# Check fields
	[ -d "$d" ] ||
		d=/sys/devices/virtual/dmi/id
	[ -n "$cpuarch" ] ||
		cpuarch="$(uname -m)"
	[ -n "$baremetal" ] ||
		baremetal="$(machine_name)"
	[ -n "$classid" ] ||
		classid="$(machine_uuid)"
	[ -n "$profile" ] ||
		profile="$(machine_profile)"
	[ -n "$datetime" ] ||
		datetime="$(env LC_TIME=C date -u +'%F %T')"
	p="$backup/profiles/$profile"
	s="$backup/profiles/@$profile"
	[ ! -f "$s"/PROFILE ] && [ ! -f "$s"/PARENT ] && [ ! -f "$s"/SHARED ] &&
	[ ! -f "$p"/PROFILE ] && [ ! -f "$p"/PARENT ] && [ ! -f "$p"/SHARED ] ||
		fatal "Profile '%s' already exists." "$profile"

	# Check the parent profile
	if [ -n "$parent" ]; then
		[ "$parent" != "$profile" ] &&
		[ ! -f "$backup/profiles/@$parent"/PARENT ] &&
		[ ! -f "$backup/profiles/@$parent"/PROFILE ] ||
			fatal "Invalid parent profile specified: '%s'." "$parent"
	fi

	# Create a profile directory
	mkdir -p -m 0755 -- "$p"

	# Fill out a regular profile
	( cat <<-EOF
		VERSION: 1.0
		CPUARCH: $cpuarch
		MACHINE: $baremetal
		CLASSID: $classid
		COMMENT: $comment
		CREATED: $datetime

		EOF

	  for field in "${DMI_CLASS_FIELDS[@]}"; do
		[ -r "$d/$field" ] ||
			continue
		x="$(head -n1 -- "$d/$field" 2>/dev/null |
			sed -E 's/^\s+//' |sed -E 's/\s+$//')"
		[ -z "$x" ] || printf "DMI.%s: %s\n" "$field" "$x"
	  done
	) >"$p"/PROFILE

	# Save a profile link
	x="${classid//\-/}"
	[ ! -s "$catalog" ] && :>"$catalog" ||
		sed -i -e "/^$x /d" "$catalog"
	printf "%s %s\n" "$x" "$profile" >>"$catalog"

	# Check the parent profile
	if [ -z "$parent" ]; then
		fill_profile "$p"
	else
		# Use parent profile
		x="$backup/profiles/@$parent"

		# Create a new shared profile
		if [ ! -s "$x"/SHARED ]; then
			fill_profile "$x"

			( cat <<-EOF
				VERSION: 1.0
				COMMENT: $shared_comment
				CREATED: $datetime

				EOF
			) >"$x"/SHARED
		fi

		# Save direct and backlinks
		printf "%s\n" "$parent" >"$p"/PARENT
		printf "USED-BY: %s\n" "$classid" >>"$x"/SHARED
	fi

	# Show relative path to created profile
	printf "profiles/%s\n" "$profile"
}

# Clears profile contents from any files
#
erase_profile()
{
	local p shared parent=

	[ -n "$profile" ] ||
		profile="$(machine_profile)"
	shared="$backup/profiles/@$profile"
	p="$backup/profiles/$profile"

	if [ -s "$shared"/SHARED ]; then
		[ ! -f "$p"/PROFILE ] && [ ! -f "$p"/PARENT ] &&
		[ ! -f "$shared"/PROFILE ] && [ ! -f "$shared"/PARENT ] ||
			fatal "Invalid profile: '%s'." "$profile"
		p="$shared"
	else
		[ -s "$p"/PROFILE ] ||
			fatal "Profile '%s' not found." "$profile"
		[ ! -f "$shared"/SHARED ] && [ ! -f "$p"/SHARED ] ||
			fatal "Invalid profile: '%s'." "$profile"
		shared=
	fi

	[ -n "$shared" ] || [ ! -s "$p"/PARENT ] ||
		parent="$(head -n1 -- "$p"/PARENT)"
	[ -z "$parent" ] || [ -s "$backup/profiles/@$parent"/SHARED ] ||
		fatal "Broken parent profile link: %s -> @%s." "$profile" "$parent"
	[ -n "$shared" ] || [ -z "$parent" ] ||
		p="$backup/profiles/@$parent"
	cd -- "$backup"/ && rm -vrf -- "$p"/rootfs "$p"/update.tgz >&2
	fill_profile "$p"
}

