#!/bin/bash
###
### This file is covered by the GNU General Public License
### version 3 or later.
###
### Copyright (C) 2021-2025, ALT Linux Team

############################################################
### Supplemental code for working with hardware profiles ###
############################################################

# Machine class
DMI_CLASS_FIELDS=(
	bios_vendor
	board_name
	board_vendor
	board_version
	chassis_type
	chassis_vendor
	chassis_version
	product_family
	product_name
	product_sku
	sys_vendor
)

# Unique instance
DMI_INSTANCE_FIELDS=(
	board_asset_tag
	board_serial
	chassis_asset_tag
	chassis_serial
	product_serial
	product_uuid
	product_version
)

# Hook: it can detect the situation where the specified or auto-detected
# profile is not found. It should return zero if new $profile is set, or
# non-zero value to reset $profile. By default, the call will be ignored
# to continue without profiles. The first hook's parameter is the current
# machine's UUID.
#
profile_not_found()
{
	[ -n "$profile_required" ] ||
		return 1
	[ -n "$profile" ] ||
		fatal "Could not determinate the profile name."
	fatal "The profile '%s' for machine %s not found." "$profile" "$1"
}

# Hook: it can set or reset the $profile variable after the profile has
# already been auto-detected or specified at startup, we have the ability
# to extend or change the default logic here
#
select_profile()
{
	: # Do nothing by default
}

# Hook: it can check the hardware and set variables
# such as: $baremetal, $hypervisor, $computer, etc
#
check_hardware()
{
	__check_hardware
}

# Default implementation of check_hardware() hook
#
__check_hardware()
{
	local v="$baremetal"

	[ -z "$restrict_hardware" ] || [ "$v" = "$restrict_hardware" ] ||
		fatal "It is another hardware: '%s', not for this backup." "$v"
	[ -z "$hypervisor" ] || baremetal=
}

# Returns the unchangeable, time-independent UUID of the current machine,
# for example: "43af9dc2-3e88-7ad3-67f8-c1684f07636161c1e565"
#
machine_uuid()
{
	local x

	__scan_pci_bus()
	{
		local sysfs h="[0-9a-f]"
		local root="${1:-/sys/devices}"
		local glob="$h$h$h$h:$h$h:$h$h.$h*"

		__handle_field()
		{
			[ -r "$sysfs/$1" ] &&
				read -r x <"$sysfs/$1" 2>/dev/null ||
					x="-"
			printf " %s" "$x"
		}

		find -- "$root" -mindepth 2 -maxdepth 2 \
			-type d -name "$glob" 2>/dev/null |
				grep -- "$root/pci" |
				sort |
		while read -r sysfs; do
			printf "%s" "${sysfs##*/}"
			__handle_field class
			__handle_field vendor
			__handle_field device
			__handle_field subsystem_vendor
			__handle_field subsystem_device
			__handle_field revision
			printf "\n"
		done

		unset __handle_field
	}

	__show_hw_data()
	{
		local field d="${1:-/sys/class/dmi/id}"

		[ -d "$d" ] ||
			d=/sys/devices/virtual/dmi/id
		uname -m

		for field in "${DMI_CLASS_FIELDS[@]}"; do
			[ -r "$d/$field" ] ||
				continue
			x="$(head -n1 -- "$d/$field" 2>/dev/null ||:)"
			[ -z "$x" ] || printf "%s: %s\n" "$field" "$x"
		done

		[ -z "$pci_bus" ] || __scan_pci_bus "${2-}"
	}

	x="$(__show_hw_data "${1-}" "${2-}" |
					sha1sum |
					cut -f1 -d' ')"
	printf "%s-%s-%s-%s-%s" "${x:0:8}" "${x:8:4}" \
			"${x:12:4}" "${x:16:4}" "${x:20}"
	unset __show_hw_data __scan_pci_bus
}

# Returns the current machine name as written to the DMI,
# for example: "LG gram PC 14Z90Q-K.ADB9U1"
#
machine_name()
{
	local l r d="${1:-/sys/class/dmi/id}"

	__read_dmi()
	{
		head -n1 -- "$d/$1" 2>/dev/null |
			grep -sv 'O.E.M.' |
			grep -svw 'OEM'   |
			sed -E 's/^\s+//' |
			sed -E 's/\s+$//'
	}

	[ -d "$d" ] ||
		d=/sys/devices/virtual/dmi/id
	l="$(__read_dmi product_family)"
	[ -n "$l" ] ||
		l="$(__read_dmi product_sku)"
	r="$(__read_dmi product_name)"

	if [ -n "$l" ] && [ -n "$r" ]; then
		printf "%s %s" "$l" "$r"
		unset __read_dmi
		return 0
	fi

	l="$(__read_dmi board_vendor)"
	r="$(__read_dmi board_name)"

	if [ -n "$l" ] && [ -n "$r" ]; then
		printf "%s %s" "$l" "$r"
		unset __read_dmi
		return 0
	fi

	l="$(__read_dmi sys_vendor)"
	printf "%s %s" "${l:-Unknown}" "$(uname -m)"
	unset __read_dmi
}

# Converts machine name to profile name,
# for example: "lg_gram_pc_14Z90q-k.adb9u1"
#
machine_profile()
{
	[ -n "$baremetal" ] ||
		baremetal="$(machine_name)"
	printf "%s\n" "$baremetal" |
		tr '[[:upper:]]' '[[:lower:]]' |
		tr '[\[\]\(\)\{\}\%\=\:\;\,\+\^\$\!\~\*\#\&\/\\_]' ' ' |
		sed -E 's/\s+/_/g' |
		sed -E 's/^[\-_\.@]+//' |
		sed -E 's/[\-_\.]+$//' |
		cut -f1-32
}

# Checks if the profile name is correct
#
validate_profile_name()
{
	local c name="$1"
	local len="${#name}"

	[ "$len" != 0 ] && [ "$len" -le 32 ] ||
		return 1
	c="${name:0:1}"
	[ "$c" != '-' ] && [ "$c" != '_' ] &&
	[ "$c" != '.' ] && [ "$c" != '@' ] ||
		return 1
	c="${name:$(( $len - 1 )):1}"
	[ "$c" != '-' ] && [ "$c" != '_' ] && [ "$c" != '.' ] ||
		return 1
}

# Checks if the specified UUID is correct
#
validate_uuid()
{
	local h="[0-9a-f]"
	local b="$h$h$h$h"
	local a="$b$b"
	local c="$a$b"

	case "$1" in
	$a-$b-$b-$b-$c)
		return 0
		;;
	esac

	return 1
}

# Tries to include a configuration file or script with user-defined hooks
#
user_config()
{
	local cfg f="$1"
	local type=configuration

	file_exists "$f" ||
		return 0

	case "$f" in
	*.sh)	type=script;;
	esac

	if [ "$backup_proto" = file ]; then
		cfg="$backup/$f"
	else
		dbg "Downloading %s '%s'..." "$type" "$f"
		cfg="$workdir/tmp-user.sh"
		read_file "$f" >"$cfg"
	fi

	dbg "Using user-defined %s: '%s'..." "$type" "$f"

	( . "$cfg" ) >/dev/null 2>&1 ||
		fatal "Invalid source file: '%s'." "$f"
	. "$cfg"

	if [ "$backup_proto" != file ]; then
		run rm -f $verbose -- "$cfg"
	fi
}

# Tries to determine the profile name using the machine catalog
#
detect_profile()
{
	local uuid detected=
	local list=profiles/CATALOG.lst

	if [ -n "$instance" ] &&
	   [ "${#DMI_INSTANCE_FIELDS[@]}" != 0 ]
	then
		DMI_CLASS_FIELDS=(
			"${DMI_CLASS_FIELDS[@]}"
			"${DMI_INSTANCE_FIELDS[@]}"
		)
		DMI_INSTANCE_FIELDS=()
	fi

	uuid="$(machine_uuid)"

	if [ -z "$profile" ] && file_exists "$list"; then
		profile="$(read_file "$list" |
				sed -n -e "s/^${uuid//\-/} //p" |
				tail -n1 |
				cut -f1 -d' ')"
		detected=1
	fi

	if [ -z "$detected" ] && [ -n "$profile" ] &&
	   file_exists profiles/"@$profile"/SHARED &&
	   ! file_exists profiles/"@$profile"/PROFILE
	then
		: # This is a shared profile, do nothing
	elif [ -n "$profile" ] &&
	     file_exists profiles/"$profile"/PROFILE
	then
		: # This is a machine profile, do nothing
	else
		track_hook profile_not_found "$uuid" || profile=
	fi
}

# Calls the hook to make the final profile selection, includes
# files of the profile if defined, sets the $baremetal variable,
# and calls the hook to check and/or restrict the hardware
#
use_profile()
{
	track_hook select_profile

	if [ -n "$profile" ]; then
		local x p="profiles/@$profile"

		if ! file_exists "$p"/SHARED; then
			p="profiles/$profile"
			file_exists "$p"/PROFILE ||
				fatal "Profile not found: '%s'." "$profile"
			[ -n "$baremetal" ] ||
				baremetal="$(read_file "$p"/PROFILE |
						head -n6 |
						sed -n -e 's/^MACHINE: //p' |
						tail -n1)"

			if file_exists "$p"/PARENT; then
				local msg="Broken parent profile link: %s -> @%s."

				x="$(read_file "$p"/PARENT |head -n1)"
				[ -n "$x" ] && file_exists profiles/"@$x"/SHARED ||
					fatal "$msg" "$profile" "$x"
				p="profiles/@$x"
			fi
		fi

		user_config "$p"/sysrest.ini
		[ -n "$disable_hooks" ] ||
			user_config "$p"/hooks.sh
		profile_subdir="$p"
	fi

	[ -n "$baremetal" ] ||
		baremetal="$(machine_name)"
	track_hook check_hardware
}

