#!/bin/bash
###
### This file is covered by the GNU General Public License
### version 3 or later.
###
### Copyright (C) 2021-2025, ALT Linux Team

#####################################
### Profile data output functions ###
#####################################

# Displays the relative path to an existing profile
#
profile_path()
{
	[ -n "$profile" ] ||
		profile="$(machine_profile)"

	if [ -s "$backup/profiles/@$profile"/SHARED ]; then
		printf "profiles/@%s\n" "$profile"
	elif [ -s "$backup/profiles/$profile"/PROFILE ]; then
		printf "profiles/%s\n" "$profile"
	else
		fatal "Profile '%s' not found." "$profile"
	fi
}

# Displays the profile information itself
#
show_profile()
{
	local x p shared v=0 parent=

	x="$profile"
	[ -n "$profile" ] ||
		profile="$(machine_profile)"
	shared="$backup/profiles/@$profile"
	p="$backup/profiles/$profile"

	if [ -s "$shared"/SHARED ]; then
		[ ! -f "$p"/PROFILE ] && [ ! -f "$p"/PARENT ] &&
		[ ! -f "$shared"/PROFILE ] && [ ! -f "$shared"/PARENT ] ||
			fatal "Invalid profile: '%s'." "$profile"
		p="$shared"
	else
		if [ ! -s "$p"/PROFILE ]; then
			if [ -z "$x" ]; then
				x="$(machine_uuid)"
				baremetal="$(machine_name)"
				printf "Current machine information:\n\n"
				printf "CPUARCH: %s\n" "$(uname -m)"
				printf "MACHINE: %s\n" "$baremetal"
				printf "CLASSID: %s\n" "$x"
				printf "\n"
			fi
			fatal "Profile '%s' not found." "$profile"
		fi
		[ ! -f "$shared"/SHARED ] && [ ! -f "$p"/SHARED ] ||
			fatal "Invalid profile: '%s'." "$profile"
		shared=
	fi

	[ -n "$shared" ] || [ ! -s "$p"/PARENT ] ||
		parent="$(head -n1 -- "$p"/PARENT)"
	[ -z "$parent" ] || [ -s "$backup/profiles/@$parent"/SHARED ] ||
		fatal "Broken parent profile link: %s -> @%s." "$profile" "$parent"
	printf "PROFILE: %s%s\n" "$profile" "${shared:+ (SHARED)}"

	[ -z "$parent" ] ||
		printf "PARENT:  %s\n" "$parent"
	[ -z "$shared" ] && cat -- "$p"/PROFILE ||
		cat -- "$shared"/SHARED
	[ -z "$parent" ] ||
		p="$backup/profiles/@$parent"
	printf "\nOVERLAY-FS: "

	if [ -d "$p"/rootfs ] &&
	   [ -n "$(find -- "$p"/rootfs -not -type d -print -quit)" ]
	then
		printf "Yes (rootfs contents)\n"
	elif [ ! -s "$p"/update.tgz ]; then
		printf "No\n"
	else
		printf "Yes (%s bytes in the archive)\n" \
			"$(stat -c %s -- "$p"/update.tgz)"
	fi

	[ ! -s "$p"/hooks.sh ] ||
		v="$(stat -c %s -- "$p"/hooks.sh)"
	[ ! -s "$p"/chroot-pre.sh ] ||
		v="$(( $(stat -c %s -- "$p"/chroot-pre.sh) + $v ))"
	[ ! -s "$p"/chroot-post.sh ] ||
		v="$(( $(stat -c %s -- "$p"/chroot-post.sh) + $v ))"
	[ "$v" = 0 ] && v=No ||
		v="Yes ($v bytes)"
	printf "USER-HOOKS: %s\n" "$v"
	[ ! -s "$p"/sysrest.ini ] && v=No ||
		v="Yes ($(stat -c %s -- "$p"/sysrest.ini) bytes):"
	printf "PARAMETERS: %s\n" "$v"

	if [ -s "$p"/sysrest.ini ]; then
		printf "\n---[ sysrest.ini:\n"
		cat -- "$p"/sysrest.ini
		printf "]---\n"
	fi
}

