#!/bin/bash
###
### This file is covered by the GNU General Public License
### version 3 or later.
###
### Copyright (C) 2021-2025, ALT Linux Team

####################################
### Command line argument parser ###
####################################

# Displays an error message when using the command line
#
show_usage()
{
	local msg

	if [ "$#" != 0 ]; then
		msg="$1"; shift
		printf "$msg\n" "$@" >&2
	fi

	msg="Invalid command line usage.\nTry '%s -h' for more details."
	fatal "$msg" "$progname"
}

# Checks for the required second argument
#
check_arg()
{
	local msg="After the '%s' option you must specify %s."

	[ -n "$2" ] || show_usage "$msg" "$1" "$3"
}

# Sets the name of the action
#
set_action()
{
	[ -z "$action" ] ||
		show_usage "The action is already specified: '%s'." "$action"
	action="$1"
}

# Displays the version of this program
#
show_version()
{
	local SYSREST_VERSION=ERROR
	local SYSREST_BUILD_DATE=ERROR

	. "$libdir"/version.sh

	printf "%s %s %s\n" "$progname" "$SYSREST_VERSION" "$SYSREST_BUILD_DATE"
	exit 0
}

# Displays a help message for this program
#
show_help()
{
	local help="$libdir/l10n/en_US/profile.msg"

	sed "s/@PROG@/$progname/g" "$help"
	exit 0
}

# Checks and parses command line argumens
#
parse_cmdline()
{
	local l_opts="arch:,check,erase,empty,list,make,path,remove,search"
	      l_opts="$l_opts,show,update,backup:,comment:,date:,baremetal:"
	      l_opts="$l_opts,hardware:,newname:,parent:,uuid:,no-pci"
	      l_opts="$l_opts,instance,version,help"
	local s_opts="+a:B:b:Cc:d:EiLMn:Pp:RSsUu:xVh"

	l_opts=$(getopt -n "$progname" -o "$s_opts" -l "$l_opts" -- "$@") ||
		show_usage
	eval set -- "$l_opts"
	while [ "$#" != 0 ]; do
		case "$1" in
		-C|--check)
			set_action check_profile
			;;
		-E|--erase|--empty)
			set_action erase_profile
			;;
		-L|--list)
			set_action list_profiles
			;;
		-M|--make)
			set_action make_profile
			;;
		-P|--path)
			set_action profile_path
			;;
		-R|--remove)
			set_action remove_profile
			;;
		-S|--search)
			set_action search_profile
			;;
		-s|--show)
			set_action show_profile
			;;
		-U|--update)
			set_action update_profile
			;;

		-a|--arch)
			check_arg --arch "${2-}" "a machine platform"
			cpuarch="$2"
			shift
			;;

		-B|--baremetal|--hardware)
			check_arg --baremetal "${2-}" "hardware name"
			baremetal="$2"
			shift
			;;

		-b|--backup)
			check_arg --backup "${2-}" "backup directory"
			[ -d "$2" ] ||
				show_usage "$msg" "$2"
			backup="$(realpath -- "$2")"
			shift
			;;

		-c|--comment)
			check_arg --comment "${2-}" "a comment string"
			comment="$2"
			shift
			;;

		-d|--date)
			check_arg --date "${2-}" "creation date and time"
			datetime="$(normalize_datetime "$2")"
			[ -n "$datetime" ] ||
				fatal "Invalid date/time specified: '%s'." "$2"
			shift
			;;

		-n|--newname)
			check_arg --newname "${2-}" "new profile name"
			[ "${2:0:1}" != "@" ] && newname="$2" ||
				newname="${2:1}"
			validate_profile_name "$newname" ||
				fatal "Invalid new profile name: '%s'." "$newname"
			shift
			;;

		-p|--parent)
			check_arg --parent "${2-}" "parent profile name"
			validate_profile_name "$2" ||
				fatal "Invalid parent profile name: '%s'." "$2"
			parent="$2"
			shift
			;;

		-u|--uuid)
			check_arg --uuid "${2-}" "machine class ID (UUID)"
			validate_uuid "$2" ||
				fatal "Invalid UUID specified: '%s'." "$2"
			classid="$2"
			shift
			;;

		-i|--instance)
			instance=1
			;;
		-x|--no-pci)
			pci_bus=
			;;
		-V|--version)
			show_version
			;;
		-h|--help)
			show_help
			;;

		--)	shift
			break
			;;
		-*)	show_usage "Unsupported option: '%s'." "$1"
			;;
		*)	break
			;;
		esac
		shift
	done

	# Action required
	[ -n "$action" ] ||
		show_usage "An action must be specified."

	# Remaining arguments
	if [ "$action" = search_profile ]; then
		[ "$#" != 0 ] ||
			show_usage "Search text required."
		search="$*"
	elif [ "$#" -gt 1 ]; then
		show_usage "Too many arguments."
	elif [ "$#" = 1 ]; then
		[ "${1:0:1}" != "@" ] && profile="$1" ||
			profile="${1:1}"
		validate_profile_name "$profile" ||
			fatal "Invalid profile name: '%s'." "$profile"
	fi

	# Use the newname argument as the profile name when creating a profile
	if [ "$action" = make_profile ] && [ -z "$profile" ]; then
		profile="$newname"
		newname=
	fi

	# Require support of the protocol with remote server
	. "$libdir/$backup_proto".sh

	# Path to the machine class catalog
	catalog="$backup/profiles/CATALOG.lst"
}

