#!/bin/bash
###
### This file is covered by the GNU General Public License
### version 3 or later.
###
### Copyright (C) 2021-2025, ALT Linux Team

################################
### Profile update functions ###
################################

# Updates information in the profile
#
update_profile()
{
	local x p newuuid=
	local u f shared h=

	[ -n "$profile" ] ||
		profile="$(machine_profile)"
	shared="$backup/profiles/@$profile"
	p="$backup/profiles/$profile"

	if [ -s "$shared"/SHARED ]; then
		[ ! -f "$p"/PROFILE ] && [ ! -f "$p"/PARENT ] &&
		[ ! -f "$shared"/PROFILE ] && [ ! -f "$shared"/PARENT ] ||
			fatal "Invalid profile: '%s'." "$profile"
		[ -z "${comment}${datetime}" ] || h=1
	else
		[ -s "$p"/PROFILE ] ||
			fatal "Profile '%s' not found." "$profile"
		[ ! -f "$shared"/SHARED ] && [ ! -f "$p"/SHARED ] ||
			fatal "Invalid profile: '%s'." "$profile"
		[ -z "${cpuarch}${baremetal}${classid}${comment}${datetime}" ] ||
			h=1
		shared=
	fi

	# Checking the action that needs to be performed
	if [ -z "$h" ] && [ -z "$newname" ]; then
		fatal "There is nothing to update."
	elif [ -n "$newname" ]
	then
		# Also checking the renaming option
		[ "$newname" != "$profile" ] ||
			fatal "A profile cannot be renamed to itself."
		[ ! -d "$backup/profiles/$newname" ] &&
		[ ! -d "$backup/profiles/@$newname" ] ||
			fatal "The profile's target directory already exists."
	fi

	# Update some header fields
	if [ -n "$h" ] && [ -n "$shared" ]
	then
		__read_profile()
		{
			head -n3 -- "$shared"/SHARED |
				sed -n -e "s/^$(safe_regex "$1"): //p" |
				tail -n1
		}

		[ -n "$comment" ] ||
			comment="$(__read_profile COMMENT)"
		[ -n "$datetime" ] ||
			datetime="$(__read_profile CREATED)"
		mv -f -- "$shared"/SHARED "$shared"/SHARED.bak
		unset __read_profile

		( cat <<-EOF
			VERSION: 1.0
			COMMENT: $comment
			CREATED: $datetime
			EOF
		  sed -n '4,$p' "$shared"/SHARED.bak
		) >"$shared"/SHARED

		rm -f -- "$shared"/SHARED.bak
	elif [ -n "$h" ] && [ -z "$shared" ]
	then
		__read_profile()
		{
			head -n6 -- "$p"/PROFILE |
				sed -n -e "s/^$(safe_regex "$1"): //p" |
				tail -n1
		}

		[ -n "$cpuarch" ] ||
			cpuarch="$(__read_profile CPUARCH)"
		[ -n "$baremetal" ] ||
			baremetal="$(__read_profile MACHINE)"
		[ -n "$comment" ] ||
			comment="$(__read_profile COMMENT)"
		[ -n "$datetime" ] ||
			datetime="$(__read_profile CREATED)"
		newuuid="$classid"
		classid="$(__read_profile CLASSID)"
		mv -f -- "$p"/PROFILE "$p"/PROFILE.bak
		unset __read_profile

		( cat <<-EOF
			VERSION: 1.0
			CPUARCH: $cpuarch
			MACHINE: $baremetal
			CLASSID: ${newuuid:-$classid}
			COMMENT: $comment
			CREATED: $datetime
			EOF
		  sed -n '7,$p' "$p"/PROFILE.bak
		) >"$p"/PROFILE

		rm -f -- "$p"/PROFILE.bak

		# Update link in the catalog
		if [ -n "$newuuid" ] && [ "$newuuid" != "$classid" ]
		then
			[ ! -s "$catalog" ] && :>"$catalog" ||
				sed -i -e "/^${classid//-/} /d" "$catalog"
			printf "%s %s\n" "${newuuid//-/}" \
					"${newname:-profile}" >>"$catalog"
		fi
	fi

	# Rename profile
	if [ -n "$newname" ]; then
		if [ -n "$shared" ] && [ -s "$catalog" ]
		then
			# Update all links to this shared profile
			sed -n '5,$p' "$shared"/SHARED |
				sed -n -e 's/^USED\-BY: //p' |
				cut -f1 -d' ' |
			while read -r h; do
				u="$(sed -n -e "s/^${h//-/} //p" "$catalog" |
						tail -n1 |
						cut -f1 -d' ')"
				f="$backup/profiles/$u/PARENT"
				x="$(head -n1 -- "$f" 2>/dev/null ||:)"
				[ "$x" = "$profile" ] ||
					continue
				printf "%s\n" "$newname" >"$f"
			done
		fi

		# Update link in catalog
		if [ -z "$newuuid" ] && [ -s "$catalog" ]; then
			x="$(safe_regex "$newname")"
			u="$(safe_regex "$profile")"
			sed -i -e "s/ $u$/ $x/g" "$catalog"
		fi

		# Rename and display profile directory
		if [ -n "$shared" ]; then
			mv -f -- "$shared" "$backup/profiles/@$newname"
			printf "profiles/@%s\n" "$newname"
		else
			mv -f -- "$p" "$backup/profiles/$newname"
			printf "profiles/%s\n" "$newname"
		fi
	fi
}

