#!/bin/bash
###
### This file is covered by the GNU General Public License
### version 3 or later.
###
### Copyright (C) 2021-2025, ALT Linux Team

##########################################
### Backup data verification functions ###
##########################################

# Default exit handler
#
exit_handler()
{
	local rv="$?"

	trap - EXIT; cd /
	[ -z "$quiet" ] ||
		exec 1>&3 3>&-
	[ -z "$workdir" ] || [ ! -d "$workdir" ] ||
		run rm -rf $verbose --one-file-system -- "$workdir"
	quiet=

	if [ -n "$alterator" ]; then
		sleep 1 && notify_alterator message "Done."
	elif [ -n "$frontend_pipe" ]; then
		sleep 1 && notify_frontend I "Done."
	fi

	exit "$rv"
}

# Specifies the options for restoring from a backup
#
set_recovery_options()
{
	# Create temporary working directory
	workdir="$(run mktemp -dt -- "$progname-XXXXXXXX.tmp")" ||
		fatal "Unable to create working directory."
	trap exit_handler EXIT

	# System restore options
	user_config sysrest.ini

	# User-defined hooks, if present and enabled
	[ -n "$disable_hooks" ] || user_config hooks.sh

	# Check for the presence of the required archives
	file_exists META.tgz && file_exists root.tgz ||
		fatal "Archives '%s' and '%s' required." META.tgz root.tgz

	# Deterimate profile name
	detect_profile

	# Check the hardware and apply the selected profile
	use_profile

	# Set target rootfs version
	release="${release:-1.0}"

	# Default target hostname
	computer="${computer:-host}"

	# Data partition mount point
	if file_exists home.tgz; then
		datapart_mp=/home
	elif file_exists var.tgz; then
		datapart_mp=/var
	fi

	# Archive required for /var
	if [ "$datapart_mp" = /var ]; then
		file_exists var.tgz ||
			fatal "The archive '%s' required." var.tgz
	else
		# Another valid choice is /home
		[ "$datapart_mp" = /home ] ||
			fatal "Unsupported choice: %s='%s'." \
					datapart_mp "$datapart_mp"
	fi

	# Include the selected disk partitioner
	if [ -z "$disable_hooks" ] && file_exists "$partitioner.sh"; then
		user_config "$partitioner.sh"
	else
		. "$libdir/$partitioner.sh"
	fi
}

# Checks backup files and backup metadata
#
check_backup_files()
{
	local dummy

	# Unpack and check metadata
	msg "Checking backup metadata..."
	read_file META.tgz |run unpigz -qnc |
		run tar -xpf - $verbose -C "$workdir" ||
			fatal "Error unpacking archive '%s'." META.tgz
	dummy="$(head -n1 -- "$workdir/VERSION" 2>/dev/null ||:)"
	#
	case "$dummy" in
	0.[1-9]*)
		# It's OK
		;;
	*)	fatal "Unsupported backup version: '%s'." "$dummy"
		;;
	esac
	#
	[ -s "$workdir/ARCH"    ] &&
	[ -s "$workdir/FSTABLE" ] &&
	[ -s "$workdir/LOADERS" ] &&
	[ -s "$workdir/RELEASE" ] &&
	[ -s "$workdir/RNDSEED" ] &&
	[ -s "$workdir/TARGETS" ] &&
	[ -s "$workdir/VOLUMES" ] &&
	[ -s "$workdir/blkid.tab" ] ||
		fatal "Invalid or unsupported backup metadata was found."
	#
	# Check the machine platform
	dummy="$(run head -n1 -- "$workdir"/ARCH)"
	[ "$(run uname -m)" = "$dummy" ] ||
		fatal "This backup is only for platform: %s." "$dummy"

	# Try to determinate the hostname inside the backup
	[ -n "$template" ] || [ ! -s "$workdir/ORGHOST" ] ||
		template="$(run head -n1 -- "$workdir/ORGHOST")"
	template="${template:-computername}"

	# Finish backup checking
	if [ "$action" = chkconf ]; then
		msg "Backup verified successfully!"
		exit 0
	elif [ "$action" = validate ]; then
		. "$libdir"/restore/validate.sh
		validate_backup_images
		exit 0
	fi
}

# Hook: changes default mount options for SSD/NVMe acceleration
#
setup_ssd_options()
{
	esp_opts="$esp_opts,discard"
	rootopts="${rootopts//relatime/noatime},discard"
	bootopts="${bootopts//relatime/noatime},discard"
	dataopts="${dataopts//relatime/noatime},discard"
}

# Hook: sets automatic size for SWAP partition
#
auto_swap_size()
{
	swapsize="$(run sed -n -E 's/^MemTotal:\s+//p' /proc/meminfo |
						cut -f1 -d' ')"
	swapsize="$(( $swapsize / 1024 / 1024 + 1 ))"
	[ "$swapsize" -gt 4 ] ||
		swapsize="$(( $swapsize + $swapsize ))"
	[ "$swapsize" -le 32 ] ||
		swapsize=32
	[ "$swapsize" != 0 ] ||
		swapsize=1
	swapsize="${swapsize}G"
}

# Defines the final configuartion for deployment
#
deployment_config()
{
	local major minor dummy
	local pcdos_limit target_size

	# Determine the mke2fs version: new ext4 features such as -O 64bit
	# and -O metadata_csum are enabled by default since version 1.43,
	# they require appropriate kernel support, and for Linux 3.x
	# kernels they should be disabled when calling mke2fs
	#
	if [ -n "$old_ext4_boot" ]; then
		will_be_run mke2fs -V
		dummy="$(mke2fs -V 2>&1 |head -n1 |cut -f2 -d' ')"
		major="$(echo "$dummy"  |cut -f1 -d.)"
		minor="$(echo "$dummy"  |cut -f2 -d.)"

		if is_number "$major" && is_number "$minor"; then
			if [ "$major" = 1 ] && [ "$minor" -lt 43 ]; then
				old_ext4_boot=
			elif [ "$major" -lt 1 ]; then
				old_ext4_boot=
			fi
		fi
	fi

	# Optimization for target SSD/NVMe drives
	dummy="$(run lsblk -d -n -o TRAN -- "$target" |
			tr '[[:upper:]]' '[[:lower:]]')"
	case "$dummy" in
	nvme)	setup_ssd_options
		;;
	*)	dummy="$(run lsblk -d -n -o ROTA -- "$target" |
					sed -E 's/^\s*//')"
		[ -z "$dummy" ] || [ "$dummy" = 1 ] ||
			setup_ssd_options
		;;
	esac

	# Determine the most suitable partitioning scheme
	if [ -n "$force_gpt" ] && [ -n "$force_mbr" ]; then
		# using mutually exclusive options at the same time is prohibited
		warn "Ignoring force_mbr because force_gpt is set!"
		force_mbr=
	fi
	#
	pcdos_limit="$(human2size 2T)"
	target_size="$(get_disk_size "$target")"
	#
	if [ "$target_size" -le "$pcdos_limit" ]; then
		# if the target disk size is <= 2 TiB,
		# then DOS/MBR partitioning is allowed
		[ -n "$force_gpt" ] || [ -n "$uefiboot" ] || pt_scheme=dos
		[ -z "$force_mbr" ] || pt_scheme=dos
	else
		# if the target disk size is > 2 TiB:

		# 1) DOS/MBR partitioning is not available
		[ -z "$force_mbr" ] ||
			fatal "Unable to create DOS/MBR label on disk larger than 2 TiB."
		pt_scheme=gpt

		# 2) On Elbrus we still use GUID/GPT scheme
		if [ "$platform" = elbrus ]; then
			: # Do nothing by default

		# 3) Legacy BIOS in Legacy/CSM mode may not boot
		elif [ -z "$uefiboot" ] && [ "$platform" = x86_64 ] ||
		     [ "$platform" = i586 ]
		then
			warn "Legacy BIOS may not boot from a disk larger than 2 TiB!"
		fi
	fi
	#
	if [ -n "$force_esp" ] || [ -n "$uefiboot" ]; then
		# ESP partition is required for UEFI boot
		esp_size="${esp_size:-256M}"
	else
		# in other cases, it is not necessary to create it
		esp_size=
	fi
	#
	if [ "$pt_scheme" = dos ]; then
		# with DOS/MBR partitioning the BBP is not needed
		bbp_size=
	elif [ "$platform" != x86_64 ] && [ "$platform" != i586 ]; then
		# on non-Intel platforms, BBP is also not needed
		bbp_size=
	elif [ -z "$uefiboot" ] || [ -n "$biosboot_too" ]; then
		# Legacy/CSM boot requires BBP in GUID/GPT table
		bbp_size="${bbp_size:-1M}"
	fi
	#
	if [ "$platform" = aarch64 ]; then
		# on aarch64 there is no need for a second boot in PC/DOS style
		biosboot_too=
	elif [ "$platform" = elbrus ]; then
		# Elbrus requires a separate /boot partition
		bootsize="${bootsize:-512M}"
		biosboot_too=
		force_esp=
		esp_size=
	elif [ "$platform" = ppc64le ]; then
		# IBM Power requires a separate PReP partition
		prepsize="${prepsize:-4M}"
		biosboot_too=
		force_esp=
		esp_size=
	fi

	# Calculating the SWAP partition size
	if [ "$swapsize" = AUTO ]; then
		track_hook auto_swap_size
		[ "$swapsize" != AUTO ] || swapsize=
	fi
}

# Unpacks the specified backup archive to $destdir
#
unpack_image()
{
	local archive="$1" subdir="${2-}"
	local fsize value numeric="-n -i 1"

	__unpack_image_inner()
	{
		read_file  "$archive.tgz" |
		( pv $numeric -s "$fsize" |
		  unpigz -qnc |
		  tar -xpSf - --overwrite -C "${destdir}${subdir}" ||
				printf "%s, tar\n" "$?" >>"$workdir"/ERROR
		) 2>&1
	}

	run rm -f $verbose -- "$workdir"/ERROR
	fsize="$(get_file_size "$archive.tgz")"
	dbg "%s.tgz (%s bytes) will be unpacked" "$archive" "$fsize"
	will_be_run tar -xpSf - --overwrite -C "${destdir}${subdir}"

	if [ -n "$alterator" ]; then
		notify_alterator message "%["
		__unpack_image_inner |
		while read -r value; do
			is_number "$value" ||
				continue
			notify_alterator message "% $value"
			printf "\r%s%%" "$value"
		done
		printf "\r"
		notify_alterator message "%]"

	elif [ -n "$frontend_pipe" ]; then
		notify_frontend I "%s" "%["
		__unpack_image_inner |
		while read -r value; do
			is_number "$value" ||
				continue
			notify_frontend '%' "%s" "$value"
			printf "\r%s%%" "$value"
		done
		printf "\r"
		notify_frontend I "%s" "%]"

	else
		numeric=
		__unpack_image_inner
	fi

	unset __unpack_image_inner
	[ -s "$workdir"/ERROR ] ||
		return 0
	value="$(head -n1 -- "$workdir"/ERROR)"

	case "$value" in
	*", tar")
		fatal "Unable to unpack archive '%s.tgz'." "$archive"
		;;
	*", curl")
		fatal "Error loading archive '%s.tgz'." "$archive"
		;;
	esac

	fatal "Error reading archive '%s.tgz'." "$archive"
}

# Reads kernel arguments from the original rootfs
#
read_kernel_args()
{
	local i v r=

	v="$(sed -n -e 's/^GRUB_CMDLINE_LINUX_DEFAULT=//p' "$1")"
	eval v=$v
	[ -n "$v" ] ||
		return 0

	for i in $v _; do
		case "$i" in
		resume=*|_)
			# Just skip it
			;;
		*)	r="${r:+$r }$i"
			;;
		esac
	done

	printf "%s" "$r"
}

