#!/bin/bash
###
### This file is covered by the GNU General Public License
### version 3 or later.
###
### Copyright (C) 2021-2025, ALT Linux Team

###########################################
### Helper script running inside chroot ###
###########################################

# Safety first
set -o errexit
set -o noglob
set -o nounset
set -o errtrace

# Working directory
[ -d "${TMPDIR-}" ] ||
	TMPDIR="$(realpath -- "${0%/*}")"
export TMPDIR

# Workaround to ignore failure in internal.sh
progname=

# Bootstrap
. "$TMPDIR"/defaults.sh
. "$TMPDIR"/internal.sh
. "$TMPDIR"/common.sh
. "$TMPDIR"/logger.sh

# User-defined parameters
[ ! -s "$TMPDIR"/sysrest.ini ] ||
	. "$TMPDIR"/sysrest.ini

# Chroot parameters
. "$TMPDIR"/chroot.ini

# Let this be a constant for now
readonly homedir=/home

# Catch all unexpected errors
trap 'unexpected_error "${BASH_SOURCE[0]##*/}" "$LINENO"' ERR


A_msg()
{
	msg "The grub-efi bootloader for %s is already installed!" "$platform"
}

BIOS_install()
{
	msg "Installing grub-pc for Legacy/CSM boot mode..."
	run grub-install --target=i386-pc --recheck $verbose \
			 --boot-directory=/boot -- "$target"
}

ESP_install()
{
	local tgt="$1"

	msg "Installing grub-efi for UEFI boot mode..."
	run grub-install --target="$tgt" --recheck $verbose   \
			 --efi-directory=/boot/efi --no-nvram \
			 --boot-directory=/boot -- "$target"
}

UEFI_install()
{
	local tgt="$1"

	if [ -n "$HAVE_ESP" ]; then
		if [ -s /boot/grub/"$tgt"/core.efi ] &&
		   [ -s /boot/efi/EFI/"$bootldr_id"/grub.cfg ]
		then
			A_msg
		else
			ESP_install "$tgt"
		fi
	fi
}

PReP_install()
{
	msg "Installing grub-ieee1275 for IBM Power..."
	run grub-install --target=powerpc-ieee1275 $verbose \
			 --recheck${no_nvram:+ --no-nvram}  \
			 --boot-directory=/boot -- "$target"
}

track_hook()
{
	local rc=0 hook="$1"

	[ -s "$TMPDIR/$hook.sh" ] ||
		return 0
	log "Entering hook '%s'..." "$hook"
	. "$TMPDIR/$hook.sh" || rc="$?"
	log "Leaving hook '%s'..." "$hook"

	return "$rc"
}

chroot_exit_handler()
{
	local rc="$?"

	trap - EXIT; run cd /
	msg "Unmounting pseudo fs..."
	run umount $quiet $verbose -- /dev/pts ||
		run umount -fl $quiet $verbose -- /dev/pts
	run umount $quiet $verbose -- /sys ||
		run umount -fl $quiet $verbose -- /sys
	run umount $quiet $verbose -- /proc ||
		run umount -fl $quiet $verbose -- /proc
	exit "$rc"
}


# Entry point
msg "Mounting pseudo fs..."
run mount -t proc $verbose -- none /proc
run mount -t sysfs $verbose -- none /sys
run mount -t devpts $verbose -- none /dev/pts

# Setup an exit handler
trap chroot_exit_handler EXIT

# Check the target platform
detect_hw_platform

# Use user-defined hook
track_hook chroot-pre

# Remove the specified packages
if [ -n "$remove_packages" ]; then
	msg "Removing unnecessary packages..."
	run apt-get remove $quiet -y --force-yes --purge -- $remove_packages
fi
list2=

# Install the specified packages
if [ -n "$install_packages" ] &&
   [ -d /var/cache/apt/archives ]
then
	run cd /var/cache/apt/archives/
	list1="$(run find . -maxdepth 1 -type f \
			-name '*.rpm' -printf '%f\n')"

	for u in $list1 _; do
		[ "$u" != _ ] && [ -s ./"$u" ] ||
			continue
		for x in $install_packages; do
			case "$u" in
			$x)	list2="$list2 ./$u"
				break
				;;
			esac
		done
		unset x
	done

	if [ -n "$list2" ]; then
		msg "Installing the specified packages..."
		run apt-get install $quiet -y --force-yes -- $list2
		run apt-get clean
	fi

	unset list1
	run cd - >/dev/null
fi

# Remove unnecessary packages
[ -z "$remove_packages" ] && [ -z "$list2" ] ||
	run apt-get autoremove $quiet -y --force-yes
unset list2

# Update the OEM Setup wizard step list
if [ -s "$TMPDIR"/OEM-steps ] && [ -d /etc/alterator-setup ]; then
	run mv -f $verbose -- "$TMPDIR"/OEM-steps /etc/alterator-setup/steps
fi

# Delete specified users
if [ -n "$remove_users" ]; then
	msg "Removing specified users..."

	# TO DO: check last user in /var/lib/ldm/,
	# /var/cache/lightdm/, /var/lib/lightdm-data,
	# ex.: /var/lib/ldm/.config/state-kde::lastUser
	#
	for u in $remove_users; do
		id -- "$u" >/dev/null 2>&1 ||
			continue
		run userdel -r -f -- "$u"
		run rm -rf $verbose -- "$homedir/$u"
	done
fi

# Change root password
if [ -n "$root_password" ] && have_binary chpasswd; then
	msg "Changing root password..."
	printf "root:%s\n" "$root_password" |run chpasswd
fi

# Change passwords for unprivileged users
if [ -n "$user_password" ] && have_binary chpasswd; then
	msg "Changing passwords for unprivileged users..."
	list2="$(run find "$homedir"/ -mindepth 1 -maxdepth 1 \
			-type d -not -name 'lost+found' -printf '%f\n')"

	for u in $list2 _; do
		[ "$u" != _ ] ||
			continue
		id -- "$u" >/dev/null 2>&1 ||
			continue
		printf "%s:%s\n" "$u" "$user_password" |run chpasswd
	done

	unset list2
fi

# Cleanup home directories
if [ -n "$cleanup_homedir" ] && [ -d /etc/skel ]; then
	msg "Re-creating user home directories..."

	for u in $cleanup_homedir; do
		id -- "$u" >/dev/null 2>&1 ||
			continue
		uid="$(run id -u -- "$u")"
		run rm -rf $verbose -- "$homedir/$u"
		run cp -af $verbose -- /etc/skel "$homedir/$u"
		run chmod -R -- g-w,o-rwx "$homedir/$u"
		run chmod -- 0700 "$homedir/$u"
		run chown -R -- "$uid":"$uid" "$homedir/$u"
		unset uid
	done
fi

# Create specified users
if [ -n "$create_users" ]; then
	msg "Creating specified users..."
	list2="$(run find "$homedir"/ -mindepth 1 -maxdepth 1 \
			-type d -not -name 'lost+found' -printf '%f\n')"

	if [ -z "$default_groups" ] && [ -n "$list2" ]; then
		for u in $list2 _; do
			[ "$u" != _ ] ||
				continue
			id -- "$u" >/dev/null 2>&1 && break || u=
		done

		if [ -n "$u" ] && [ "$u" != _ ]; then
			u="$(run groups -- "$u" |
				sed -n "s/^$u : //p" |
				tr ' ' '\n' |
				grep -vE "^($u|wheel)$" |
				tr '\n' ',' |
				sed -e 's/\,$//')"
			default_groups="$u"
		fi
	fi

	uid_min=
	[ ! -s /etc/login.defs ] ||
		uid_min="$(run sed -n -E 's/^UID_MIN\s+//p' /etc/login.defs)"
	[ -z "$uid_min" ] ||
		uid_min=1000

	for u in $create_users; do
		if [ -n "${u##*:*}" ]; then
			p=
		else
			p="${u#*:}"
			u="${u%%:*}"
		fi

		if ! id -- "$u" >/dev/null 2>&1; then
			run rm -rf $verbose -- "$homedir/$u"
			run useradd -K UID_MIN=$uid_min -s /bin/bash -d "$homedir/$u" \
					-m${default_groups:+ -G "$default_groups"} -- "$u"
			[ -z "$p" ] || ! have_binary chpasswd ||
				printf "%s:%s\n" "$u" "$p" |run chpasswd
		fi

		unset p
	done

	unset uid_min list2
fi
unset u

# Hardware specialization
msg "Binding the installation to the hardware..."
for flavour in $kernel_flavours _; do
	[ -d /lib/modules ] && [ -d /boot ] ||
		break
	[ "$flavour" != _ ] ||
		continue

	for kver in $(run ls -1 /lib/modules/ |sort -Vr) _; do
		[ -z "${kver##*-$flavour-*}" ] ||
			continue
		[ -s "/lib/modules/$kver/modules.alias" ] ||
			continue
		run make-initrd $verbose --kernel="$kver"
		[ -e /boot/initrd.img ] ||
			run ln -snf $verbose -- \
				"initrd-$kver.img" /boot/initrd.img
		[ -e /boot/"initrd-$flavour.img" ] ||
			run ln -snf $verbose -- \
				"initrd-$kver.img" /boot/"initrd-$flavour.img"
	done

	unset kver
done
unset flavour

# Install bootloader
case "$platform" in
aarch64)
	UEFI_install arm64-efi
	run update-grub
	;;
i586)
	[ -z "$HAVE_BIOS" ] ||
		BIOS_install
	run update-grub
	;;
ppc64le)
	[ -z "$HAVE_PReP" ] ||
		PReP_install
	run update-grub
	;;
x86_64)
	[ -z "$HAVE_BIOS" ] ||
		BIOS_install
	UEFI_install x86_64-efi
	run update-grub
	;;
esac

# Disable systemd services
[ -z "$disable_services" ] ||
	run systemctl $quiet disable $disable_services

# Enable systemd services
[ -z "$enable_services" ] ||
	run systemctl $quiet enable $enable_services

# Remove "Suspend" button
[ -z "$no_suspend" ] ||
	run systemctl $quiet mask suspend.target

# Remove "Hibernate" button
[ -z "$no_hibernate" ] ||
	run systemctl $quiet mask hibernate.target

# Remove old and generate new SSH host keys
if [ -d /etc/openssh ] && [ -z "$keep_ssh_host_keys" ]
then
	run find /etc/openssh -maxdepth 1 -type f \
			-name 'ssh_host_*_key.pub' -or \
			-name 'ssh_host_*_key' -delete ||:

	if have_binary ssh-keygen; then
		run ssh-keygen -A${quiet:+ -q}${verbose:+ -vvv}
	fi
fi

# Use user-defined hook
track_hook chroot-post

# Remove old Linux kernels with modules
if [ -n "$remove_kernels" ] && [ -d /lib/modules ]; then
	msg "Removing old Linux kernels with modules..."
	d=

	for k in $remove_kernels; do
		d="$(find /lib/modules/ -mindepth 1 -maxdepth 1 \
				-type d -name "$k" -printf '%f\n')"
		[ -n "$d" ] ||
			continue
		k="$(rpm -qf -- /lib/modules/"$d" 2>/dev/null ||:)"
		[ -n "$k" ] ||
			continue
		modules="$(rpm -e --test -- "$k" 2>&1 |
				grep ' is needed by (installed) ' |
				awk '{print $8;}' |
				tr '\n' ' ')"
		run rpm -e $quiet $verbose -- $modules $k
		run rm -rf $verbose -- /lib/modules/"$d"
		unset modules
	done

	unset k d
fi

