#!/bin/bash
###
### This file is covered by the GNU General Public License
### version 3 or later.
###
### Copyright (C) 2021-2025, ALT Linux Team

####################################
### Command line argument parser ###
####################################

# Displays an error message when using the command line
#
show_usage()
{
	local msg

	if [ "$#" != 0 ]; then
		msg="$1"; shift
		printf "$msg\n" "$@" >&2
	fi

	msg="Invalid command line usage.\nTry '%s -h' for more details."
	fatal "$msg" "$progname"
}

# Checks for the required second argument
#
check_arg()
{
	local msg="After the '%s' option you must specify %s."

	[ -n "$2" ] || show_usage "$msg" "$1" "$3"
}

# Sets the name of the action
#
set_action()
{
	[ -z "$action" ] ||
		show_usage "The action is already specified: '%s'." "$action"
	action="$1"
}

# Action holder
#
no_action()
{
	set_action "$1"
	fatal "This action is not yet implemented: '%s'." "$1"
}

# Displays the version of this program
#
show_version()
{
	local SYSREST_VERSION=ERROR
	local SYSREST_BUILD_DATE=ERROR

	. "$libdir"/version.sh

	printf "%s %s %s\n" "$progname" "$SYSREST_VERSION" "$SYSREST_BUILD_DATE"
	exit 0
}

# Displays a help message for this program
#
show_help()
{
	local help="$libdir/l10n/en_US/restore.msg"

	sed "s/@PROG@/$progname/g" "$help"
	exit 0
}

# Checks and parses command line argumens
#
parse_cmdline()
{
	local msg=
	local l_opts="check-only,check-conf,checksums,scan-only,deploy"
	      l_opts="$l_opts,full,system,validate,fullrest,system-only"
	      l_opts="$l_opts,alterator,frontend:,verbose,quiet,no-hooks"
	      l_opts="$l_opts,reboot,poweroff,backup:,profile:,removable"
	      l_opts="$l_opts,exclude:,logfile:,no-colors,no-syslog"
	      l_opts="$l_opts,instance,no-log,no-pci,version,help"
	local s_opts="+cCDFMSTab:f:il:p:Ps:qrRvx:Vh"

	l_opts=$(getopt -n "$progname" -o "$s_opts" -l "$l_opts" -- "$@") ||
		show_usage
	eval set -- "$l_opts"
	while [ "$#" != 0 ]; do
		case "$1" in
		-C|--check-only|--check-conf)
			set_action chkconf
			;;
		-D|--deploy)
			set_action deploy
			;;
		-F|--full|--fullrest)
			no_action fullrest
			;;
		-M|--checksums|--validate)
			set_action validate
			;;
		-S|--system|--system-only)
			no_action sysrest
			;;
		-T|--scan-only)
			set_action scandisk
			use_backup=
			;;
		-V|--version)
			show_version
			;;
		-h|--help)
			show_help
			;;

		-b|--backup)
			check_arg --backup "${2-}" "backup directory or storage"
			case "${2-}" in
			http://*)
				backup="$2"
				backup_proto=http
				;;
			file://*)
				msg="Backup directory not found: '%s'."
				[ -d "${2:7}" ] ||
					show_usage "$msg" "${2:7}"
				backup="$(realpath -- "${2:7}")"
				backup_proto=file
				;;
			*) # local filesystem too
				msg="Backup directory not found: '%s'."
				[ -d "$2" ] ||
					show_usage "$msg" "$2"
				backup="$(realpath -- "$2")"
				backup_proto=file
				;;
			esac
			shift
			;;

		-f|--frontend)
			check_arg --frontend "${2-}" "pipe with frontend"
			frontend_pipe="$2"
			shift
			;;

		-l|--logfile)
			if [ -z "$2" ]; then
				logfile=
			else
				msg="Invalid path to the log file: '%s'."
				[ -d "${2%/*}" ] ||
					show_usage "$msg" "$2"
				[ -n "${2##*/*}" ] && logfile="$(realpath .)/$2" ||
					logfile="$(realpath -- "${2%/*}")/${2##*/}"
			fi
			shift
			;;

		-p|--profile)
			check_arg --profile "${2-}" "profile name"
			[ "${2:0:1}" != "@" ] && profile="$2" ||
				profile="${2:1}"
			check_arg --profile "$profile" "profile name"
			shift
			;;

		-s|--syslog)
			logprio="$2"
			shift
			;;

		-x|--exclude)
			check_arg --exclude "${2-}" "device or mount point"
			msg="The value '%s' must be device or mount point."
			if [ -b "$2" ]; then
				protected_devices="$protected_devices $2"
			elif mountpoint -q -- "$2"; then
				protected_mpoints="$protected_mpoints $2"
			else
				show_usage "$msg" --exclude=
			fi
			shift
			;;

		--no-colors)
			CLR_NORM=
			CLR_BOLD=
			CLR_LC1=
			CLR_LC2=
			CLR_OK=
			CLR_ERR=
			CLR_WARN=
			;;

		-a|--alterator)
			alterator=1
			;;
		-i|--instance)
			instance=1
			;;
		-P|--poweroff)
			finalact="poweroff"
			;;
		-q|--quiet)
			quiet="--quiet"
			;;
		-r|--removable)
			use_removable=1
			;;
		-R|--reboot)
			finalact="reboot"
			;;
		-v|--verbose)
			verbose="--verbose"
			;;
		--no-hooks)
			disable_hooks=1
			;;
		--no-log)
			logfile=
			;;
		--no-pci)
			pci_bus=
			;;
		--no-syslog)
			logprio=
			;;

		--)	shift
			break
			;;
		-*)	show_usage "Unsupported option: '%s'." "$1"
			;;
		*)	break
			;;
		esac
		shift
	done

	# Action required
	[ -n "$action" ] ||
		show_usage "An action must be specified."
	msg="%s: the target should be an existing block special device."

	# Optional target
	if [ "$#" = 1 ]; then
		[ -b "$1" ] ||
			show_usage "$msg" "$1"
		if [ "$action" != scandisk ]; then
			target="$1"
		else
			warn "The target will be ignored with the --scan-only action."
			target=
		fi
	elif [ "$#" != 0 ]; then
		show_usage "Too many arguments."
	fi

	# Ignoring the final action in some cases
	case "$action" in
	deploy|fullrest|sysrest)
		;;
	*)	finalact=
		;;
	esac

	# Require support of the protocol with remote server
	. "$libdir/$backup_proto".sh

	# Suppress output in quiet mode
	[ -z "$quiet" ] || exec 3>&1 1>/dev/null
}

