#!/bin/bash
###
### This file is covered by the GNU General Public License
### version 3 or later.
###
### Copyright (C) 2021-2025, ALT Linux Team

############################################
### Supplemental code for system restore ###
############################################

# Shows the standard Deployment System banner
#
__welcome_text()
{
	printf "\n\n${CLR_LC2}%s${CLR_NORM} started" "$product_title"
	printf " [${CLR_BOLD}STICK-v%s${CLR_NORM}]\n" "$release"
}

# Hook: just shows a banner
#
welcome_text()
{
	__welcome_text
}

# Shows diagnostic message before starting the process
#
show_diag_msg()
{
	local x="GUID/GPT"

	[ "$pt_scheme" = gpt ] || x="DOS/MBR"

	printf "\n"
	printf "Computer:    %s\n" "$computer"
	printf "Target disk: %s (${CLR_BOLD}%s${CLR_NORM})\n" "$target" "$diskinfo"
	printf "System time: %s\n" "$(env LC_TIME=C date +'%F %H:%M %Z')"
	printf "Partitioner: %s (%s)\n" "$partitioner" "$x"

	if [ -n "$uefiboot" ]; then
		printf "Boot mode:   UEFI%s\n" "${biosboot_too:+ + Legacy/CSM}"
	else
		printf "Boot mode:   Legacy/CSM%s\n" "${esp_part:+ + ESP-ready}"
	fi

	[ -z "$profile"    ] ||
		printf "Profile:     ${CLR_LC2}%s${CLR_NORM}\n" "$profile"
	[ -z "$baremetal"  ] ||
		printf "Bare metal:  Yes (${CLR_BOLD}%s${CLR_NORM})\n" "$baremetal"
	[ -z "$hypervisor" ] ||
		printf "Hypervisor:  Yes (${CLR_BOLD}%s${CLR_NORM})\n" "$hypervisor"
	[ -z "$preppart"   ] ||
		printf "PReP size:   %s (%s)\n" "$prepsize" "$preppart"
	[ -z "$esp_part"   ] ||
		printf "ESP size:    %s (%s)\n" "$esp_size" "$esp_part"
	[ -z "$bbp_part"   ] ||
		printf "BBP size:    %s (%s)\n" "$bbp_size" "$bbp_part"
	[ -z "$bootpart"   ] ||
		printf "BOOT size:   %s (%s)\n" "$bootsize" "$bootpart"
	[ -z "$swappart"   ] ||
		printf "SWAP size:   %s (%s)\n" "$swapsize" "$swappart"
	if [ -z "$datapart" ]; then
		printf "ROOT size:   * (%s)\n" "$rootpart"
	else
		printf "ROOT size:   %s (%s)\n" "$rootsize" "$rootpart"
		printf "DATA size:   * (%s)\n" "$datapart"
	fi

	printf "\n"
	printf "All devices:\n"
	run lsblk -f
	printf "\n"
}

# Clears NVRAM in UEFI boot mode before chroot
#
clear_nvram()
{
	__clear_nvram_inner()
	{
	  local bootnum junk

	  run efibootmgr -q -D ||:
	  run efibootmgr -q -N ||:
	  run efibootmgr |
		run grep -i -E "^Boot[0-9A-F][0-9A-F][0-9A-F][0-9A-F]" |
	  while read bootnum junk; do
		run efibootmgr -q -B -b "${bootnum:4:4}" ||:
	  done
	  run efibootmgr -q -O ||:
	}

	run mountpoint -q /sys/firmware/efi/efivars ||
		run mount -t efivarfs efivarfs /sys/firmware/efi/efivars

	if [ -n "$SYSREST_DEBUG" ]; then
		__clear_nvram_inner >/dev/null
	else
		__clear_nvram_inner >/dev/null 2>&1
	fi

	unset __clear_nvram_inner
}

# Returns the full path to the EFI binary inside the target system
#
get_efi_boot_file()
{
	local f

	for f in \
		"$bootldr_id/shim$efi_suffix" \
		"$bootldr_id/grub$efi_suffix" \
		"BOOT/shim$efi_suffix" \
		"BOOT/grub$efi_suffix" \
		"BOOT/boot$efi_suffix"
	do
		# In any case, the last value is used if no other is found
		[ ! -s "$destdir/boot/efi/EFI/$f" ] || break
	done

	# Returns a path similar to "\EFI\ALTLINUX\SHIMX64.EFI"
	echo "\\EFI\\${f//\//\\}" |tr '[[:lower:]]' '[[:upper:]]'
}

# Writes boot record(s) to NVRAM
#
record_nvram()
{
	local f

	msg "Adding boot record(s) to NVRAM..."

	# Add a standard "UEFI OS" entry pointing to
	# /EFI/BOOT/BOOT${ARCH}.EFI. Otherwise, with
	# the NVRAM recovery flag enabled, such an entry
	# may be created by the firmware with an error.
	#
	if [ -n "$safe_uefi_boot" ] &&
	   [ -z "$no_clear_nvram" ]
	then
		local e

		e="$(echo "$efi_suffix" |
				tr '[[:lower:]]' '[[:upper:]]')"

		if [ -s "$destdir/boot/efi/EFI/BOOT/BOOT$e" ] ||
		   [ -s "$destdir/boot/efi/EFI/BOOT/boot$efi_suffix" ]
		then
			local b="UEFI OS"

			f="\\EFI\\BOOT\\BOOT$e"
			run efibootmgr -q -c -d "$target" -t 0 -L "$b" -l "$f" ||:
			run efibootmgr |run grep -qws -- "$b" ||
				run efibootmgr -q -c -d "$target" -L "$b" -l "$f"
		fi
	fi

	# Add an ALT-specific UEFI boot entry
	# pointing to /EFI/altlinux/shim${ARCH}.efi
	#
	f="$(get_efi_boot_file)"
	run efibootmgr -q -c -d "$target" -t 0 -L "$bootldr_id" -l "$f" ||:
	run efibootmgr |run grep -qws -- "$bootldr_id" ||
		run efibootmgr -q -c -d "$target" -L "$bootldr_id" -l "$f"
}

# Hook
#
before_chroot()
{
	: # Do nothing by default
}

# Hook
#
after_chroot()
{
	: # Do nothing by default
}

