#!/bin/bash
###
### This file is covered by the GNU General Public License
### version 3 or later.
###
### Copyright (C) 2021-2025, ALT Linux Team

############################################################
### Supplemental functions for working with target disks ###
############################################################

# Determinates the device name of the whole disk
# for any specified device such as a disk partition
#
get_whole_disk()
{
	local varname="$1" partdev="$2"
	local number sysfs partn whole=""

	number="$(run mountpoint -x -- "$partdev")"
	sysfs="$(run readlink -fv -- "/sys/dev/block/$number")"

	if [ -r "$sysfs/partition" ]; then
		read -r partn <"$sysfs/partition" ||
			partn=
		if [ -n "$partn" ]; then
			case "$partdev" in
			*[0-9]p$partn)
				whole="${partdev%%p$partn}"
				;;
			*$partn)
				whole="${partdev%%$partn}"
				;;
			esac
		fi
		[ -n "$whole" ] && [ -b "$whole" ] &&
		[ -r "/sys/block/${whole##/dev/}/${partdev##/dev/}/dev" ] ||
			whole=
	fi

	[ -z "$whole" ] || eval "$varname=\"$whole\""
}

# Determinates a size of the specified whole disk
#
get_disk_size()
{
	local dname="$1" nblocks disksize
	local sysfs="/sys/block/${dname##/dev/}"

	if [ ! -s "$sysfs/size" ]; then
		disksize="$(run blockdev --getsize64 "/dev/${dname##/dev/}")"
	else
		read -r nblocks <"$sysfs/size" ||:
		disksize="$((512 * ${nblocks:-0}))"
	fi

	echo -n "$disksize"
}

# Reads information about specified whole disk
#
read_disk_info()
{
	local dev="${1##/dev/}"
	local di field dsz=

	di="$(get_disk_size "$dev")"
	di="$(size2human -w "$di")"

	for field in MODEL SERIAL_SHORT REVISION; do
		dsz="$(run udevadm info -- "/dev/$dev"  |
			sed -n -e "s/^E: ID_$field=//p" |
			tr '_' ' ')"
		[ -z "$dsz" ] || di="${di:+$di }$dsz"
	done

	printf "%s" "$di"
}

# Populates the list of protected devices
# with the specified mount points and/or devices
#
protect_boot_devices()
{
	local number sysfs mp pdev wdev

	if [ -n "$protected_devices" ]; then
		mp="$protected_devices"
		protected_devices=

		for wdev in $mp; do
			wdev="${wdev##/dev/}"
			will_be_run mountpoint -x -- "/dev/$wdev"
			number="$(mountpoint -x -- "/dev/$wdev" 2>/dev/null ||:)"
			[ -n "$number" ] ||
				continue
			sysfs="$(run readlink -fv -- "/sys/dev/block/$number")"
			[ -r "$sysfs"/uevent ] ||
				continue
			pdev="$(run grep -sE ^DEVNAME= "$sysfs"/uevent |cut -f2 -d=)"
			[ -n "$pdev" ] ||
				continue
			pdev="/dev/$pdev"
			[ -b "$pdev" ] ||
				continue
			wdev="$pdev"
			get_whole_disk wdev "$pdev"
			in_array "$wdev" $protected_devices ||
				protected_devices="$protected_devices $wdev"
		done
	fi

	for mp in $protected_mpoints; do
		mountpoint -q -- "$mp" ||
			continue
		will_be_run mountpoint -d -- "$mp"
		number="$(mountpoint -d -- "$mp" 2>/dev/null ||:)"
		[ -n "$number" ] ||
			continue
		sysfs="$(run readlink -fv -- "/sys/dev/block/$number")"
		[ -r "$sysfs"/uevent ] ||
			continue
		pdev="$(run grep -sE ^DEVNAME= "$sysfs"/uevent |cut -f2 -d=)"
		[ -n "$pdev" ] ||
			continue
		pdev="/dev/$pdev"
		[ -b "$pdev" ] ||
			continue
		wdev="$pdev"
		get_whole_disk wdev "$pdev"
		in_array "$wdev" $protected_devices ||
			protected_devices="$protected_devices $wdev"
	done

	protected_devices="${protected_devices:1}"
}

# Shows information about the selected disk
#
show_disk_info()
{
	local disk="$1"
	local cmd="env LC_ALL=C dialog"
	local buffer="$workdir/diskinfo.txt"

	run lsblk -f -- "$disk" >"$buffer"
	fdump "$buffer"

	cmd="$cmd --backtitle \"$title\""
	cmd="$cmd --title \"[ $disk info ]\""
	cmd="$cmd --textbox \"$buffer\" 0 0"

	run eval $cmd ||:
	run rm -f -- "$buffer"
}

# Shows the dialog for selecting the target disk drive
#
select_target_drive()
{
	local devices="$1"
	local title="$product_title: STICK-v$release"
	local text="Select the target device to install the system to:"
	local height=1 width=$(( 4 + ${#text} ))
	local di cmd iw maxw right="" rc=0

	# Determinating console width
	echo -ne "\e[s\e[1000;1000H\e[6n\e[u"
	IFS=';[' read -s -t2 -dR cmd di iw || {
		di=24
		iw=80
	}
	maxw="$(( $iw - 6 ))"
	stty rows "$di" cols "$iw" 2>/dev/null ||:

	# Menu width and count items
	for cmd in $devices _; do
		[ "$cmd" != _ ] && [ -b "$cmd" ] ||
			continue
		di="$(read_disk_info "$cmd")"
		right="$right \"$cmd\" \"${di//\"/\'}\""
		iw=$(( 12 + ${#di} + ${#cmd} ))
		[ "$iw" -le "$width" ] ||
			width="$iw"
		rc="$((1 + $rc))"
	done

	# We must have more than one item here
	[ "$rc" -gt 1 ]

	# Fixing the box dimensions to fit the screen
	if [ "$width" -lt 40 ]; then
		width=40
	elif [ "$width" -gt "$maxw" ]; then
		height="$(( $width / $maxw + 1 ))"
		width="$maxw"
	fi
	if [ "$rc" -gt 13 ]; then
		height="$((20 + $height))"
	else
		height="$((7 + $height + $rc))"
	fi

	# Left part of the dialog command
	cmd="env LC_ALL=C dialog --backtitle \"$title\""
	cmd="$cmd --title \"[ Target drive ]\""
	cmd="$cmd --help-button --help-label Info"
	cmd="$cmd --no-cancel --menu \"\n$text\""
	cmd="$cmd $height $width $rc"

	while :; do
		rc=0
		exec 3>&1
		will_be_run "eval ${cmd}${right}"
		target="$(eval "${cmd}${right}" 2>&1 1>&3)" ||
			rc=$?
		exec 3>&-
		[ "$rc" != 0 ] ||
			break
		[ "$rc" != 2 ] ||
			show_disk_info "${target##HELP }"
		sleep .5
	done

	# Single target selected and it is an existing block special device
	[ -n "$target" ] && [ -b "$target" ]

	clear
	welcome_text
}

# Hook: returns 0 if device $1 must be protected
#
filter_drive()
{
	return 1
}

# Looks for a target device(s)
#
search_target_drive()
{
	local dev dsz msz="" mxz="" cnt=0

	__skip_dev()
	{
		log "Skipping /dev/%s because it is %s" "$dev" "$1"
	}

	[ -z "$min_target_size" ] ||
		msz="$(human2size "$min_target_size")"
	[ -z "$max_target_size" ] ||
		mxz="$(human2size "$max_target_size")"

	for dev in $(ls /sys/block/) _; do
		case "$dev" in
		loop[0-9]*|ram[0-9]*|sr[0-9]*|dm-[0-9]*|md[0-9]*|_)
			continue
			;;
		esac

		if [ ! -b "/dev/$dev" ]; then
			__skip_dev "not a block special device"
			continue
		fi

		if [ -r "/sys/block/$dev/ro" ] &&
		   read -r dsz <"/sys/block/$dev/ro" && [ "$dsz" = 1 ]
		then
			__skip_dev "a read-only device"
			continue
		fi

		if [ -r "/sys/block/$dev/removable" ] &&
		   read -r dsz <"/sys/block/$dev/removable" &&
		   [ "$dsz" = 1 ] && [ -z "$use_removable" ]
		then
			__skip_dev "a removable disk drive"
			continue
		fi

		if in_array "/dev/$dev" $protected_devices; then
			__skip_dev "a write protected device"
			continue
		fi

		if [ -n "$target_model" ]; then
			dsz=

			for dsz in $(set +f;
				ls -1 /dev/disk/by-id/*${target_model}* \
				2>/dev/null |grep -vE -- '\-part[0-9]+$') _
			do
				if [ "$dsz" != _ ]; then
					dsz="$(readlink -fv -- "$dsz")"
					[ "$dsz" != "/dev/$dev" ] ||
						break
				fi
				dsz=
			done

			if [ -z "$dsz" ]; then
				__skip_dev "not a $target_model"
				continue
			fi
		fi

		if [ -n "$msz" ] || [ -n "$mxz" ]; then
			dsz="$(get_disk_size "$dev")"
		fi

		if [ -n "$msz" ]; then
			if [ "$dsz" -lt "$msz" ] 2>/dev/null; then
				__skip_dev "less than allowed capacity"
				continue
			fi
		fi

		if [ -n "$mxz" ]; then
			if [ "$dsz" -gt "$mxz" ] 2>/dev/null; then
				__skip_dev "greater than allowed capacity"
				continue
			fi
		fi

		if track_hook filter_drive "/dev/$dev"; then
			__skip_dev "filtered by the user-defined hook"
			continue
		fi

		cnt=$((1 + $cnt))
		target="$target /dev/$dev"
	done
	unset __skip_dev

	# Disk drives not found
	[ -n "$target" ] && target="${target:1}" ||
		fatal "Target device(s) not found."

	# Finish disk scan here
	if [ "$action" = scandisk ]; then
		echo "$target" |tr ' ' '\n'
		exit 0
	fi

	# We found more than one disk
	if [ "$cnt" -gt 1 ]; then
		# but manual selection is not allowed
		[ -n "$choose_target" ] && [ -z "$quiet" ] ||
			fatal "Manual selection of the target device is not allowed."
		select_target_drive "$target"
	fi

	# Separator between device name of the whole disk and the partition number
	case "$target" in
	*[0-9])	ppartsep="p";;
	*)	ppartsep="";;
	esac

	# Basic information about the whole disk
	diskinfo="$(read_disk_info "$target")"
}

# Return the partition device name given
# the target device name and partition number
#
devnode()
{
	printf "%s%s%s" "$target" "$ppartsep" "$1"
}

# Returns a list of partitions on the target device
#
list_partitions()
{
	local pname extpart=

	if [ "x${1-}" = "x-x" ]; then
		pname="$(run env LC_ALL=C sfdisk -l -- "$target" |
				run sed -n 's/^Disklabel type: //p')"
		if [ "$pname" = dos ]; then
			extpart="$(run env LC_ALL=C sfdisk \
					-q -l -o Device,Id -- "$target" |
					run sed -n -E 's/ (85| 5)$//p'  |
					run head -n1)"
		fi
	fi

	set +f

	if [ -z "$extpart" ]; then
		# List all partitions
		run ls -r -- ${target}${ppartsep}?* ||:
	else
		# Exclude first extended partition
		run ls -1r -- ${target}${ppartsep}?* |
		while read -r pname; do
			[ "$pname" = "$extpart" ] ||
				printf "%s " "$pname"
		done
	fi
}

# Tells the kernel to reread a partition table on the specified device
# and waits for new partitions to be ready. If device is not specified,
# the target disk drive will be used.
#
rereadpt()
{
	local n x partname start lenght device="${1:-$target}"
	local junk cmd="env LC_ALL=C sfdisk -q -f -l --color=never"

	sync "$device"

	case "$device" in
	*[0-9])	x="${device}p";;
	*)	x="$device";;
	esac

	run $cmd -- "$device" |run sed '1d;s/  */ /g' |
	while IFS=' ' read -r partname start lenght junk; do
		n="${partname##$x}"
		will_be_run addpart "$device" "$n" "$start" "$lenght"
		addpart "$device" "$n" "$start" "$lenght" 2>/dev/null ||:
	done

	will_be_run blockdev --rereadpt "$device"
	blockdev --rereadpt "$device" 2>/dev/null ||:
	run udevadm trigger -q -- "$device" ||:

	junk=( $(run $cmd -- "$device" |run sed '1d;s/ .*//g') )
	n="${#junk[@]}"
	lenght="$n"

	while :; do
		for partname in "${junk[@]}"; do
			[ ! -b "$partname" ] ||
				n=$(( $n - 1 ))
		done
		[ "$n" -gt 0 ] ||
			break
		n="$lenght"
		sleep .2
	done

	run udevadm settle -t5 ||:
}

# Reads or writes a partition label on the specified GUID/GPT disk
#
gpt_part_label()
{
	local device="$1" partno="$2" label="${3-}"
	local cmd="env LC_ALL=C sfdisk -q -f --part-label"

	[ -n "$device" ] && [ "$device" != '-' ] ||
		device="$target"
	if [ -n "$label" ]; then
		run $cmd "$device" "$partno" "$label"
	else
		run $cmd "$device" "$partno"
	fi
}

# Returns the UUID of the file system on the specified device
#
get_fs_uuid()
{
	run blkid -c /dev/null -o value -s UUID -- "$1"
}

# Hook: shows top of the target /etc/fstab (special filesystems)
#
mkfstab_top()
{
	local rg

	rg="UUID=|LABEL=|\/dev\/disk\/by\-|\/dev\/sd[a-z]|\/dev\/nvme[0-9]"
	rg="$rg|\/dev\/md[0-9]|\/dev\/dm\-[0-9]|\/dev\/mapper\/|$"
	cat -- "$workdir/FSTABLE" |sed -E "/^\s*($rg)/d"
}

# Default implementation that shows a single /etc/fstab entry
#
__mkfstab_entry()
{
	local uuid mp="$1"

	case "$mp" in
	/)
		uuid="$(get_fs_uuid "$rootpart")"
		echo "UUID=$uuid	$mp	ext4	$rootopts	1 1"
		;;

	/boot)
		echo "UUID=$bootuuid	$mp	ext2	$bootopts	1 2"
		;;

	/boot/efi)
		uuid="$(get_fs_uuid "$esp_part")"
		echo "UUID=$uuid	$mp	vfat	$esp_opts	1 2"
		;;

	"$datapart_mp")
		uuid="$(get_fs_uuid "$datapart")"
		echo "UUID=$uuid	$mp	ext4	$dataopts	1 2"
		;;

	swap)
		echo "UUID=$swapuuid	$mp	swap	defaults	0 0"
		;;
	esac
}

# Hook: shows bottom of the target /etc/fstab (optional mount points)
#
mkfstab_bottom()
{
	: # Do nothing by default
}

# Shows complete target /etc/fstab
#
mkfstab()
{
	track_hook mkfstab_top
	echo
	[ -z "$rootpart" ] ||
		track_hook mkfstab_entry /
	[ -z "$bootpart" ] ||
		track_hook mkfstab_entry /boot
	[ -z "$esp_part" ] ||
		track_hook mkfstab_entry /boot/efi
	[ -z "$datapart" ] ||
		track_hook mkfstab_entry "$datapart_mp"
	[ -z "$swappart" ] ||
		track_hook mkfstab_entry swap
	echo
	track_hook mkfstab_bottom
}

