#!/bin/bash
###
### This file is covered by the GNU General Public License
### version 3 or later.
###
### Copyright (C) 2021-2025, ALT Linux Team

##############################
### Backup image validator ###
##############################

validate_checksum()
{
	local subdir="$1" chkfile="$2" util="$3"
	local validsum filename testsum n i=0
	local value numeric="-n -i 1"

	__read_image_inner()
	{
		( read_file "${subdir}$filename" |
			pv $numeric -s "$testsum" | "$util" |
			sed -E 's/\s+.+$//g' >"$workdir/testsum.chk"
		) 2>&1
	}

	n="$(wc -l -- "$chkfile" |
		sed -E 's/\s+.*$//g')"
	while [ "$i" -lt "$n" ]
	do
		i="$((1 + $i))"
		validsum="$(sed -n "${i}p" "$chkfile" |
					sed -E 's/\s+.+$//g')"
		filename="$(sed -n "${i}p" "$chkfile" |
					sed -E 's/^\w+\s+//g')"

		if ! file_exists "${subdir}$filename"; then
			[ "${subdir}$filename" != esp.tgz ] ||
				continue
			[ "${subdir}$filename" != home.tgz ] ||
				continue
			fatal "Backup not found: %s" "${subdir}$filename"
		fi

		msg "%s '%s'..." "-" "${subdir}$filename"
		testsum="$(get_file_size "${subdir}$filename")"
		run rm -f $verbose -- "$workdir"/ERROR

		if [ -n "$alterator" ]; then
			notify_alterator message "%["
			__read_image_inner |
			while read -r value; do
				is_number "$value" ||
					continue
				notify_alterator message "% $value"
				printf "\r%s%%" "$value"
			done
			printf "\r"
			notify_alterator message "%]"

		elif [ -n "$frontend_pipe" ]; then
			notify_frontend I "%s" "%["
			__read_image_inner |
			while read -r value; do
				is_number "$value" ||
					continue
				notify_frontend '%' "%s" "$value"
				printf "\r%s%%" "$value"
			done
			printf "\r"
			notify_frontend I "%s" "%]"

		else
			numeric=
			__read_image_inner
		fi

		read -r testsum <"$workdir/testsum.chk" 2>/dev/null ||
			testsum="<ERROR>"
		[ -n "$quiet" ] ||
			printf "%s %s=%s " "-" "${subdir}$filename" "$testsum"
		dbg "Expected: %s" "$validsum"
		dbg "Has got:  %s" "$testsum"

		if [ "$testsum" = "$validsum" ]
		then
			[ -n "$quiet" ] ||
				printf "${CLR_OK}OK${CLR_NORM}\n"
			run rm -f $verbose -- "$workdir/testsum.chk"
		else
			[ -n "$quiet" ] ||
				printf "${CLR_ERR}FAIL${CLR_NORM}\n"
			run rm -f $verbose -- "$workdir/testsum.chk"
			fatal "Backup image checksum mismatch."
		fi
	done

	unset __read_image_inner
}

validate_update_archive()
{
	local subdir="$1" ext="$2" util="$3"
	local fname="${subdir}update.$ext"
	local dst="$workdir/update.$ext"

	if have_binary "$util" && file_exists "$fname"; then
		run rm -f $verbose -- "$workdir"/ERROR
		read_file "$fname" >"$dst"
		[ ! -s "$workdir"/ERROR ] ||
			fatal "Error loading checksums: '%s'." "$fname"
		validate_checksum "$subdir" "$dst" "$util"
		run rm -f $verbose -- "$dst"
	fi
}

validate_in_subdir()
{
	validate_update_archive "$1" MD5 md5sum
	validate_update_archive "$1" SHA sha1sum
	validate_update_archive "$1" 256 sha256sum
}

validate_backup_images()
{
	msg "Validating backup images..."
	[ -z "$profile_subdir" ] || ! file_exists "$profile_subdir"/update.tgz ||
		validate_in_subdir "$profile_subdir"/
	! file_exists update.tgz ||
		validate_in_subdir ""
	[ ! -s "$workdir"/checksum.MD5 ] || ! have_binary md5sum ||
		validate_checksum "" "$workdir"/checksum.MD5 md5sum
	[ ! -s "$workdir"/checksum.SHA ] || ! have_binary sha1sum ||
		validate_checksum "" "$workdir"/checksum.SHA sha1sum
	[ ! -s "$workdir"/checksum.256 ] || ! have_binary sha256sum ||
		validate_checksum "" "$workdir"/checksum.256 sha256sum
	msg "Backup images validated successfully!"
}

