
/*
  $Id: cmdproxyd.c,v 1.4 2004/05/23 18:52:53 ldv Exp $
  Copyright (C) 2004  Dmitry V. Levin <ldv@altlinux.org>

  The command proxy daemon.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#define _GNU_SOURCE

#include <stdio.h>
#include <errno.h>
#include <error.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <syslog.h>
#include <fcntl.h>
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/un.h>
#include <sys/wait.h>

static void
sigchld_handler ( __attribute__ ((unused))
		 int signo)
{
	while (waitpid (-1, 0, WNOHANG) > 0)
		;
}

static void __attribute__ ((__noreturn__)) usage (int rc)
{
	extern const char *__progname;

	fprintf ((rc == EXIT_SUCCESS) ? stdout : stderr,
		 "Usage: %s bind_address.. runas program...\n", __progname);
	exit (rc);
}

static int
bind_address (const char *address)
{
	struct sockaddr_un sun;

	if (strlen (address) >= sizeof (sun))
		error (EXIT_FAILURE, EINVAL, "cannot bind socket `%s'",
		       address);

	memset (&sun, 0, sizeof (sun));
	sun.sun_family = AF_UNIX;
	strcpy (sun.sun_path, address);

	int     fd = socket (PF_UNIX, SOCK_STREAM, 0);

	if (fd < 0)
		error (EXIT_FAILURE, errno, "cannot create socket `%s'",
		       address);

	if (bind (fd, (struct sockaddr *) &sun, sizeof (sun)))
	{
		unlink (address);

		if (bind (fd, (struct sockaddr *) &sun, sizeof (sun)))
			error (EXIT_FAILURE, errno, "cannot bind socket `%s'",
			       address);
	}

	if (chmod (address, 0666))
		error (EXIT_FAILURE, errno,
		       "cannot set permissions of socket `%s'", address);

	if (listen (fd, 5) < 0)
		error (EXIT_FAILURE, errno, "listen");

	int     flags;

	if ((flags = fcntl (fd, F_GETFL, 0)) == -1)
		error (EXIT_FAILURE, errno, "fcntl: F_GETFL");

	flags |= O_NONBLOCK;

	if (fcntl (fd, F_SETFL, flags) == -1)
		error (EXIT_FAILURE, errno, "fcntl: F_SETFL");

	if ((flags = fcntl (fd, F_GETFD, 0)) == -1)
		error (EXIT_FAILURE, errno, "fcntl: F_GETFD");

	flags |= FD_CLOEXEC;

	if (fcntl (fd, F_SETFD, flags) == -1)
		error (EXIT_FAILURE, errno, "fcntl: F_SETFD");

	return fd;
}

static void
handle_socket (int fd, const char *address, const char *const *args)
{
	struct sockaddr_un sun;

	memset (&sun, 0, sizeof (sun));
	sun.sun_family = AF_UNIX;
	socklen_t sunlen = sizeof (sun);
	int     accfd = accept (fd, (struct sockaddr *) &sun, &sunlen);

	if (accfd < 0)
	{
		if (errno != EINTR)
		{
			syslog (LOG_ERR, "accept: %s[%d]: %m", address, fd);
			sleep (1);
		}
		return;
	}

	struct ucred sucred;
	socklen_t credlen = sizeof (struct ucred);

	if (getsockopt (accfd, SOL_SOCKET, SO_PEERCRED, &sucred, &credlen))
	{
		close (accfd);
		syslog (LOG_ERR, "getsockopt: SO_PEERCRED: %s[%d]: %m",
			address, fd);
		sleep (1);
		return;
	}

	pid_t   pid = fork ();

	if (pid < 0)
	{
		close (accfd);
		syslog (LOG_ERR, "fork: %m");
		sleep (1);
		return;
	}

	if (pid)
	{
		close (accfd);
		return;
	}

	dup2 (accfd, 0);
	close (accfd);
	syslog (LOG_INFO, "request from %s, uid=%u, executing %s", address,
		sucred.uid, args[0]);
	execvp (args[0], (char *const *) args);
	syslog (LOG_ERR, "execvp: %s: %m", args[0]);
	exit (EXIT_FAILURE);
}

int
main (int ac, const char *av[])
{
	signal (SIGCHLD, sigchld_handler);

	if (ac < 3)
		usage (EXIT_FAILURE);

	const char *runas = 0;
	const char *const *args = 0;
	int     nsockets = 0;
	int     i;

	for (i = 1; i < ac; ++i)
	{
		if (runas)
		{
			args = &av[i];
			break;
		}

		if (av[i][0] == '/')
		{
			++nsockets;
			continue;
		}

		if (!nsockets)
			usage (EXIT_FAILURE);

		runas = av[i];
	}

	if (!nsockets || !runas || !args)
		usage (EXIT_FAILURE);

	struct passwd *pw = getpwnam (runas);

	if (!pw)
		error (EXIT_FAILURE, 0, "user `%s' lookup failed", runas);

	int     sockfd[nsockets];

	for (i = 1; i <= nsockets; ++i)
		sockfd[i - 1] = bind_address (av[i]);

	if (initgroups (runas, pw->pw_gid))
		error (EXIT_FAILURE, errno, "initgroups");

	if (setgid (pw->pw_gid))
		error (EXIT_FAILURE, errno, "setgid");

	if (setuid (pw->pw_uid))
		error (EXIT_FAILURE, errno, "setuid");

	endpwent ();

	if (daemon (0, 0))
		error (EXIT_FAILURE, errno, "daemon");

	openlog ("cmdproxyd", LOG_PERROR | LOG_PID, LOG_MAIL);

	for (;;)
	{
		int     i, maxfd = 0;
		fd_set  r;

		FD_ZERO (&r);
		for (i = 0; i < nsockets; ++i)
		{
			FD_SET (sockfd[i], &r);
			if (sockfd[i] > maxfd)
				maxfd = sockfd[i];
		}
		int     rc = select (1 + sockfd[nsockets - 1], &r, 0, 0, 0);

		if (rc < 0)
		{
			if (errno == EINTR)
				continue;

			syslog (LOG_ERR, "select: %m");
			sleep (1);
			continue;
		}
		for (i = 0; i < nsockets; ++i)
			if (FD_ISSET (sockfd[i], &r))
				handle_socket (sockfd[i], av[1 + i], args);
	}
}
