
/*
  $Id: diff.c,v 1.4 2006/03/19 14:44:19 ldv Exp $
  Copyright (C) 2004, 2006  Dmitry V. Levin <ldv@altlinux.org>

  The diff generator for the commitlogger program.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#define _GNU_SOURCE

#include <stdio.h>
#include <errno.h>
#include <error.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include "xmalloc.h"
#include "filter.h"
#include "diff.h"

#define PATH_BIN_CVS "/usr/bin/cvs"
#define MAX_DIFF_SIZE 16384

void
print_diff(FILE * out, const char *dir, const char *arg, struct diff *diff)
{
	const char *name, *rev1, *rev2, *rest;
	char   *argbuf = xstrdup(arg);

	memset(diff, 0, sizeof(*diff));

	name = strtok(argbuf, ",");
	if (!name)
	{
		fprintf(out, "print_diff: %s: %s\n", arg, strerror(EINVAL));
		return;
	}
	rev1 = strtok(0, ",");
	if (!rev1)
	{
		fprintf(out, "print_diff: %s: %s\n", arg, strerror(EINVAL));
		return;
	}
	rev2 = strtok(0, ",");
	if (!rev2)
	{
		fprintf(out, "print_diff: %s: %s\n", arg, strerror(EINVAL));
		return;
	}
	rest = strtok(0, ",");
	if (rest)
	{
		fprintf(out, "print_diff: %s: %s\n", arg, strerror(EINVAL));
		return;
	}

	if (!strcmp(rev1, "NONE"))
	{
		fprintf(out, "New file: %s\n", name);
		return;
	}

	if (!strcmp(rev2, "NONE"))
	{
		fprintf(out, "Removed file: %s\n", name);
		return;
	}

	int     pipefd[2];

	if (pipe(pipefd) < 0)
	{
		fprintf(out, "pipe: %s\n", strerror(errno));
		return;
	}
	pid_t   pid = fork();

	if (pid < 0)
	{
		fprintf(out, "fork: %s\n", strerror(errno));
		return;
	} else if (!pid)
	{
		const char *path = PATH_BIN_CVS;
		char   *fname;
		const char *const args[] = {
			"cvs", "-f", "rdiff", "-u", "-r", rev1, "-r", rev2,
			xasprintf(&fname, "%s/%s", dir, name), 0
		};
		close(pipefd[0]);
		dup2(pipefd[1], 1);
		dup2(pipefd[1], 2);
		if (pipefd[1] > 2)
			close(pipefd[1]);
		execv(path, (char *const *) args);
		error(EXIT_FAILURE, errno, "execv: %s", path);
	}

	close(pipefd[1]);

	FILE   *fp = fdopen(pipefd[0], "r");

	if (!fp)
	{
		fprintf(out, "fdopen: %s\n", strerror(errno));
		close(pipefd[0]);
		free(argbuf);
		return;
	}

	FILE   *stream = open_memstream(&diff->buffer, &diff->size);

	if (!stream)
	{
		fprintf(out, "open_memstream: %s\n", strerror(errno));
		fclose(fp);
		free(argbuf);
		return;
	}

	copy_in_out(fp, stream);

	fclose(stream);
	fclose(fp);

	if (diff->size > MAX_DIFF_SIZE)
	{
		fprintf(out,
			"Changed file: %s, diff between revisions %s and %s is too large (%u bytes)\n",
			name, rev1, rev2, diff->size);
		free(diff->buffer);
		memset(diff, 0, sizeof(*diff));
	} else
	{
		xasprintf(&diff->name, "%s.diff", name);
		fprintf(out,
			"Changed file: %s, diff is included (%u bytes)\n",
			name, diff->size);
	}

	free(argbuf);
}
