
/*
  Copyright (C) 1994 Free Software Foundation, Inc.
  Copyright (C) 2002-2006  Dmitry V. Levin <ldv@altlinux.org>

  The locate database open/close methods.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA

  Written by James A. Woods <jwoods@adobe.com>.
  Modified by David MacKenzie <djm@gnu.ai.mit.edu>.
  Modified by Dmitry V. Levin <ldv@altlinux.org>.
*/

#include <errno.h>
#include <error.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <sys/stat.h>

#include "ldb.h"
#include "locatedb.h"
#include "lstats.h"

/* Warn if a database is older than this.  8 days allows for a weekly
   update that takes up to a day to perform.  */
#define WARN_NUMBER_UNITS (8)

#define SECONDS_PER_UNIT (60 * 60 * 24)

#define WARN_SECONDS ((SECONDS_PER_UNIT) * (WARN_NUMBER_UNITS))

/* Printable name of units used in WARN_SECONDS */
static const char warn_name_units[] = "days";

FILE *
ldb_open(const char *dbfile, locate_settings *settings, locate_stats *stats)
{
	FILE   *fp;
	time_t  now;
	ssize_t nread;
	struct stat st;
	char magic[sizeof(LOCATEDB_MAGIC)];

	if (!(fp = fopen(dbfile, "r")) || fstat(fileno(fp), &st))
	{
		if (!settings->quiet)
			error(EXIT_SUCCESS, errno, "%s", dbfile);
		if (fp)
			(void) fclose(fp);
		return NULL;
	}

	time(&now);
	if (now - st.st_mtime > WARN_SECONDS)
	{
		if (!settings->quiet)
			error(EXIT_SUCCESS, 0,
			      "warning: database `%s' is more than %d %s old",
			      dbfile, WARN_NUMBER_UNITS, warn_name_units);
	}

	nread = fread(magic, 1UL, sizeof(magic), fp);
	if (nread != sizeof(magic)
	    || memcmp(magic, LOCATEDB_MAGIC, sizeof(magic)))
	{
		if (!settings->quiet)
			error(EXIT_SUCCESS, 0,
			      "warning: file `%s' is not a locate database",
			      dbfile);
		if (fclose(fp) == EOF && !settings->quiet)
			error(EXIT_SUCCESS, errno, "%s", dbfile);
		return NULL;
	}

	if (settings->print_stats)
	{
		memset(stats, 0, sizeof(*stats));
		stats->file_size = st.st_size;
	}

	return fp;
}

void
ldb_close(FILE *fp, const char *dbfile, locate_settings *settings,
	  locate_stats *stats)
{
	if (fclose(fp) == EOF && !settings->quiet)
		error(EXIT_SUCCESS, errno, "%s", dbfile);

	if (settings->print_stats)
		print_stats(stats);
}
