
/*
  Copyright (C) 1994 Free Software Foundation, Inc.
  Copyright (C) 2002-2006  Dmitry V. Levin <ldv@altlinux.org>

  locate -- search databases for filenames that match patterns

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA

  Written by James A. Woods <jwoods@adobe.com>.
  Modified by David MacKenzie <djm@gnu.ai.mit.edu>.
  Modified by Dmitry V. Levin <ldv@altlinux.org>.
*/

#include <stdio.h>
#include <errno.h>
#include <error.h>
#include <string.h>
#include <stdlib.h>

#include "ldb.h"
#include "nextelem.h"
#include "readstr.h"
#include "xmalloc.h"

/*
 * Return number of lines matched.
 */
uintmax_t
locate(int argc, match_context *ctxv, const char *dbfile,
       locate_settings *settings)
{
	FILE   *fp;
	uintmax_t found = 0;
	ssize_t nread;
	int     c;
	size_t  pathsize = 1024;
	size_t  count = 0;
	char   *path;
	const char *sample;
	locate_stats stats;

	if (!(fp = ldb_open(dbfile, settings, &stats)))
		return 0;

	path = xmalloc(pathsize);

	for (nread = 0, c = fgetc(fp);
	     c >= 0 && (!settings->limit_output || found < settings->limit);
	     c = fgetc(fp))
	{
		if (c == 0x80)
		{
			short   c1, c2, r;

			if (((c1 = fgetc(fp)) == EOF)
			    || ((c2 = fgetc(fp)) == EOF))
			{
				error(EXIT_SUCCESS, errno,
				      "%s: unexpected end of input", dbfile);
				break;
			}

			r = (c1 << 8) | (c2 & 0xff);
			count += r;
		} else if (c > 127)
			count += c - 256;
		else
			count += c;

		if (nread && (count > (size_t) nread))
		{
			error(EXIT_SUCCESS, 0,
			      "%s: corrupted database", dbfile);
			break;
		}

		/* Overlay the old path with the remainder of the new.  */
		nread = read_str(&path, &pathsize, fp, count);
		if (nread < 0)
		{
			if (EINVAL == errno)
				error(EXIT_SUCCESS, 0,
				      "%s: corrupted database", dbfile);
			break;
		}
		nread += count;

		if (settings->print_stats)
			count_stats(path, &stats);

		if (settings->match_basename)
		{
			sample = strrchr(path, '/');
			if (sample)
				++sample;
			else
				sample = path;
		} else {
			sample = path;
		}

		if (match(argc, ctxv, path, sample, settings))
			++found;
	}

	if (ferror(fp) && !settings->quiet)
		error(EXIT_SUCCESS, errno, "%s", dbfile);

	ldb_close(fp, dbfile, settings, &stats);

	return found;
}
