
/*
  Copyright (C) 2006  Dmitry V. Levin <ldv@altlinux.org>

  The locate statistics.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
*/

#include <stdio.h>
#include <errno.h>
#include <error.h>
#include <string.h>
#include <ctype.h>
#include <stdbool.h>
#include <inttypes.h>

#include "lstats.h"

void
print_stats(locate_stats *stats)
{
	printf("Locate database size: %" PRIu64 " bytes\n",
	       (uintmax_t) stats->file_size);

	printf("Filenames: %" PRIu64 " with a cumulative length of %" PRIu64
	       " bytes\n", stats->total_filename_count,
	       stats->total_filename_length);

	printf("\tof which %" PRIu64 " contain%s whitespace,\n",
	       stats->whitespace_count,
	       stats->whitespace_count == 1 ? "s" : "");
	printf("\t%" PRIu64 " contain%s newline characters,\n",
	       stats->newline_count,
	       stats->newline_count == 1 ? "s" : "");
	printf("\tand %" PRIu64
	       " contain%s characters with the high bit set.\n",
	       stats->highbit_filename_count,
	       stats->highbit_filename_count == 1 ? "s" : "");

	printf("Compression ratio: %4.2f%%\n",
	       100.0 * ((double) stats->total_filename_length
			- (double) stats->file_size)
	       / (double) stats->total_filename_length);
}

void
count_stats(const char *path, locate_stats *stats)
{
	size_t  len = strlen(path);
	const char *s;
	bool    highbit = false, whitespace = false, newline = false;

	++(stats->total_filename_count);
	stats->total_filename_length += len;

	highbit = whitespace = newline = 0;
	for (s = path; *s; ++s)
	{
		if ((int) (*s) & 128)
			highbit = true;
		if ('\n' == *s)
		{
			newline = whitespace = true;
		} else if (isspace((unsigned char) *s))
		{
			whitespace = true;
		}
	}

	if (highbit)
		++(stats->highbit_filename_count);
	if (whitespace)
		++(stats->whitespace_count);
	if (newline)
		++(stats->newline_count);
}
