
/*
  Copyright (C) 1994 Free Software Foundation, Inc.
  Copyright (C) 2002-2006  Dmitry V. Levin <ldv@altlinux.org>

  locate -- search databases for filenames that match patterns

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA

  Usage: locate [options] pattern...

  Scan a pathname list for the full pathname of a file, given only
  a piece of the name (possibly containing shell globbing metacharacters).
  The list has been processed with front-compression, which reduces
  the list size by a factor of 4-5.

  Described more fully in Usenix ;login:, Vol 8, No 1,
  February/March, 1983, p. 8.

  Written by James A. Woods <jwoods@adobe.com>.
  Modified by David MacKenzie <djm@gnu.ai.mit.edu>.
  Modified by Dmitry V. Levin <ldv@altlinux.org>.
*/

#include <stdio.h>
#include <errno.h>
#include <error.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <inttypes.h>
#include <getopt.h>
#include <locale.h>

#include "match.h"
#include "nextelem.h"
#include "xmalloc.h"
#include "locatedb.h"

#define PACKAGE_BUGREPORT "http://bugs.altlinux.ru/"

extern int __libc_enable_secure;

static char *
xgetenv(const char *name)
{
	char   *p = getenv(name);

	return (p && *p) ? p : 0;
}

static void
drop_priv(void)
{
	gid_t   gid = getgid();

	if (setresgid(gid, gid, gid))
		error(EXIT_FAILURE, errno, "setresgid: %u", gid);

	__libc_enable_secure = 0;
}

static void __attribute__((__noreturn__))
print_usage(const char *str)
{
	if (str)
		error(EXIT_SUCCESS, 0, "%s", str);

	fprintf(stderr, "Try `%s --help' for more information.\n",
		program_invocation_short_name);
	exit(EXIT_FAILURE);
}

static uintmax_t str2limit(char const *value)
{
	char   *p = 0;
	unsigned long long n;

	if (!*value)
		print_usage("Invalid argument to --limit");

	errno = 0;
	n = strtoull(value, &p, 10);
	if (!p || *p || n > UINTMAX_MAX || (n == UINTMAX_MAX && errno == ERANGE))
		print_usage("Invalid argument to --limit");

	return n;
}

static void __attribute__((__noreturn__))
print_help(void)
{
	printf(
"Usage: %s [OPTION]... [PATTERN]...\n"
"Search for entries in a slocate database.\n"
"Valid options are:\n"
"  -A, --all              print only names which match all non-option arguments,\n"
"                         not those matching one or more non-option arguments\n"
"  -b, --basename         match only the base name of path names\n"
"  -c, --count            only print number of found entries\n"
"  -d, --database DBPATH  use DBPATH instead of default database\n"
"                         (which is %s)\n"
"  -e, --existing         only print entries for currently existing files\n"
"  -E, --no-existing      print entries regardless of its existence\n"
"  -L, --follow           follow trailing symbolic links when checking file\n"
"                         existence (this is the default)\n"
"  -h, --help             print this help\n"
"  -i, --ignore-case      ignore case distinctions when matching patterns\n"
"  -l, --limit LIMIT      limit output (or counting) to LIMIT entries\n"
"  -0, --null             separate entries with NUL on output\n"
"  -p, --print            print search results when they normally would not\n"
"  -P, --nofollow         don't follow trailing symbolic links when checking\n"
"                         file existence\n"
"  -q, --quiet            report no error messages about reading databases\n"
"  -r, --regex            patterns are regexps\n"
"  -S, --statistics       don't search for entries, print statistics about\n"
"                         each used database\n"
"  -V, --version          print version information\n"
"  -w, --wholename        match whole path name (this is the default)\n",
	       program_invocation_short_name, LOCATEDB_DBFILE);
	printf("\nReport bugs to %s.\n", PACKAGE_BUGREPORT);
	exit(EXIT_SUCCESS);
}

static struct option const longopts[] = {
	{"all", no_argument, NULL, 'A'},
	{"basename", no_argument, NULL, 'b'},
	{"count", no_argument, NULL, 'c'},
	{"database", required_argument, NULL, 'd'},
	{"existing", no_argument, NULL, 'e'},
	{"follow", no_argument, NULL, 'L'},
	{"help", no_argument, NULL, 'h'},
	{"ignore-case", no_argument, NULL, 'i'},
	{"limit", required_argument, NULL, 'l'},
	{"no-existing", no_argument, NULL, 'E'},
	{"nofollow", no_argument, NULL, 'P'},
	{"null", no_argument, NULL, '0'},
	{"print", no_argument, NULL, 'p'},
	{"quiet", no_argument, NULL, 'q'},
	{"regex", no_argument, NULL, 'r'},
	{"statistics", no_argument, NULL, 'S'},
	{"version", no_argument, NULL, 'v'},
	{"wholename", no_argument, NULL, 'w'},
	{"wholepath", no_argument, NULL, 'w'},	/* Synonym. */
	{NULL, no_argument, NULL, 0}
};

int
main(int argc, char *const *argv)
{
	locate_settings settings;
	match_context *ctxv;
	const char *dbpath = LOCATEDB_DBFILE, *e;
	uintmax_t found = 0;
	int     optc;
	bool    check_existence_set = false;
	bool    print_count = false;

	setlocale(LC_ALL, "");

	if (xgetenv("LOCATE_PATH"))
	{
		drop_priv();
		dbpath = xgetenv("LOCATE_PATH");
	}

	memset(&settings, 0, sizeof settings);
	settings.follow_symlinks = true;
	settings.separator = '\n';

	while ((optc =
		getopt_long(argc, argv, "Abcd:eLhil:EP0pqrSvw", longopts,
			    (int *) 0)) != -1)
		switch (optc)
		{
			case 'A':
				settings.match_all = true;
				break;

			case 'b':
				settings.match_basename = true;
				break;

			case 'c':
				print_count = true;
				break;

			case 'd':
				drop_priv();
				dbpath = optarg;
				break;

			case 'e':
				settings.check_existence = true;
				check_existence_set = true;
				break;

			case 'L':
				settings.follow_symlinks = true;
				break;

			case 'h':
				print_help();
				break;

			case 'i':
				settings.ignore_case = true;
				break;

			case 'l':
				settings.limit_output = true;
				settings.limit = str2limit(optarg);
				break;

			case 'E':
				settings.check_existence = false;
				check_existence_set = true;
				break;

			case 'P':
				settings.follow_symlinks = false;
				break;

			case '0':
				settings.separator = '\0';
				break;

			case 'p':
				settings.print_match = true;
				break;

			case 'q':
				settings.quiet = true;
				break;

			case 'r':
				settings.match_regex = true;
				break;

			case 'S':
				settings.print_stats = true;
				break;

			case 'v':
				printf("ALT locate version %s\n",
				       PROJECT_VERSION);
				return EXIT_SUCCESS;

			case 'w':
				settings.match_basename = false;
				break;

			default:
				print_usage(NULL);
		}

	if (!settings.print_stats && optind == argc)
		print_usage("Insufficient arguments");

	if (!check_existence_set)
		settings.check_existence = __libc_enable_secure;

	if (__libc_enable_secure && !settings.check_existence)
		drop_priv();

	if (!print_count && !settings.print_stats)
		settings.print_match = true;

	ctxv = match_init(argc - optind, &argv[optind], &settings);

	next_element(dbpath);
	while ((e = next_element(0))
	       && (!settings.limit_output || found < settings.limit))
	{
		found += locate(argc - optind, ctxv,
				strcmp(e, ".") ? e : LOCATEDB_DBFILE,
				&settings);
	}

	if (print_count)
		printf("%" PRIu64 "\n", found);

	if (found || (settings.limit_output && !settings.limit)
	    || settings.print_stats)
		return EXIT_SUCCESS;
	else
		return EXIT_FAILURE;
}
